package com.digiwin.athena.km_deployer_service.helper;

import cn.hutool.core.lang.Assert;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.km_deployer_service.config.ModuleConfig;
import com.digiwin.athena.km_deployer_service.domain.system.BusinessException;
import com.digiwin.athena.km_deployer_service.povo.asa.BasicInfoDto;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;

/**
 * @Description
 * @Author cai chao
 * @Date 2024/8/2 16:09
 */
@Slf4j
@Component
public class AssistantRhApiHelper {
    /**
     * RH api url
     */
    private static final String URL_COMMON_HDL = "/api/ai/RequestHDL";

    /**
     * 大模型"意图"提示词创建
     */
    private static final String URL_MODEL_LLM_PROMPT = "model/llm_prompt";

    /**
     * 大模型提示词修改
     */
    private static final String URL_MODEL_LLM_PROMPT_MODIFY = "model/llm_prompt_modify";

    /**
     * 大模型部署
     */
    private static final String URL_MODEL_LLM_DEPLOY = "model/llm/deploy";

    /**
     * 小模型打标签
     */
    private static final String URL_MODEL_DEPLOY_TAG = "model/deploy_tag";

    /**
     * ES写入
     */
    private static final String URL_ES_METADATA_UPSERT = "es/metadata/upsert";

    /**
     * ES删除
     */
    private static final String URL_ES_METADATA_DELETE = "es/metadata/delete";

    /**
     * 同义词ES部署
     */
    private static final String URL_ES_METADATA_DEPLOY = "es/metadata/deploy";

    @Autowired
    public void setModule(ModuleConfig module) {
        this.rhDomain = module.getKafka_middle().getDomain();
    }

    @Value("${appToken}")
    private String appToken;

    private String rhDomain;

    /**
     * 大模型部署
     */
    public void llmDeployer(Map map, String deployLlmPromptId, String deployMode, String curTenantToken) {
        try {
            JSONObject requestParam = new JSONObject();
            requestParam.put("basic_info", map);
            requestParam.put("deploy_llm_prompt_id", deployLlmPromptId);
            requestParam.put("deploy_mode", deployMode);
            JSONObject data = this.executeRhRequest(URL_MODEL_LLM_DEPLOY, requestParam, curTenantToken);

            if (data == null) {
                throw new BusinessException("大模型部署失败");
            }

            Assert.isTrue(data.getIntValue("code") == 0, () -> new BusinessException(data.getString("msg")));
        } catch (Exception e) {
            log.error("AssistantRhApiHelper.llmDeployer.error", e);
            throw new BusinessException("大模型部署失败");
        }
    }

    /**
     * 小模型指定部署
     */
    public void tagDeployer(String create_mode, Map projectName, String deployMode, String deploy_name, String curTenantToken) {
        try {
            JSONObject requestParam = new JSONObject();
            requestParam.put("create_mode", create_mode);
            requestParam.put("basic_info", projectName);
            requestParam.put("deploy_mode", deployMode);
            requestParam.put("deploy_name", deploy_name);
            JSONObject data = this.executeRhRequest(URL_MODEL_DEPLOY_TAG, requestParam, curTenantToken);

            if (data == null) {
                throw new BusinessException("小模型部署失败");
            }

            Assert.isTrue(data.getIntValue("code") == 0, () -> new BusinessException(data.getString("msg")));
        } catch (Exception e) {
            log.error("AssistantRhApiHelper.modelDeployer.error", e);
            throw new BusinessException("小模型部署失败");
        }
    }

    /**
     * 创建大模型提示词
     *
     * @param basicInfoDto
     * @param intentDefinition
     * @param example
     */
    public JSONObject addLlmPrompt(BasicInfoDto basicInfoDto, List<Map> intentDefinition, List<Map> example, String curTenantToken, List<Map> llmPredictNg, String assistantIntroduction) {
        try {
            JSONObject requestParam = new JSONObject();
            requestParam.put("example", example);
            requestParam.put("basic_info", basicInfoDto);
            requestParam.put("intent_definition", intentDefinition);
            requestParam.put("assistant_introduction", assistantIntroduction);
            requestParam.put("llm_predict_ng", llmPredictNg);
            JSONObject responseData = this.executeRhRequest(URL_MODEL_LLM_PROMPT, requestParam, curTenantToken);

            if (responseData == null) {
                throw new BusinessException("创建提示词失败");
            }

            return new JSONObject((Map<String, Object>) responseData.get("data"));
        } catch (Exception e) {
            log.error("AssistantRhApiHelper.addLlmPrompt.error", e);
            throw new BusinessException("创建提示词失败");
        }
    }

    /**
     * 大模型提示词修改
     *
     * @param llmPromptId  提示词代号
     * @param mode         提示词模式 intent|guide
     * @param systemPrompt 系统提示词
     * @param inputPrompt  用户提示词
     * @param iamToken     用户Token
     * @return
     */
    public JSONObject modifyLLMPrompt(String llmPromptId, String mode, String systemPrompt, String inputPrompt, String iamToken) {
        JSONObject requestData = new JSONObject();
        requestData.put("llm_prompt_id", llmPromptId);
        requestData.put("mode", mode);
        requestData.put("system_prompt", systemPrompt);
        requestData.put("input_prompt", inputPrompt);

        JSONObject responseData = this.executeRhRequest(URL_MODEL_LLM_PROMPT_MODIFY, requestData, iamToken);

        return responseData;
    }

    /**
     * 同义词ES部署
     */
    public void esDeployer(String application, String fromMode, String toMode, String curTenantToken) {
        try {
            JSONObject requestParam = new JSONObject();
            requestParam.put("assistant_id", application);
            requestParam.put("from_deploy_mode", fromMode);
            requestParam.put("to_deploy_mode", toMode);
            JSONObject data = this.executeRhRequest(URL_ES_METADATA_DEPLOY, requestParam, curTenantToken);

            if (data == null) {
                throw new BusinessException("同义词ES部署失败");
            }

            Assert.isTrue(data.getIntValue("code") == 0, () -> new BusinessException(data.getString("msg")));
        } catch (Exception e) {
            log.error("AssistantRhApiHelper.esDeployer.error", e);
            throw new BusinessException("同义词ES部署失败");
        }
    }

    /**
     * 调用es接口批次写入资料
     *
     * @param requestParam
     * @return
     */
    public void writeEsUpdateData(JSONObject requestParam, String curTenantToken) {
        try {
            JSONObject responseData = this.executeRhRequest(URL_ES_METADATA_UPSERT, requestParam, curTenantToken);

            if (responseData == null) {
                throw new BusinessException("调用调用es接口批次写入资料失败");
            }

            Assert.isTrue(responseData.getIntValue("code") == 0, () -> new BusinessException(responseData.getString("msg")));
        } catch (Exception e) {
            log.error("调用调用es接口批次写入资料失败", e);
            throw new BusinessException(e.getMessage());
        }
    }

    /**
     * 调用es接口批次删除资料
     *
     * @param requestParam
     * @return
     */
    public void writeEsDeleteData(JSONObject requestParam, String curTenantToken) {
        try {
            JSONObject responseData = this.executeRhRequest(URL_ES_METADATA_DELETE, requestParam, curTenantToken);

            if (responseData == null) {
                throw new BusinessException("调用调用es接口批次删除资料失败");
            }

            Assert.isTrue(responseData.getIntValue("code") == 0, () -> new BusinessException(responseData.getString("msg")));
        } catch (Exception e) {
            log.error("调用调用es接口批次删除资料失败", e);
            throw new BusinessException(e.getMessage());
        }
    }

    private JSONObject executeRhRequest(String modelUrl, JSONObject requestData, String iamToken) {
        String requestUrl = this.rhDomain + URL_COMMON_HDL;
        HttpHeaders httpHeaders = new HttpHeaders();
        httpHeaders.setContentType(MediaType.APPLICATION_JSON);
        httpHeaders.add("token", iamToken);
        httpHeaders.add("digi-middleware-auth-user", iamToken);
        httpHeaders.add("digi-middleware-auth-app", appToken);

        JSONObject requestParam = this.buildRequestBody(modelUrl, requestData);
        HttpEntity<JSONObject> entity = new HttpEntity<>(requestParam, httpHeaders);

        log.info("executeRhRequest #pre,url : {}, param : {}", requestUrl, JSON.toJSONString(entity));
        RestTemplate restTemplate = new RestTemplate();
        ResponseEntity<JSONObject> result = restTemplate.postForEntity(requestUrl, entity, JSONObject.class);
        log.info("executeRhRequest #post,result : {}", JSON.toJSONString(result));

        JSONObject responseData = this.getResponseData(result.getBody());

        return responseData;
    }

    private JSONObject buildRequestBody(String modelUrl, JSONObject data) {
        JSONObject dataBody = new JSONObject();
        dataBody.fluentPut("method", "CLU_IE").fluentPut("model", modelUrl).fluentPut("topic", "AI").fluentPut("timeout", 60).fluentPut("data", data);

        JSONObject body = new JSONObject();
        body.fluentPut("key", "0").fluentPut("asynchronous", false).fluentPut("data", dataBody);

        return body;
    }

    private JSONObject getResponseData(JSONObject responseBody) {
        Integer code = responseBody.getInteger("code");
        if (code != null && code.intValue() == 0) {
            return responseBody.getJSONObject("data");
        }

        return null;
    }
}
