package com.digiwin.athena.km_deployer_service.service.km;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.collection.ListUtil;
import com.digiwin.athena.km_deployer_service.constant.Constant;
import com.digiwin.athena.km_deployer_service.domain.km.CleanMongoParam;
import com.digiwin.athena.deploy.ApplicationMongoData;
import com.digiwin.athena.km_deployer_service.povo.CrudReq;
import com.digiwin.athena.km_deployer_service.povo.PageVo;
import com.mongodb.client.FindIterable;
import com.mongodb.client.MongoCursor;
import com.mongodb.client.MongoIterable;
import com.mongodb.client.model.Filters;
import com.mongodb.client.model.Updates;
import org.bson.BsonDocument;
import org.bson.Document;
import org.bson.conversions.Bson;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.convert.MongoConverter;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;
import java.util.stream.Collectors;

@Service
public class MongoCrudService {

    @Autowired
    MongoTemplate mongoTemplate;

    @Autowired
    private MongoConverter mongoConverter;

    public List<String> listTables(String db){
        List<String> tables = new ArrayList<>();
        MongoIterable<String>  iterable= mongoTemplate.getMongoDatabaseFactory().getMongoDatabase(db).listCollectionNames();
        iterable.forEach(new Consumer<String>() {
            @Override
            public void accept(String s) {
                tables.add(s);
            }
        });

        return tables;
    }


    public void insert(ApplicationMongoData req){
        if(null==req.getDb() || null==req.getCol() || CollectionUtil.isEmpty(req.getDocs())){
            return;
        }

        List<List<Document>> docsSplit = ListUtil.split(req.getDocs(),100);
        for (List<Document> documents : docsSplit) {
            documents.forEach(doc->{doc.remove("_id");});
            mongoTemplate.getMongoDatabaseFactory().getMongoDatabase(req.getDb()).getCollection(req.getCol()).insertMany(documents);
        }
    }

    public void insertOne(CrudReq req){
        if(null==req.getDbName() || null==req.getColName() || null==req.getData()){
            return;
        }
       Document doc = toDoc(req.getData());
        mongoTemplate.getMongoDatabaseFactory().getMongoDatabase(req.getDbName()).getCollection(req.getColName()).insertOne(doc);
    }


    public void delete(CrudReq req){
        if(null==req.getDbName() || null==req.getColName() || CollectionUtil.isEmpty(req.getParams())){
            return;
        }

        mongoTemplate.getMongoDatabaseFactory().getMongoDatabase(req.getDbName()).getCollection(req.getColName()).deleteMany(buildBson(req.getParams()));
    }

    public void update(CrudReq req){
        if(null==req.getDbName() || null==req.getColName() || CollectionUtil.isEmpty(req.getParams())){
            return;
        }
        mongoTemplate.getMongoDatabaseFactory().getMongoDatabase(req.getDbName()).getCollection(req.getColName()).updateMany(buildBson(req.getParams()),buildUpdateSet(req.getData()));
    }

    public static List<Bson> buildUpdateSet(Map<String,Object> data){
        List<Bson> bsons = new ArrayList<>();
        data.forEach((k,v)->{
            bsons.add(Updates.set(k,v));

        });
        return bsons;
    }


    public List<Document> query(CrudReq req){
        if(null==req.getDbName() || null==req.getColName() ){
            return null;
        }
        List list = new ArrayList();
        FindIterable<Document> docs = mongoTemplate.getMongoDatabaseFactory().getMongoDatabase(req.getDbName()).getCollection(req.getColName()).find(buildBson(req.getParams()));
        docs.forEach((Consumer<? super Document>) doc->{
            list.add(doc);
        });

        return list;
    }

    public void updateVersion(ApplicationMongoData req,String app,String version,String updateVersion){
        Map<String,Object> param1 = new HashMap<>();
        param1.put(Constant.athena_namespace,app);
        param1.put(Constant.version,version);
        Bson setv = Updates.set(Constant.version,updateVersion);
        mongoTemplate.getMongoDatabaseFactory().getMongoDatabase(req.getDb()).getCollection(req.getCol()).updateMany(buildBson(param1),setv);
    }


    public void cleanMongoData(CleanMongoParam cleanMongoParam) {
        Map<String, List<String>> collectionInfo = cleanMongoParam.getCollectionInfo();
        String application = cleanMongoParam.getApplication();
        collectionInfo.forEach((dbName, collectionNameList) -> {
            Bson bson = Filters.and(
                    Filters.eq("version", cleanMongoParam.getDeployVersion()),
                    Filters.or(Filters.eq("application", application), Filters.eq("athena_namespace", application)),
                    Filters.or(Filters.eq("tenantId", null), Filters.eq("tenantId", "SYSTEM"), Filters.eq("athena_publishType", "individualCase"))
            );
            for (String collectionName : collectionNameList) {
                mongoTemplate.getMongoDatabaseFactory().getMongoDatabase(dbName).getCollection(collectionName).deleteMany(bson);
            }
        });
    }

    public static List<Document> toDocs(List<Map<String,Object>> docs){
        return docs.stream().map(map->{
            return toDoc(map);
        }).collect(Collectors.toList());
    }

    public static Document toDoc(Map<String,Object> doc){
        return new Document(doc);
    }


    public static Bson buildBson(Map<String,Object> params){
        if(null==params || params.isEmpty()){return new BsonDocument();}
        List<Bson> bsons = new ArrayList<>();
        params.forEach((k,v)->{
            if("OR".equals(k)){
                Map<String,Object> v2 = (Map<String, Object>) v;
                bsons.add(buildBson(v2));
            }else if(v instanceof List){
                bsons.add(Filters.in(k,(List)v));
            }else{
                bsons.add(Filters.eq(k,v));
            }
        });
        return Filters.and(bsons);
    }

    public void processLargeQuery(String db, String col, Map<String, Object> params,
                                  int batchSize, BatchProcessCallBack callBack) {
        try (MongoCursor<Document> cursor = mongoTemplate
                .getMongoDatabaseFactory()
                .getMongoDatabase(db)
                .getCollection(col)
                .find(params.isEmpty() ? new BsonDocument() : buildBson(params))
                .noCursorTimeout(true)
                .batchSize(batchSize)
                .cursor()) {
            List<Map<String, Object>> batch = new ArrayList<>(batchSize);
            while (cursor.hasNext()) {
                Document doc = cursor.next();
                batch.add(doc);
                if (batch.size() == batchSize) {
                    callBack.process(batch, false);
                    batch.clear();
                }
            }
            callBack.process(batch, true);
        }
    }

    public <T> PageVo<T> queryWithPage(String dbName, String colName,
                                       Map<String, Object> params, int page, int pageSize, Class<T> tClass) {
        Bson bson = buildBson(params);
        long totalCount = mongoTemplate.getMongoDatabaseFactory().getMongoDatabase(dbName).getCollection(colName).countDocuments(bson);
        FindIterable<Document> findIterable = mongoTemplate.getMongoDatabaseFactory().getMongoDatabase(dbName).getCollection(colName)
                .find(bson, Document.class).skip((page - 1) * pageSize).limit(pageSize);
        List<T> list = new ArrayList<>();
        for (Document each : findIterable) {
            list.add(mongoConverter.read(tClass, each));
        }
        return new PageVo<>(totalCount, list, page, pageSize, (int) Math.ceil((double) totalCount / pageSize));
    }

    public <T> List<T> find(String db, String col, Bson filter, Bson projections, Class<T> tClass) {
        FindIterable<Document> findIterable = mongoTemplate.getMongoDatabaseFactory().getMongoDatabase(db).getCollection(col)
                .find(filter).projection(projections);
        List<T> list = new ArrayList<>();
        for (Document each : findIterable) {
            list.add(mongoConverter.read(tClass, each));
        }
        return list;
    }
}
