package com.digiwin.athena.km_deployer_service.service.km.handler;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.map.MapUtil;
import cn.hutool.core.util.HashUtil;
import cn.hutool.core.util.RandomUtil;
import cn.hutool.http.HttpUtil;
import cn.hutool.http.Method;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.km_deployer_service.constant.Constant;
import com.digiwin.athena.km_deployer_service.domain.km.EventCacheEntity;
import com.digiwin.athena.km_deployer_service.domain.km.KmEventParam;
import com.digiwin.athena.km_deployer_service.neo4jbasepkg.master.repository.KmPublishRepo;
import com.digiwin.athena.deploy.ApplicationMongoData;
import com.digiwin.athena.km_deployer_service.povo.CrudReq;
import com.digiwin.athena.km_deployer_service.service.km.MongoCrudService;
import com.digiwin.athena.km_deployer_service.service.km.Neo4jCrudService;
import com.mongodb.client.FindIterable;
import com.mongodb.client.model.Filters;
import com.mongodb.client.model.Sorts;
import lombok.extern.slf4j.Slf4j;
import org.bson.Document;
import org.bson.conversions.Bson;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.*;
import java.util.function.Consumer;
import java.util.stream.Collectors;

/**
 * @Author QinQiang
 * @Description
 * @Date 2024/1/19
 **/
@Slf4j
@Service
public class MonitorRuleKmDeployEventHandler extends AbstractKmDeployEventHandler {

    private final static String DEFAULT_VERSION = "1.0";

    @Resource
    private MongoCrudService mongoCrudService;
    @Resource
    private MongoTemplate mongoTemplate;
    @Resource
    private Neo4jCrudService neo4jCrudService;

    @Value("${module.km.domain}/restful/service/knowledgegraph/")
    private String knowledgeGraphUrl;
    @Resource
    KmPublishRepo kmPublishRepo;

    private static final List<String> EXCLUDE_FIELDS = Arrays.asList("_id", "id", "version", "compileVersion", "oldNodeId", "publishTime");

    @Override
    public String handlerComponentType() {
        return "monitorRule";
    }


    @Override
    public void handleBeforePublish(KmEventParam param) {
        // 1、查询应用下所有的侦测
        Map<String, Integer> monitorRulesHashData = getMonitorRulesHashData(param, null, null, null);
        // 2、将每个侦测数据hash
        List<EventCacheEntity> monitorChangeCacheEntities = new ArrayList<>();
        for (Map.Entry<String, Integer> entry : monitorRulesHashData.entrySet()) {
            EventCacheEntity monitorChangeCacheEntity = new EventCacheEntity()
                    .setEventId(param.getEventId())
                    .setObjectCode(entry.getKey())
                    .setObjectType("monitorRule")
                    .setHash(entry.getValue());
            monitorChangeCacheEntities.add(monitorChangeCacheEntity);
        }

        // 3、入库
        ApplicationMongoData insertRequest = new ApplicationMongoData();
        insertRequest.setDb(Constant.db_kg_sys);
        insertRequest.setCol("kmEventDeployCache");
        List<Document> documents = monitorChangeCacheEntities.stream().map(e -> Document.parse(JSON.toJSONString(e))).collect(Collectors.toList());
        insertRequest.setDocs(documents);
        mongoCrudService.insert(insertRequest);
    }

    @Override
    public void handleAfterPublish(KmEventParam param) {
        compareAndPush(param, 1);
        // 更新事件状态为已完成发版
        updateEventStatus(param.getEventId(), 2);
    }

    @Override
    public void handleBeforeSwitch(KmEventParam param) {
        // 不需要做任何操作
    }

    @Override
    public void handleAfterSwitch(KmEventParam param) {
        // 跟发布事件一样
        compareAndPush(param, 2);
    }

    /**
     * 比较并推送
     *
     * @param param  入参
     * @param status 事件状态
     */
    private void compareAndPush(KmEventParam param, int status) {
        // 1、找到应用发版成功最新的eventId
        List<Document> events = new ArrayList<>();
        // 发版后找最新的发版中的事件，切版后找最新的发版完成的事件
        Bson filter = Filters.and(Filters.eq("appId", param.getAppCode()), Filters.eq("status", status));
        Bson sort = Sorts.descending("updateTime");
        FindIterable<Document> docs = mongoTemplate.getMongoDatabaseFactory().getMongoDatabase(Constant.db_kg_sys).getCollection("kmDeployEvent").find(filter).sort(sort).limit(1);
        docs.forEach((Consumer<? super Document>) events::add);

        if (CollectionUtil.isEmpty(events)) {
            log.warn("未找到发版事件");
            return;
        }
        String eventId = events.get(0).getString("eventId");
        param.setEventId(eventId);
        // 2、查询应用下发版后所有的侦测
        List<String> spacialMonitorRuleCodes = new ArrayList<>();
        List<String> directPushMonitorRuleCodes = new ArrayList<>();
        Map<String, Map<String, Object>> newMonitorRule = new HashMap<>();
        Map<String, Integer> monitorRulesHashData = getMonitorRulesHashData(param, spacialMonitorRuleCodes, directPushMonitorRuleCodes, newMonitorRule);
        // 3、查询出发版时已经缓存的hash数据
        CrudReq req = new CrudReq();
        req.setDbName(Constant.db_kg_sys);
        req.setColName("kmEventDeployCache");
        req.setParams(new HashMap<>());
        req.getParams().put("eventId", eventId);
        req.getParams().put("objectType", "monitorRule");
        List<Document> monitorChangeCacheEntities = mongoCrudService.query(req);
        // 本次掉删除的侦测
        Set<String> deletedMonitorRuleCodes = monitorChangeCacheEntities.stream().map(e -> MapUtil.getStr(e, "objectCode")).filter(key -> !monitorRulesHashData.containsKey(key)).collect(Collectors.toSet());
        if (CollectionUtil.isNotEmpty(deletedMonitorRuleCodes)) {
            log.info("检测到被删除的侦测code: {}", JSON.toJSONString(deletedMonitorRuleCodes));
            // 查出这些侦测中还在运行的，去停用调
            List<Document> ruleTenants = getTenantMonitorRules(deletedMonitorRuleCodes, 1);
            for (Document ruleTenant : ruleTenants) {
                JSONObject config = new JSONObject();
                JSONObject eocMap = new JSONObject();
                eocMap.put("eocCompanyId", "");
                config.fluentPut("tenantId", ruleTenant.getString("tenantId"))
                        .fluentPut("monitorRuleId", ruleTenant.getString("monitorRuleId"))
                        .fluentPut("eocMap", ruleTenant.get("eocMap") == null ? eocMap : ruleTenant.get("eocMap"));
                deleteMonitorRules(config);
            }

        }
        // 转成map
        Map<String, Map<String, Object>> monitorChangeCacheEntityMap = monitorChangeCacheEntities.stream().collect(Collectors.toMap(o -> MapUtil.getStr(o, "objectCode"), o -> o));
        Set<String> pushMonitorRuleCodes = new HashSet<>();
        Set<String> newMonitorRuleCodes = new HashSet<>();
        // 4、对比每个侦测的数据hash后是否有变化
        for (Map.Entry<String, Integer> entry : monitorRulesHashData.entrySet()) {
            Map<String, Object> monitorChangeCacheEntity = monitorChangeCacheEntityMap.get(entry.getKey());
            if (Objects.isNull(monitorChangeCacheEntity) && (spacialMonitorRuleCodes.contains(entry.getKey()) || directPushMonitorRuleCodes.contains(entry.getKey()))) {
                // 新加的侦测并且actionType=workflow或者为个案
                newMonitorRuleCodes.add(entry.getKey());
            } else {
                // 个案不用对比
                if (directPushMonitorRuleCodes.contains(entry.getKey())) {
                    pushMonitorRuleCodes.add(entry.getKey());
                } else if (!entry.getValue().equals(MapUtil.getInt(monitorChangeCacheEntity, "hash"))) {
                    // 有变化
                    pushMonitorRuleCodes.add(entry.getKey());
                }
            }
        }
        // 对于新增的侦测需要提前往租户表插数据
        if (CollectionUtil.isNotEmpty(newMonitorRuleCodes)) {
            log.info("检测到新增的侦测code: {}", JSON.toJSONString(newMonitorRuleCodes));
            Set<String> useAppTenants = kmPublishRepo.findUseAppTenantsWithOutVersion(param.getAppCode());
            if (CollectionUtil.isNotEmpty(useAppTenants)) {
                List<Document> ruleTenants = getTenantMonitorRules(newMonitorRuleCodes, null);

                List<Document> documents = new ArrayList<>();
                for (String tenantId : useAppTenants) {
                    for (String code : newMonitorRuleCodes) {
                        boolean b = ruleTenants.stream().anyMatch(ruleTenant -> ruleTenant.getString("monitorRuleId").equals(code) && ruleTenant.getString("tenantId").equals(tenantId));
                        // 已经存在数据
                        if (b) {
                            continue;
                        }
                        Map<String, Object> ruleData = newMonitorRule.get(code);
                        Document d = new Document();
                        d.put("monitorRuleId", code);
                        d.put("tenantId", tenantId);
                        d.put("status", 1);
                        d.put("productName", ruleData.containsKey("productName") ? MapUtil.getStr(ruleData, "productName") : param.getAppCode());
                        d.put("configId", RandomUtil.randomStringUpper(33));
                        if (ruleData.containsKey("standardPollingRule")) {
                            d.put("standardPollingRule", JSONObject.parse(MapUtil.getStr(ruleData, "standardPollingRule")));
                        }
                        if (ruleData.containsKey("standardPollingRules")) {
                            d.put("standardPollingRules", JSONArray.parse(MapUtil.getStr(ruleData, "standardPollingRule")));
                        }
                        if (ruleData.containsKey("eocMap")) {
                            d.put("eocMap", ruleData.get("eocMap"));
                        }
                        documents.add(d);
                    }
                }
                for (Document document : documents) {
                    startMonitorRules(document);
                }
            }

        }
        log.info("检测到发生变动的侦测code: {}", JSON.toJSONString(pushMonitorRuleCodes));
        // 调用kg接口推送到sd
        pushModifyEvent(pushMonitorRuleCodes, param.getVersion());
    }

    private List<Document> getTenantMonitorRules(Set<String> monitorRulesCodes, Integer status) {
        CrudReq ruleTenantQuery = new CrudReq();
        ruleTenantQuery.setDbName(Constant.db_kg);
        ruleTenantQuery.setColName("monitorRuleTenantConfig");
        ruleTenantQuery.setParams(new HashMap<>());
        ruleTenantQuery.getParams().put("monitorRuleId", new ArrayList<>(monitorRulesCodes));
        if (status != null) {
            ruleTenantQuery.getParams().put("status", status);
        }
        return mongoCrudService.query(ruleTenantQuery);
    }

    private void startMonitorRules(Document document) {
        JSONObject paramJson = new JSONObject();
        paramJson.put("config", document);
        requestKg("MonitorRuleConfig", paramJson, Method.PUT);
    }

    /**
     * 推送变更的侦测
     *
     * @param pushMonitorRuleCodes 变更的侦测code
     * @param version
     */
    private void pushModifyEvent(Set<String> pushMonitorRuleCodes, String version) {
        if (CollectionUtil.isEmpty(pushMonitorRuleCodes)) {
            return;
        }
        JSONObject noticeInfo = new JSONObject();
        noticeInfo.fluentPut("ruleIds", pushMonitorRuleCodes)
                .fluentPut("version", version)
                .fluentPut("limitedTenantIdList", Collections.emptyList())
                .fluentPut("excludedTenantIdList", Collections.emptyList())
                .fluentPut("productChangeInfo", Collections.emptyMap());
        JSONObject paramJson = new JSONObject();
        paramJson.put("noticeInfo", noticeInfo);
        requestKg("MonitorRuleConfig/noticeWhenMonitorRuleConfigChange", paramJson, Method.POST);
    }


    private void deleteMonitorRules(JSONObject paramJson) {
        JSONObject param = new JSONObject();
        param.put("config", paramJson);
        requestKg("MonitorRuleConfig", param, Method.DELETE);
    }

    /**
     * 更新事件状态
     *
     * @param eventId 事件id
     * @param status  状态
     */
    private void updateEventStatus(String eventId, int status) {
        CrudReq crudReq = new CrudReq();
        crudReq.setDbName(Constant.db_kg_sys);
        crudReq.setColName("kmDeployEvent");
        crudReq.setParams(new HashMap<>());
        crudReq.getParams().put("eventId", eventId);
        crudReq.setData(new HashMap<>());
        crudReq.getData().put("status", status);
        crudReq.getData().put("updateTime", System.currentTimeMillis());
        mongoCrudService.update(crudReq);
    }

    /**
     * 获取侦测数据
     *
     * @param param                      入参
     * @param spacialMonitorRuleCodes    需要特殊判断的侦测code
     * @param directPushMonitorRuleCodes 直接推送的侦测code
     * @param newMonitorRule             侦测数据
     * @return {侦测code：hash}
     */
    private Map<String, Integer> getMonitorRulesHashData(KmEventParam param, List<String> spacialMonitorRuleCodes, List<String> directPushMonitorRuleCodes, Map<String, Map<String, Object>> newMonitorRule) {
        CrudReq req = new CrudReq();
        req.setDbName(Constant.db_kg_sys);
        req.setColName("applicationRelation");
        req.setParams(new HashMap<>());
        req.getParams().put("version", DEFAULT_VERSION);
        req.getParams().put("appCode", param.getAppCode());
        req.getParams().put("type", "monitorRule");
        List<Document> applicationRelations = mongoCrudService.query(req);

        if (CollectionUtil.isEmpty(applicationRelations)) {
            return Collections.emptyMap();
        }
        // 侦测code
        List<String> monitorCodes = applicationRelations.stream().map(e -> MapUtil.getStr(e, "code")).collect(Collectors.toList());
        // 查询侦测定义--产品级定义
        CrudReq monitorRuleProductConfigReq = new CrudReq();
        monitorRuleProductConfigReq.setDbName(Constant.db_kg_sys);
        monitorRuleProductConfigReq.setColName("monitorRuleProductConfig");
        monitorRuleProductConfigReq.setParams(new HashMap<>());
        monitorRuleProductConfigReq.getParams().put("version", DEFAULT_VERSION);
        monitorRuleProductConfigReq.getParams().put("application", param.getAppCode());
        monitorRuleProductConfigReq.getParams().put("monitorRuleId", monitorCodes);
        List<Document> monitorRuleProductConfigs = mongoCrudService.query(monitorRuleProductConfigReq);
        Map<String, Document> monitorRuleProductConfigMap = new HashMap<>();
        if (CollectionUtil.isNotEmpty(monitorRuleProductConfigs)) {
            // 找出monitorRuleId相同的侦测，将productName字段合并，用都逗号隔开
            Map<Object, String> mergedMap = monitorRuleProductConfigs.stream()
                    .collect(Collectors.groupingBy(rule -> MapUtil.getStr(rule, "monitorRuleId"),
                            Collectors.mapping(rule -> MapUtil.getStr(rule, "productName"), Collectors.joining(","))));
            // 转成map
            Map<String, Document> resultMap = new HashMap<>();
            mergedMap.forEach((key, value) -> {
                Document originalRule = monitorRuleProductConfigs.stream()
                        .filter(rule -> rule.get("monitorRuleId").equals(key))
                        .findFirst()
                        .orElse(null);
                if (originalRule != null) {
                    // 将产品排序一下
                    String[] split = value.split(",");
                    Arrays.sort(split);
                    String mergedProductName = String.join(",", split);
                    originalRule.put("productName", mergedProductName);
                    resultMap.put((String) key, originalRule);
                }
            });
            monitorRuleProductConfigMap.putAll(resultMap);
        }

        Map<String, Object> monitorParam = new HashMap<>();
        monitorParam.put("athena_namespace", param.getAppCode());
        monitorParam.put("version", DEFAULT_VERSION);
        monitorParam.put("code", monitorCodes);
        List<Map<String, Object>> monitorRules = neo4jCrudService.query("MonitorRule", monitorParam);
        Map<String, Map<String, Object>> monitorRuleMap = new HashMap<>();
        //  查询基础定义
        if (CollectionUtil.isNotEmpty(monitorRules)) {
            monitorRuleMap = monitorRules.stream().collect(Collectors.toMap(e -> MapUtil.getStr(e, "code"), e -> e));
        }
        Map<String, Integer> minotorRuleHashMap = new HashMap<>();
        for (String code : monitorCodes) {
            // 合并侦测定义
            Map<String, Object> map = new Hashtable<>();
            Document monitorRuleProductConfig = monitorRuleProductConfigMap.get(code);
            if (MapUtil.isNotEmpty(monitorRuleProductConfig)) {
                map.putAll(monitorRuleProductConfig);
            }
            Map<String, Object> monitorRule = monitorRuleMap.get(code);
            if (!Objects.isNull(monitorRule)) {
                map.putAll(monitorRule);
            }
            if (MapUtil.isNotEmpty(map)) {
                // 剔除不需要的字段
                EXCLUDE_FIELDS.forEach(map::remove);
                int hash = HashUtil.apHash(JSON.toJSONString(map));
                minotorRuleHashMap.put(code, hash);
                // 即新增的情况如果actionType=workflow 也需要推送到sd
                if (spacialMonitorRuleCodes != null && "workflow".equals(MapUtil.getStr(map, "actionType"))) {
                    spacialMonitorRuleCodes.add(code);
                    newMonitorRule.put(code, map);
                }
                // 个案可以不用对比直接推送sd
                // mongo里面会加"athena_publishType"："individualCase"
                // neo4j里面会加属性inclusionTenant（数组）
                if (directPushMonitorRuleCodes != null && (map.get("inclusionTenant") != null || "individualCase".equals(MapUtil.getStr(map, "athena_publishType")))) {
                    directPushMonitorRuleCodes.add(code);
                }
                log.info("{}侦测数据：{}", code, JSON.toJSONString(map));
            }
        }
        return minotorRuleHashMap;
    }

    /**
     * 请求kg接口
     *
     * @param serviceName 接口路径
     * @param paramJson   参数
     * @param method      请求方式
     */
    public void requestKg(String serviceName, JSONObject paramJson, Method method) {
        String url = knowledgeGraphUrl + serviceName;
        try {
            String response = HttpUtil.createRequest(method, url).body(paramJson.toJSONString()).execute().body();
            log.info("请求kg成功:{}", response);
        } catch (Exception e) {
            log.error("请求kg失败:{}", e.getMessage());
        }

    }
}
