package com.digiwin.athena.km_deployer_service.service.km.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.convert.Convert;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.km_deployer_service.config.neo4j.Neo4jManager;
import com.digiwin.athena.km_deployer_service.constant.Constant;
import com.digiwin.athena.km_deployer_service.domain.neo4j.Cql;
import com.digiwin.athena.km_deployer_service.domain.neo4j.FromNode;
import com.digiwin.athena.km_deployer_service.domain.neo4j.Relation;
import com.digiwin.athena.km_deployer_service.domain.neo4j.ToNode;
import com.digiwin.athena.km_deployer_service.povo.CreateApplicationRelationParam;
import com.digiwin.athena.km_deployer_service.povo.UpdateVersionParam;
import com.digiwin.athena.km_deployer_service.service.km.ApplicationService;
import com.digiwin.athena.km_deployer_service.service.km.TenantService;
import com.digiwin.athena.km_deployer_service.util.MongoHelper;
import com.digiwin.athena.km_deployer_service.util.Neo4jMultipleUtil;
import com.mongodb.client.FindIterable;
import com.mongodb.client.model.Filters;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.text.StringEscapeUtils;
import org.bson.Document;
import org.bson.conversions.Bson;
import org.neo4j.driver.Driver;
import org.neo4j.driver.internal.InternalNode;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Consumer;

/**
 * @author liyuetao
 * @title: ApplicationServiceImpl
 * @projectName athena_deployer_service
 * @description: TODO
 * @date 2022/11/109:59
 */
@Service
@Slf4j
public class ApplicationServiceImpl implements ApplicationService {

    private static ConcurrentHashMap<String, ReentrantLock> LOCK_MAP = new ConcurrentHashMap();

    @Autowired
    private Driver driver1;

    @Autowired(required = false)
    @Qualifier("domain2Driver")
    private Driver driver2;

    @Autowired
    private MongoTemplate mongoTemplate;

    @Autowired
    private MongoHelper mongoHelper;
    @Autowired
    private TenantService tenantService;



    @Override
    public void updateVersion(UpdateVersionParam updateVersionParam) {
        List<Cql> cqlList = getUpdateAppDataVersionCqlList(updateVersionParam);

        Neo4jMultipleUtil.executeCqlTrans(cqlList, driver1, driver2);
    }

    @Override
    public List<Cql> getUpdateVersionCqlList(UpdateVersionParam updateVersionParam) {
        List<Cql> cqlList = getUpdateAppDataVersionCqlList(updateVersionParam);

        return cqlList;
    }

    private List<Cql> getUpdateAppDataVersionCqlList(UpdateVersionParam updateVersionParam) {
        // common当作正常应用来看，不跟espCommon放在一起，这边入参只有一个应用code
        String application = updateVersionParam.getApplication();
        String oldVersion = updateVersionParam.getOldVersion();
        String newVersion = updateVersionParam.getNewVersion();
        List<String> tenantIdList = updateVersionParam.getTenantIdList();
        List<JSONObject> publishDbMongoData = updateVersionParam.getPublishDbMongoData();

        //更新Mongo数据version,先删除新版本数据，再将旧版本数据的version更新成新版本
        Bson filter = Filters.and(
                Filters.eq("version", oldVersion),
                Filters.or(Filters.eq("application", application), Filters.eq("athena_namespace", application)),
                Filters.or(Filters.eq("tenantId", null), Filters.eq("tenantId", "SYSTEM"))
        );
        Bson deleteFilter = Filters.and(
                Filters.eq("version", newVersion),
                Filters.or(Filters.eq("application", application), Filters.eq("athena_namespace", application)),
                Filters.or(Filters.eq("tenantId", null), Filters.eq("tenantId", "SYSTEM"))
        );
//        Bson filter = Filters.and(Filters.eq("version", oldVersion), Filters.eq("athena_namespace", application));
//        Bson deleteFilter = Filters.and(Filters.eq("version", newVersion), Filters.eq("athena_namespace", application));
        Bson update = new Document("$set", new Document().append("version", newVersion));
        mongoHelper.deleteAndUpdate(publishDbMongoData, filter, deleteFilter, update);
        // 获取租户当前对应的版本信息
        Map<String, String> tenantVersionMap = tenantService.getTenantVersion(tenantIdList);
        //更新neo4j数据version,先删除新版本数据，再将旧版本数据的version更新成新版本
        List<Cql> cqlList = new ArrayList<>();

        HashMap<String, Object> param = new HashMap<>();
        param.put("oldVersion", oldVersion);
        param.put("newVersion", newVersion);
        param.put("application", application);
        Cql updateCql = new Cql();
        updateCql.setCql("match (node) where node.version = $oldVersion and node.athena_namespace = $application set node.version=$newVersion");
        updateCql.setParams(param);

        Cql deleteCql = new Cql();
        deleteCql.setCql("match (node) where node.version = $newVersion and node.athena_namespace = $application detach delete node");
        deleteCql.setParams(param);

        Cql updateTenantCql = new Cql();
        param.put("tenantIdList", tenantIdList);
        updateTenantCql.setCql("match (n:TenantEntity) where n.tenantId in $tenantIdList set n.version=$newVersion");
        updateTenantCql.setParams(param);

        cqlList.add(deleteCql);
        cqlList.add(updateCql);
        cqlList.add(updateTenantCql);

        if (!Constant.COMMON_CODE.equals(application)) {
            // 非common应用需要更新appEntity的version
            Cql updateAppCql = new Cql();
            updateAppCql.setCql("match (n:AppEntity) where n.code = $application set n.version=$newVersion");
            updateAppCql.setParams(param);
            cqlList.add(updateAppCql);
        }
        // 处理切版应用的租户和common/espCommon之间的关系
        List<Cql> cqls = tenantService.mergeRelationBetweenTenantAndCommon(application, tenantIdList, newVersion, tenantVersionMap);
        cqlList.addAll(cqls);
        return cqlList;
    }

    // 原始的入参是应用list时的逻辑，入参是一个应用时走getUpdateAppDataVersionCqlList逻辑
    private List<Cql> getUpdateAppListDataVersionCqlList(UpdateVersionParam updateVersionParam) {
        List<String> applicationList = updateVersionParam.getApplicationList();
        String oldVersion = updateVersionParam.getOldVersion();
        String newVersion = updateVersionParam.getNewVersion();
        List<String> tenantIdList = updateVersionParam.getTenantIdList();
        List<JSONObject> publishDbMongoData = updateVersionParam.getPublishDbMongoData();
        //更新Mongo数据version,先删除新版本数据，再将旧版本数据的version更新成新版本
        Bson filter = Filters.and(Filters.eq("version", oldVersion), Filters.in("athena_namespace", applicationList));
        Bson deleteFilter = Filters.and(Filters.eq("version", newVersion), Filters.in("athena_namespace", applicationList));
        Bson update = new Document("$set", new Document().append("version", newVersion));
        mongoHelper.deleteAndUpdate(publishDbMongoData, filter, deleteFilter, update);

        //更新neo4j数据version,先删除新版本数据，再将旧版本数据的version更新成新版本
        List<Cql> cqlList = new ArrayList<>();

        HashMap<String, Object> param = new HashMap<>();
        param.put("oldVersion", oldVersion);
        param.put("newVersion", newVersion);
        param.put("applicationList", applicationList);
        Cql updateCql = new Cql();
        updateCql.setCql("match (node) where node.version = $oldVersion and node.athena_namespace in $applicationList set node.version=$newVersion");
        updateCql.setParams(param);

        Cql deleteCql = new Cql();
        deleteCql.setCql("match (node) where node.version = $newVersion and node.athena_namespace in $applicationList detach delete node");
        deleteCql.setParams(param);

        Cql updateTenantCql = new Cql();
        param.put("tenantIdList", tenantIdList);
        updateTenantCql.setCql("match (n:TenantEntity) where n.tenantId in $tenantIdList set n.version=$newVersion");
        updateTenantCql.setParams(param);

        Cql updateAppCql = new Cql();
        updateAppCql.setCql("match (n:AppEntity) where n.code in $applicationList set n.version=$newVersion");
        updateAppCql.setParams(param);

        cqlList.add(deleteCql);
        cqlList.add(updateCql);
        cqlList.add(updateTenantCql);
        cqlList.add(updateAppCql);
        return cqlList;
    }

    @Override
    public void createApplication2CommonRelation(CreateApplicationRelationParam createApplicationRelationParam) {
        String applicationVersion = createApplicationRelationParam.getApplicationVersion();
        String commonVersion = createApplicationRelationParam.getCommonVersion();
        JSONObject neo4jNodeKeyJson = createApplicationRelationParam.getNeo4jNodeKeyJson();
        List<Relation> relationList = createApplicationRelationParam.getRelationList();
        if (CollUtil.isEmpty(relationList)) {
            return;
        }
        relationList.stream().forEach(relation -> {
            relation.setFromNodeVersion(applicationVersion);
            relation.setToNodeVersion(commonVersion);
        });
        List<Cql> cqlList = getNodeRelationCql(neo4jNodeKeyJson, relationList);
        Neo4jMultipleUtil.executeCqlTrans(cqlList, driver1, driver2);
    }

    @Override
    public List<Cql> getCqlByRelation(CreateApplicationRelationParam createApplicationRelationParam) {
        String applicationVersion = createApplicationRelationParam.getApplicationVersion();
        String commonVersion = createApplicationRelationParam.getCommonVersion();
        JSONObject neo4jNodeKeyJson = createApplicationRelationParam.getNeo4jNodeKeyJson();
        List<Relation> relationList = createApplicationRelationParam.getRelationList();
        if (CollUtil.isEmpty(relationList)) {
            return new ArrayList<>();
        }
        relationList.stream().forEach(relation -> {
            relation.setFromNodeVersion(applicationVersion);
            relation.setToNodeVersion(commonVersion);
        });
        List<Cql> cqlList = getNodeRelationCql(neo4jNodeKeyJson, relationList);

        return cqlList;
    }

    @Override
    public void combineNodeProperties(Map<String, Object> properties, StringBuffer nodeCypher) {
        properties.forEach((k, v) -> {
            if (!"version".equals(k) && !"oldNodeId".equals(k)) {
                if (k.contains(".")) {
                    nodeCypher.append(String.format("`%s`:", k));
                } else {
                    nodeCypher.append(String.format("%s:", k));
                }
                if (v instanceof String) {
                    String propertyValue = (String) v;
                    propertyValue = propertyValue.replace("'", "\\\"");
                    nodeCypher.append(String.format("'%s',", propertyValue));
                } else if (v instanceof Collection) {
                    nodeCypher.append("[");
                    List propertyValueList = (List) v;
                    propertyValueList.forEach(propertyValue -> {
                        if (propertyValue instanceof String) {
                            nodeCypher.append(String.format("'%s'", propertyValue)).append(",");
                        } else {
                            nodeCypher.append(propertyValue).append(",");
                        }
                    });
                    if (!propertyValueList.isEmpty()) {
                        nodeCypher.deleteCharAt(nodeCypher.length() - 1);
                    }
                    nodeCypher.append("],");
                } else {
                    nodeCypher.append(v).append(",");
                }
            }
        });
    }

    public List<Cql> getNodeRelationCql(JSONObject neo4jNodeKeyJson, List<Relation> relations) {
        List<Cql> cqlList = new ArrayList<>();
        for (Relation relation : relations) {
            FromNode fromNode = relation.getFromNode();
            ToNode toNode = relation.getToNode();
            String cqlStr = "match (fromNode:{}) where fromNode.{} = {} and fromNode.athena_namespace = '{}' and fromNode.version='{}'  match(toNode:{}) where toNode.{} = {} and toNode.athena_namespace = '{}' and toNode.version='{}' merge (fromNode)-[:{}]->(toNode)";
            Object fromNodePrimaryKey = fromNode.getPrimaryKey();
            if (fromNodePrimaryKey instanceof String) {
                fromNodePrimaryKey = "'" + fromNodePrimaryKey + "'";
            }
            Object toNodePrimaryKey = toNode.getPrimaryKey();
            if (toNodePrimaryKey instanceof String) {
                toNodePrimaryKey = "'" + toNodePrimaryKey + "'";
            }
            cqlStr = StrUtil.format(cqlStr,
                    fromNode.getLabel(),
                    neo4jNodeKeyJson.getString(fromNode.getLabel()),
                    fromNodePrimaryKey,
                    fromNode.getApplication(),
                    fromNode.getVersion(),
                    toNode.getLabel(),
                    neo4jNodeKeyJson.getString(toNode.getLabel()),
                    toNodePrimaryKey,
                    toNode.getApplication(),
                    toNode.getVersion(),
                    toNode.getType());
            Cql cql = new Cql().setCql(cqlStr);
            cqlList.add(cql);
        }
        return cqlList;
    }

    /**
     * 　@description: 复制neo4j数据
     * 　@author liyuetao
     * 　@date 2022/9/13 10:57
     */
    private void copyNeo4jData(String application, String oldVersion, String newVersion, Neo4jManager neo4jManager) {
        long t0 = System.currentTimeMillis();
        // 作为查询
        ConcurrentHashMap<Long, Long> nodeNeo4jIdMap = new ConcurrentHashMap<>();
        // 收集所有的node节点
        HashMap<String, Object> param = new HashMap<>();
        param.put("version", oldVersion);
        param.put("application", application);
        List<Map<String, Object>> nodeResultList = neo4jManager.ExecuteQuery("match (node) where node.version = $version and (node.athena_namespace = $application or node.nameSpace = $application) and not any(label in labels(node) WHERE label in ['TenantEntity','AppEntity']) return node", param);
        // 收集所有的relation以及前后nodeId(排除租户对外的relation)
        List<Map<String, Object>> relationList = neo4jManager.ExecuteQuery("match (startNode)-[relation]->(endNode) where startNode.version = $version and (startNode.athena_namespace = $application or startNode.nameSpace = $application) and endNode.version = $version and (endNode.athena_namespace = $application or endNode.nameSpace = $application) return id(startNode) as startNodeId,type(relation) as relationType,id(endNode) as endNodeId", param);
        // 遍历nodeResultList组装新的节点属性，添加oldNodeId属性
        List<Cql> cqlList = new ArrayList<>();
        for (Map<String, Object> nodeResult : nodeResultList) {
            StringBuffer nodeCypher = new StringBuffer("create (node");
            Collection<String> labels = ((InternalNode) nodeResult.get("node")).labels();
            long nodeId = ((InternalNode) nodeResult.get("node")).id();
            for (String label : labels) {
                nodeCypher.append(String.format(":%s", label));
            }
            nodeCypher.append("{");
            Map<String, Object> properties = ((InternalNode) nodeResult.get("node")).asMap();
            // 取到属性组装cql
            combineNodeProperties(properties, nodeCypher);
            nodeCypher.append("oldNodeId:" + nodeId + ",");
            nodeCypher.append(String.format("version:'%s', publishTime:'%s'}) return id(node) as nodeId", newVersion, DateUtil.now()));
            cqlList.add(new Cql().setCql(StringEscapeUtils.escapeJava(nodeCypher.toString())));
        }
        // 执行所有复制操作
        neo4jManager.ExecuteTransactionNoQuery(cqlList);

        param.put("version", newVersion);
        param.put("application", application);
        List<Map<String, Object>> newNodeList = neo4jManager.ExecuteQuery("match (node) where node.version = $version and (node.athena_namespace = $application or node.nameSpace = $application) and not any(label in labels(node) WHERE label in ['TenantEntity','AppEntity']) return node", param);
        for (Map<String, Object> nodeResult : newNodeList) {
            long newNodeId = ((InternalNode) nodeResult.get("node")).id();
            Map<String, Object> properties = ((InternalNode) nodeResult.get("node")).asMap();
            Long oldNodeId = Convert.toLong(properties.get("oldNodeId"));
            nodeNeo4jIdMap.put(oldNodeId,newNodeId);
        }

        relationList.forEach(relation -> {
            relation.put("startNodeId", nodeNeo4jIdMap.get(Long.valueOf(relation.get("startNodeId").toString())));
            relation.put("endNodeId", nodeNeo4jIdMap.get(Long.valueOf(relation.get("endNodeId").toString())));
        });

        List<Cql> cqlList1 = new ArrayList<>();
        for (Map<String, Object> relation : relationList) {
            String relationCypher = String.format("match (startNode),(endNode) WHERE id(startNode)=%d and id(endNode)=%d merge (startNode)-[relation:%s]->(endNode)", (Long) relation.get("startNodeId"), (Long) relation.get("endNodeId"), relation.get("relationType"));
            cqlList1.add(new Cql().setParams(new HashMap<>()).setCql(relationCypher));
        }
        neo4jManager.ExecuteTransactionNoQuery(cqlList1);
        long t1 = System.currentTimeMillis();
        log.info(application + "应用复制neo4j数据耗时（ms）：" + (t1 - t0));
    }

    /**
     * 　@description: 复制mongodb数据
     * 　@author liyuetao
     * 　@date 2022/9/13 10:57
     */
    private void copyMongoData(String application, String oldVersion, String newVersion, List<JSONObject> publishDbMongoData) {
        Bson filters = Filters.and(
                Filters.eq("version", oldVersion),
                Filters.or(Filters.eq("application", application), Filters.eq("athena_namespace", application)),
                Filters.or(Filters.eq("tenantId", null), Filters.eq("tenantId", "SYSTEM"))
        );
        for (JSONObject dbMongo : publishDbMongoData) {
            ((List<String>) dbMongo.get("collectionName")).forEach(collection -> {
                List<Document> newDocuments = new ArrayList<>();
                FindIterable<Document> documents = mongoTemplate.getMongoDatabaseFactory().getMongoDatabase((String) dbMongo.get("dbName")).getCollection(collection).find(filters);
                documents.forEach((Consumer<? super Document>) document -> {
                    Document newDocument = new Document(document);
                    newDocument.remove("_id");
                    newDocument.put("version", newVersion);
                    newDocument.put("publishTime", new Date());
                    newDocuments.add(newDocument);
                });
                if (!newDocuments.isEmpty())
                    mongoTemplate.getMongoDatabaseFactory().getMongoDatabase((String) dbMongo.get("dbName")).getCollection(collection).insertMany(newDocuments);
            });
        }
    }

    /**
     * 　@description: 回滚数据
     * 　@author liyuetao
     * 　@date 2022/11/1 10:34
     */
    private void revert(String application, String version, List<JSONObject> publishDbMongoData) {
        //删除neo4j数据
        String cql1 = StrUtil.format("MATCH (node)  where node.version ='{}' and node.athena_namespace = '{}' and not any(label in labels(node) WHERE label in ['AppEntity','TenantEntity']) detach delete node", version, application);
//        String cql2 = StrUtil.format("MATCH (node)  where node.version ='{}' and node.athena_namespace = '{}' and node.nameSpace in ['espCommon','common'] and not any(label in labels(node) WHERE label in ['AppEntity','TenantEntity']) detach delete node", version, application);
        Neo4jManager neo4jManager1 = new Neo4jManager(driver1);
        neo4jManager1.ExecuteNoQuery(cql1);
//        neo4jManager1.ExecuteNoQuery(cql2);
        if (driver2 != null) {
            Neo4jManager neo4jManager2 = new Neo4jManager(driver2);
            neo4jManager2.ExecuteNoQuery(cql1);
//            neo4jManager2.ExecuteNoQuery(cql2);
        }

        //删除MongoDB数据
        Bson filter1 = Filters.eq("version", version);
        Bson filter2 = Filters.eq("athena_namespace", application);
        Bson filters = Filters.and(filter1, filter2);
        for (JSONObject map : publishDbMongoData) {
            String dbName = (String) map.get("dbName");
            List<String> collections = (List<String>) map.get("collectionName");
            collections.forEach(collection -> {
                mongoTemplate.getMongoDatabaseFactory().getMongoDatabase(dbName).getCollection(collection).deleteMany(filters);
            });
        }
    }
}
