package com.digiwin.athena.km_deployer_service.service.km.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.convert.Convert;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.http.HttpResponse;
import cn.hutool.http.HttpUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.km_deployer_service.config.neo4j.Neo4jManager;
import com.digiwin.athena.km_deployer_service.constant.Constant;
import com.digiwin.athena.km_deployer_service.domain.TenantUser;
import com.digiwin.athena.km_deployer_service.domain.neo4j.Cql;
import com.digiwin.athena.km_deployer_service.domain.neo4j.CqlMapper;
import com.digiwin.athena.km_deployer_service.domain.system.BusinessException;
import com.digiwin.athena.km_deployer_service.domain.system.TenantUserResult;
import com.digiwin.athena.km_deployer_service.service.km.TenantService;
import com.digiwin.athena.km_deployer_service.util.Neo4jMultipleUtil;
import com.mongodb.client.FindIterable;
import com.mongodb.client.model.Filters;
import com.mongodb.client.model.UpdateOptions;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.text.StringEscapeUtils;
import org.bson.Document;
import org.bson.conversions.Bson;
import org.neo4j.driver.Driver;
import org.neo4j.driver.internal.InternalNode;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.stereotype.Service;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * @author yueyyc
 * @version 1.0
 * @date 2022/10/24 10:42
 */
@Service
@Slf4j
public class TenantServiceImpl implements TenantService {

    private static final String queryAllTenants = "/api/iam/v2/tenant/tenants/by/customer";
    private static final String queryAllTenantUser = "/api/iam/v2/tenant/user/list";

    private static final String tokenParseUrl = "/api/iam/v2/identity/token/analyze";
    //iam给用户添加授权
    private static final String authUserUrl = "/api/iam/v2/tenant/auth/with/users";
    //iam给用户开启授权
    private static final String policyAddUrl = "/api/iam/v2/policy/batch/add";

    private static final String URL_GET_INTERNAL_TOKEN = "/api/iam/v2/identity/login/internal";

    //查询商品信息，中括号内为商品code
    private static final String GMC_MODULE_SEARCH = "/api/cloudgoods/{}/simple";

    //更新商品模组
    private static final String GMC_MODULE_UPDATE = "/api/cloudgoods/modules";

    @Value("${appToken}")
    private String appToken;

    @Value("${module.iam.domain}")
    private String iamUrl;

    @Value("${module.gmc.domain}")
    private String gmcUrl;

    @Autowired
    private Driver driver1;

    @Autowired(required = false)
    @Qualifier("domain2Driver")
    private Driver driver2;

    @Autowired
    private MongoTemplate mongoTemplate;


    @Override
    public JSONObject queryUserTenants(String tenantId) {
        return null;
    }

    @Override
    public TenantUserResult queryUserInTenant(String tenantId, String token, Integer pageSize, Integer pageNum) {
        String param = String.format("?pageSize=%d&pageNum=%d", pageSize, pageNum);
        String queryUserInTenantsUrl = queryAllTenantUser + param;
        JSONObject paramJson = new JSONObject();
        paramJson.put("tenantId", tenantId);
        String response = HttpUtil.createPost(iamUrl + queryUserInTenantsUrl).header("digi-middleware-auth-app", appToken).header("digi-middleware-auth-user", token).body(paramJson.toJSONString()).execute().body();
        try {
            JSONObject responseObj = JSON.parseObject(response);
            List<JSONObject> users = responseObj.getJSONArray("list").toJavaList(JSONObject.class);
            Integer total = responseObj.getInteger("total");
            return new TenantUserResult().setUserList(users).setTotal(total);
        } catch (Exception e) {
            JSONObject responseObj = JSON.parseObject(response);
            log.error("查询用户失败:{}", response);
            throw new BusinessException(responseObj.getString("errorMessage"));
        }
    }

    @Override
    public List<JSONObject> queryAllUserInTenant(String tenantId, String token) {
        List<JSONObject> allUsers = new ArrayList<>();
        Integer pageSize = 100;
        Integer pageNum = 1;
        String param = String.format("?pageSize=%d&pageNum=%d", pageSize, pageNum);
        String queryUserInTenantsUrl = queryAllTenantUser + param;
        JSONObject paramJson = new JSONObject();
        paramJson.put("tenantId", tenantId);
        String response = HttpUtil.createPost(iamUrl + queryUserInTenantsUrl).header("digi-middleware-auth-app", appToken).header("digi-middleware-auth-user", token).body(paramJson.toJSONString()).execute().body();
        log.info("queryAllUserInTenant response:{}", response);
        try {
            JSONObject responseJson = JSON.parseObject(response);
            String code = responseJson.getString("code");
            if (!StringUtils.isEmpty(code) && "500".equals(code)) {
                // 返回失败的话，仅返回自己的信息
                JSONObject tenantInfoByToken = getTenantInfoByToken(token);
                String id = tenantInfoByToken.getString("id");
                JSONArray list = new JSONArray();
                JSONObject supplementObject = new JSONObject();
                supplementObject.put("id", id);
                list.add(supplementObject);
                responseJson.put("list", list);
                responseJson.put("total", 1);
            }

            List<JSONObject> users = responseJson.getJSONArray("list").toJavaList(JSONObject.class);
            allUsers.addAll(users);
            Integer total = responseJson.getInteger("total");
            if (total > pageSize) {
                Integer size = total / pageSize;
                for (int i = 0; i < size; i++) {
                    pageNum++;
                    TenantUserResult tenantUserResult = queryUserInTenant(tenantId, token, pageSize, pageNum);
                    List<JSONObject> userJson = tenantUserResult.getUserList();
                    allUsers.addAll(userJson);
                }
            }
            return allUsers;
        } catch (Exception e) {
            JSONObject responseObj = JSON.parseObject(response);
            log.error("查询用户失败异常信息:", e);
            throw new BusinessException(responseObj.getString("errorMessage"));
        }
    }

    @Override
    public boolean currentUserExistInTenant(String userId, String tenantId, String token) {
        return false;
    }

    private void checkModule(JSONArray modules, List<String> ids, String id, String name) {
        if (!ids.contains(id)) {
            JSONObject module = new JSONObject();
            module.put("id",id);
            module.put("name",name);
            modules.add(module);
        }
    }

    private JSONObject getGmcModule(String gmcCode) {
        String url = gmcUrl + StrUtil.format(GMC_MODULE_SEARCH, gmcCode);
        Map<String, String> header = new HashMap<>();
        header.put("digi-middleware-auth-app", appToken);
        HttpResponse response = HttpUtil.createGet(url).addHeaders(header).execute();
        if ((response != null) && (200 == response.getStatus())) {
            String responseStr = response.body();
            JSONObject jsonObject = JSONObject.parseObject(responseStr);
            JSONObject modules = jsonObject.getJSONObject("modules");
            return modules;
        }
        return null;
    }

    private void updateGmcModule(JSONObject modules, String gmcCode, String token) {
        String url = gmcUrl + GMC_MODULE_UPDATE;
        Map<String, String> header = new HashMap<>();
        header.put("digi-middleware-auth-app", appToken);
        header.put("digi-middleware-auth-user", token);
        JSONObject param = new JSONObject();
        param.put("code",gmcCode);
        param.put("multiLanguageResource",new ArrayList<>());
        param.put("modules",modules);
        HttpResponse response = HttpUtil.createPost(url).addHeaders(header).body(param.toJSONString()).execute();
        if ((response != null) && (200 == response.getStatus())) {
            log.info("更新商品模组成功!");
        } else {
            log.error("更新商品模组失败:"+Optional.ofNullable(response).map(r->response.body()).orElse("body is null"));
            throw new BusinessException(Optional.ofNullable(response).map(r->response.body()).orElse("body is null"));
        }
    }

    private List<String> getUserIdList(List<JSONObject> userJson) {
        List<String> userIdList = new ArrayList<>();
        userJson.forEach(user -> userIdList.add(user.getString("id")));
        return userIdList;
    }

    private List<JSONObject> handleUserIdList(List<String> userIds) {
        List<JSONObject> userIdJson = new ArrayList<>();
        userIds.forEach(userId -> {
            JSONObject jsonObject = new JSONObject();
            jsonObject.put("id", userId);
            userIdJson.add(jsonObject);
        });
        return userIdJson;
    }



    @Override
    public void createTenants(List<String> tenantIds, String version) {
        for (String tenantId : tenantIds) {
            Map<String, Object> param = new HashMap<>();
            param.put("tenantId", tenantId);
            param.put("version", version);
            String findCql = "MATCH (n:TenantEntity) where n.tenantId = $tenantId RETURN properties(n) as tenant";
            List<Map<String, Object>> tenants = new Neo4jManager(driver1).ExecuteQuery(findCql, param);
            if (CollUtil.isEmpty(tenants)) {
                String cql = "create (n:TenantEntity{tenantId:$tenantId,tenantName:$tenantId,version:$version})";
                Neo4jMultipleUtil.executeCql(cql, param, driver1, driver2);
            }
            try {
                // 保存mongo数据
                Document document = new Document();
                document.put("tenantId", tenantId);
                document.put("tenantName", tenantId);
                document.put("version", version);
                Bson filter = Filters.and(Filters.eq("tenantId", tenantId));
                mongoTemplate.getMongoDatabaseFactory().getMongoDatabase(Constant.db_kg_sys).getCollection("tenantEntity").updateOne(
                        filter,
                        new Document("$set", document),
                        new UpdateOptions().upsert(true)
                );
            } catch (Exception e) {
                log.warn("娜娜发版新增mongoDB tenantEntity数据异常：", e);
            }
        }
    }

    @Override
    public void createOrUpdateTenants(List<String> tenantIds, String version) {
        for (String tenantId : tenantIds) {
            Map<String, Object> param = new HashMap<>();
            param.put("tenantId", tenantId);
            param.put("version", version);
            String findCql = "MATCH (n:TenantEntity) where n.tenantId = $tenantId RETURN properties(n) as tenant";
            List<Map<String, Object>> tenants = new Neo4jManager(driver1).ExecuteQuery(findCql, param);
            String cql = "";
            if (CollUtil.isEmpty(tenants)) {
                cql = "create (n:TenantEntity{tenantId:$tenantId,tenantName:$tenantId,version:$version})";
            } else {
                cql = "MATCH (n:TenantEntity) where n.tenantId = $tenantId set n.version = $version";
            }
            Neo4jMultipleUtil.executeCql(cql, param, driver1, driver2);
        }
    }

    @Override
    public void createTenantAndApplicationRelation(String application, String tenantVersion, String appVersion, List<String> tenantIds) {

        List<String> authAppTenantIdList = getAuthAppTenantIdList(application, tenantVersion, tenantIds);

        mergeRelation(application, tenantVersion, appVersion, authAppTenantIdList);
    }

    @Override
    public void createAllAuthTenantAndApplicationRelation(String application, String tenantVersion, String appVersion) {

        List<String> authAppTenantIdList = getAllAuthAppTenantIdList(application, tenantVersion);

        mergeRelation(application, tenantVersion, appVersion, authAppTenantIdList);
    }

    @Override
    public void createRelation(String application, String tenantVersion, String appVersion, List<String> tenantIds) {

        mergeRelation(application, tenantVersion, appVersion, tenantIds);
    }

    private void mergeRelation(String application, String tenantVersion, String appVersion, List<String> authAppTenantIdList) {
        List<Cql> executeCqlList = getMergeTenantAndAppDataRelationCql(application, tenantVersion, appVersion, authAppTenantIdList);
        Neo4jMultipleUtil.executeCqlTrans(executeCqlList, driver1, driver2);
    }

    // 获取多个租户和一个应用数据关联的cql语句，考虑流程引擎
    @Override
    public List<Cql> getMergeTenantAndAppDataRelationCql(String application, String tenantVersion, String appVersion, List<String> authAppTenantIdList) {
        List<Cql> executeCqlList = new ArrayList<>();
        Map<String, Object> param = new HashMap<>();
        param.put("nameSpace", application);
        param.put("tenantVersion", tenantVersion);
        param.put("appVersion", appVersion);
//        param.put("publishTime", DateUtil.now());
        param.put("tenantIdList", authAppTenantIdList);
        // 这种写法，不会在已有关系的基础上再新增数据
        String cql2 = "match(n:TenantEntity{version:$tenantVersion}) where n.tenantId in $tenantIdList match(m:Activity{version:$appVersion}) " +
                "where (m.nameSpace = $nameSpace or m.athena_namespace = $nameSpace) and " +
                "((m.inclusionTenant is null and m.notInclusionTenant is null) or " +
                "(m.inclusionTenant is not null and n.tenantId in m.inclusionTenant) or " +
                "(m.notInclusionTenant is not null and not n.tenantId in m.notInclusionTenant)) merge (n)-[:ACTIVITY]->(m)";
        executeCqlList.add(new Cql().setCql(cql2).setParams(param));
        String cql3 = "match(n:TenantEntity{version:$tenantVersion}) where n.tenantId in $tenantIdList match(m:MonitorRule{version:$appVersion}) " +
                "where (m.nameSpace = $nameSpace or m.athena_namespace = $nameSpace) and " +
                "((m.inclusionTenant is null and m.notInclusionTenant is null) or " +
                "(m.inclusionTenant is not null and n.tenantId in m.inclusionTenant) or " +
                "(m.notInclusionTenant is not null and not n.tenantId in m.notInclusionTenant)) merge (n)-[:USE]->(m)";
        executeCqlList.add(new Cql().setCql(cql3).setParams(param));
        String cql4 = "match(n:TenantEntity{version:$tenantVersion}) where n.tenantId in $tenantIdList match(m:Action{version:$appVersion}) " +
                "where (m.nameSpace = $nameSpace or m.athena_namespace = $nameSpace) and " +
                "((m.inclusionTenant is null and m.notInclusionTenant is null) or " +
                "(m.inclusionTenant is not null and n.tenantId in m.inclusionTenant) or " +
                "(m.notInclusionTenant is not null and not n.tenantId in m.notInclusionTenant)) merge (n)-[:ACTION]->(m)";
        executeCqlList.add(new Cql().setCql(cql4).setParams(param));
//        String cql5 = "match(n:TenantEntity{version:$tenantVersion}) where n.tenantId in $tenantIdList match(m:Mechanism{version:$appVersion,athena_namespace:$application}) merge (n)-[:MECHANISM]->(m)";
//        executeCqlList.add(new Cql().setCql(cql5).setParams(param));
        // 考虑流程引擎数据
        List<Cql> cqls = addTenantRelaWithProcess(Stream.of(application).collect(Collectors.toSet()), tenantVersion, appVersion, authAppTenantIdList);
        executeCqlList.addAll(cqls);
        return executeCqlList;
    }

    // 获取新增指定租户和多个应用code的指定版本之间关系的cql，考虑流程引擎
    private List<Cql> mergeSpecificTenantAndAppDataRelation(String tenantId, String tenantVersion, String appVersion, Set<String> appCodeList) {
        List<Cql> executeCqlList = new ArrayList<>();
        Map<String, Object> param = new HashMap<>();
        param.put("tenantId", tenantId);
        param.put("tenantVersion", tenantVersion);
        param.put("appVersion", appVersion);
        param.put("appCodeList", appCodeList);
        // 这种写法，不会在已有关系的基础上再新增数据
        String cql2 = "match(n:TenantEntity{version:$tenantVersion, tenantId:$tenantId}) match(m:Activity{version:$appVersion}) " +
                "where (m.athena_namespace in $appCodeList or m.nameSpace in $appCodeList) and " +
                "((m.inclusionTenant is null and m.notInclusionTenant is null) or " +
                "(m.inclusionTenant is not null and n.tenantId in m.inclusionTenant) or " +
                "(m.notInclusionTenant is not null and not n.tenantId in m.notInclusionTenant)) merge (n)-[:ACTIVITY]->(m)";
        executeCqlList.add(new Cql().setCql(cql2).setParams(param));
        String cql3 = "match(n:TenantEntity{version:$tenantVersion, tenantId:$tenantId}) match(m:MonitorRule{version:$appVersion}) " +
                "where (m.athena_namespace in $appCodeList or m.nameSpace in $appCodeList) and " +
                "((m.inclusionTenant is null and m.notInclusionTenant is null) or " +
                "(m.inclusionTenant is not null and n.tenantId in m.inclusionTenant) or " +
                "(m.notInclusionTenant is not null and not n.tenantId in m.notInclusionTenant)) merge (n)-[:USE]->(m)";
        executeCqlList.add(new Cql().setCql(cql3).setParams(param));
        String cql4 = "match(n:TenantEntity{version:$tenantVersion, tenantId:$tenantId}) match(m:Action{version:$appVersion}) " +
                "where (m.athena_namespace in $appCodeList or m.nameSpace in $appCodeList) and " +
                "((m.inclusionTenant is null and m.notInclusionTenant is null) or " +
                "(m.inclusionTenant is not null and n.tenantId in m.inclusionTenant) or " +
                "(m.notInclusionTenant is not null and not n.tenantId in m.notInclusionTenant)) merge (n)-[:ACTION]->(m)";
        executeCqlList.add(new Cql().setCql(cql4).setParams(param));
        // 考虑流程引擎数据
        List<Cql> cqls = addTenantRelaWithProcess(appCodeList, tenantVersion, appVersion, Collections.singletonList(tenantId));
        executeCqlList.addAll(cqls);
        return executeCqlList;
    }

    @Override
    public List<String> getAuthAppTenantIdList(String application, String version, List<String> excludeTenants) {
        Map<String, Object> param = new HashMap<>();
        param.put("application", application);
        param.put("excludeTenants", excludeTenants);
        param.put("version", version);
        String cql = "match(n:TenantEntity{version:$version}) -[]-> (m:AppEntity{code:$application}) where not n.tenantId in $excludeTenants return n.tenantId as tenantId";
        List<Map<String, Object>> authAppTenantList = new Neo4jManager(driver1).ExecuteQuery(cql, param);
        List<String> authAppTenantIdList = authAppTenantList.stream().map(authAppTenant -> String.valueOf(authAppTenant.get("tenantId"))).collect(Collectors.toList());
        return authAppTenantIdList;
    }


    private List<String> getAllAuthAppTenantIdList(String application, String version) {
        Map<String, Object> param = new HashMap<>();
        param.put("application", application);
        param.put("version", version);
        String cql = "match(n:TenantEntity{version:$version}) -[]-> (m:AppEntity{code:$application}) return n.tenantId as tenantId";
        List<Map<String, Object>> authAppTenantList = new Neo4jManager(driver1).ExecuteQuery(cql, param);
        List<String> authAppTenantIdList = authAppTenantList.stream().map(authAppTenant -> String.valueOf(authAppTenant.get("tenantId"))).collect(Collectors.toList());
        return authAppTenantIdList;
    }

    @Override
    public void createTenantAndCommonRelation(List<String> tenantIds, String version) {
        Map<String, Object> param = new HashMap<>();
        param.put("tenantIdList", tenantIds);
        param.put("version", version);
        param.put("publishTime", DateUtil.now());
        List<Cql> executeCqlList = new ArrayList<>();
        // 这个cql背后有个前提是，租户一旦和common做了关联  一定是全关联的，不会只关联一部分
//        String cql1 = "match(m:Task{version:$version}) where not (m)<-[:TASK]-(:TenantEntity) and m.nameSpace in ['common','espCommon'] match(n:TenantEntity) where not n.tenantId in $tenantIdList merge (n)-[:TASK]->(m)";
//        executeCqlList.add(new Cql().setCql(cql1).setParams(param));
        String cql2 = "match(m:Activity{version:$version}) where not (m)<-[:ACTIVITY]-(:TenantEntity) and m.nameSpace in ['common','espCommon'] match(n:TenantEntity) where not n.tenantId in $tenantIdList merge (n)-[:ACTIVITY]->(m)";
        executeCqlList.add(new Cql().setCql(cql2).setParams(param));
        String cql3 = "match(m:MonitorRule{version:$version}) where not (m)<-[:USE]-(:TenantEntity) and m.nameSpace in ['common','espCommon'] match(n:TenantEntity) where not n.tenantId in $tenantIdList merge (n)-[:USE]->(m)";
        executeCqlList.add(new Cql().setCql(cql3).setParams(param));
        String cql4 = "match(m:Action{version:$version}) WHERE NOT (m)<-[:ACTION]-(:TenantEntity) and m.nameSpace in ['common','espCommon'] match(n:TenantEntity) where not n.tenantId in $tenantIdList merge (n)-[:ACTION]->(m)";
        executeCqlList.add(new Cql().setCql(cql4).setParams(param));
//        String cql5 = "match(m:Mechanism{version:$version}) where not (m)<-[:MECHANISM]-(:TenantEntity) and m.nameSpace in ['common','espCommon'] match(n:TenantEntity) where not n.tenantId in $tenantIdList merge (n)-[:MECHANISM]->(m)";
//        executeCqlList.add(new Cql().setCql(cql5).setParams(param));
        Neo4jMultipleUtil.executeCqlTrans(executeCqlList, driver1, driver2);
    }

    @Override
    public List<Cql> createAllTenantAndCommonRelation(String version) {
        Map<String, Object> param = new HashMap<>();
        param.put("version", version);
        List<Cql> executeCqlList = new ArrayList<>();
        String cql2 = "match(m:Activity{version:$version}) match(n:TenantEntity{version:$version}) where m.nameSpace = 'common' merge (n)-[:ACTIVITY]->(m)";
        executeCqlList.add(new Cql().setCql(cql2).setParams(param));
        String cql3 = "match(m:MonitorRule{version:$version}) match(n:TenantEntity{version:$version}) where m.nameSpace = 'common' merge (n)-[:USE]->(m)";
        executeCqlList.add(new Cql().setCql(cql3).setParams(param));
        String cql4 = "match(m:Action{version:$version}) match(n:TenantEntity{version:$version}) where m.nameSpace = 'common' merge (n)-[:ACTION]->(m)";
        executeCqlList.add(new Cql().setCql(cql4).setParams(param));
        return executeCqlList;
    }

    @Override
    public List<Cql> getSpecificTenantAndCommonAndEspRelationCql(String tenantId, String version) {
        Map<String, Object> param = new HashMap<>();
        param.put("tenantId", tenantId);
        param.put("version", version);
        List<Cql> executeCqlList = new ArrayList<>();
        String cql2 = "match(m:Activity{version:$version}) match(n:TenantEntity{version:$version, tenantId:$tenantId}) where m.nameSpace in ['common','espCommon'] merge (n)-[:ACTIVITY]->(m)";
        executeCqlList.add(new Cql().setCql(cql2).setParams(param));
        String cql3 = "match(m:MonitorRule{version:$version}) match(n:TenantEntity{version:$version, tenantId:$tenantId}) where m.nameSpace in ['common','espCommon'] merge (n)-[:USE]->(m)";
        executeCqlList.add(new Cql().setCql(cql3).setParams(param));
        String cql4 = "match(m:Action{version:$version}) match(n:TenantEntity{version:$version, tenantId:$tenantId}) WHERE m.nameSpace in ['common','espCommon'] and (m.tenantId is null or m.tenantId = $tenantId) merge (n)-[:ACTION]->(m)";
        executeCqlList.add(new Cql().setCql(cql4).setParams(param));
        return executeCqlList;
    }

    @Override
    public Cql getSpecificTenantAndEspActionRelationCql(String tenantId, String version) {
        Map<String, Object> param = new HashMap<>();
        param.put("tenantId", tenantId);
        param.put("version", version);
        String addRelaCql1 = "match(m:Action{version:$version}) match(n:TenantEntity{version:$version, tenantId:$tenantId}) WHERE m.nameSpace = 'espCommon' and (m.tenantId is null or m.tenantId = $tenantId) merge (n)-[:ACTION]->(m)";
        return new Cql().setCql(addRelaCql1).setParams(param);
    }

    @Override
    public JSONObject getTenantInfoByToken(String token) {
        String response = analyzeIamToken(token);
        JSONObject responseObj = JSON.parseObject(response);
        String tenantId = responseObj.getString("tenantId");
        if (StringUtils.isEmpty(tenantId)) {
            log.error("解析token失败:", response);
            throw new BusinessException("租户获取失败：" + responseObj.getString("errorMessage"));
        }
        log.info("getTenantInfoByToken response is -->{}", responseObj);
        return responseObj;
    }


    @Override
    public String analyzeIamToken(String token) {
        String response = HttpUtil.createPost(iamUrl + tokenParseUrl).header("digi-middleware-auth-user", token).execute().body();
        return response;
    }


    @Override
    public Set<String> getAllTenants(String version) {
        String findCql = "MATCH (n:TenantEntity{version:$version}) RETURN n.tenantId as tenantId";
        Map<String, Object> param = new HashMap<>();
        param.put("version", version);
        List<Map<String, Object>> tenants = new Neo4jManager(driver1).ExecuteQuery(findCql, param);
        Set<String> tenantIdList = new HashSet<>();
        if (CollUtil.isNotEmpty(tenants)) {
            tenants.forEach(tenant -> tenantIdList.add(String.valueOf(tenant.get("tenantId"))));
        }
        return tenantIdList;
    }

    @Override
    public Map<String, String> getTenantVersion(List<String> tenantIdList) {
        String findCql = "MATCH (n:TenantEntity) WHERE n.tenantId in $tenantIdList RETURN n.tenantId as tenantId, n.version as version";
        Map<String, Object> param = new HashMap<>();
        param.put("tenantIdList", tenantIdList);
        List<Map<String, Object>> tenants = new Neo4jManager(driver1).ExecuteQuery(findCql, param);
        Map<String, String> re = new HashMap<>();
        if (CollUtil.isNotEmpty(tenants)) {
            tenants.forEach(tenant -> re.put((String) tenant.get("tenantId"), (String) tenant.get("version")));
        }
        return re;
    }

    @Override
    public Set<String> getAppEntityFullNamespace(Set<String> appCode) {
        String findCql = "MATCH (n:AppEntity) WHERE n.code in $appCode RETURN n.fullNamespace as fullNamespace";
        Map<String, Object> param = new HashMap<>();
        param.put("appCode", appCode);
        List<Map<String, Object>> appInfo = new Neo4jManager(driver1).ExecuteQuery(findCql, param);
        Set<String> re = new HashSet<>();
        if (CollUtil.isNotEmpty(appInfo)) {
            appInfo.forEach(info -> {
                if (!String.valueOf(info.get("fullNamespace")).equals("NULL") && !ObjectUtils.isEmpty(info.get("fullNamespace"))) {
                    re.add(String.valueOf(info.get("fullNamespace")));
                }
            });
        }
        return re;
    }

    @Override
    public List<Cql> mergeRelationBetweenTenantAndCommon(String application, List<String> tenantIdList, String targetVersion, Map<String, String> tenantVersionMap) {
        List<Cql> cqlList = new ArrayList<>();
        HashMap<String, Object> param = new HashMap<>();
        param.put("targetVersion", targetVersion);
        param.put("application", application);
        if (Constant.COMMON_CODE.equals(application)) {
            // 首先，发版/切版时需要所有1.0/2.0租户和1.0/2.0的common节点全关联
            List<Cql> allTenantAndCommonRelation = createAllTenantAndCommonRelation(targetVersion);
            cqlList.addAll(allTenantAndCommonRelation);
            // common应用
            tenantIdList.forEach(tenantId -> {
                if (StringUtils.isEmpty(tenantVersionMap.get(tenantId))) {
                    // 没有对应的version说明租户是在发版时新加的租户 需要租户和1.0的espCommon数据全关联
                    Cql cql = getSpecificTenantAndEspActionRelationCql(tenantId, targetVersion);
                    cqlList.add(cql);
                } else if (!targetVersion.equals(tenantVersionMap.get(tenantId))) {
                    param.put("oldVersion", tenantVersionMap.get(tenantId));
                    param.put("tenantId", tenantId);
                    // 租户需要更改版本, 需要断开租户和当前版本的common/espCommon关系， 绑定租户和targetVersion的espCommon关系
                    String deleteRelaCql = "match (tenant:TenantEntity{version:$targetVersion})-[relation]->(commonNode{version:$oldVersion}) where tenant.tenantId = $tenantId and commonNode.nameSpace in ['common','espCommon'] delete relation";
                    cqlList.add(new Cql().setCql(deleteRelaCql).setParams(param));
                    Cql cql = getSpecificTenantAndEspActionRelationCql(tenantId, targetVersion);
                    cqlList.add(cql);
                    // 租户变更版本后需要重新关联租户和除当前发布应用外的其他应用的数据关系
                    Set<String> appCodeList = getAppCodeRelaToTenantId(tenantId);
                    List<Cql> cql2 = mergeSpecificTenantAndAppDataRelation(tenantId, targetVersion, targetVersion, appCodeList);
                    cqlList.addAll(cql2);
                    // 增加切版租户相关应用的流程引擎的节点关系
                    List<Cql> cqlDelAndAdd = addTenantRelaWithProcess(appCodeList, targetVersion, targetVersion, Collections.singletonList(tenantId));
                    cqlList.addAll(cqlDelAndAdd);
                }
            });
        } else {
            // 非common应用
            tenantIdList.forEach(tenantId -> {
                if (StringUtils.isEmpty(tenantVersionMap.get(tenantId))) {
                    // 没有对应的version说明租户是在发版时新加的租户 需要租户和1.0的common/espCommon数据全关联
                    List<Cql> cql = getSpecificTenantAndCommonAndEspRelationCql(tenantId, targetVersion);
                    cqlList.addAll(cql);
                } else if (!targetVersion.equals(tenantVersionMap.get(tenantId))) {
                    // 租户需要更改版本 断开租户和oldVersion的common/espCommon关系，绑定租户和targetVersion的common/espCommon关系
                    param.put("oldVersion", tenantVersionMap.get(tenantId));
                    param.put("tenantId", tenantId);
                    String deleteRelaCql = "match (tenant:TenantEntity{version:$targetVersion})-[relation]->(commonNode{version:$oldVersion}) where tenant.tenantId = $tenantId and commonNode.nameSpace in ['common','espCommon'] delete relation";
                    cqlList.add(new Cql().setCql(deleteRelaCql).setParams(param));
                    cqlList.addAll(getSpecificTenantAndCommonAndEspRelationCql(tenantId, targetVersion));
                    // 租户相关的应用code
                    Set<String> appCodeList = getAppCodeRelaToTenantId(tenantId);
                    // 增加切版租户相关应用的流程引擎的节点关系
                    List<Cql> cqlDelAndAdd = addTenantRelaWithProcess(appCodeList, targetVersion, targetVersion, Collections.singletonList(tenantId));
                    cqlList.addAll(cqlDelAndAdd);
                    // 租户变更版本后需要重新关联租户和除当前发布应用外的其他应用的数据关系
                    appCodeList.remove(application);
                    List<Cql> cql2 = mergeSpecificTenantAndAppDataRelation(tenantId, targetVersion, targetVersion, appCodeList);
                    cqlList.addAll(cql2);
                }
            });
        }
        return cqlList;
    }

    // 切版时修改租户和应用数据不匹配的情况
    @Override
    public void modifyTenantRelaWithNotMatchAppDataInSwitch(String application, List<String> tenantIdList) {
        List<String> authAppTenantIdList;
        List<Cql> cqlList = new ArrayList<>();
        // 1.查询所有跟切版应用相关的1.0租户（正在切版的租户已经是2.0了，不在范围内）
        if (Constant.COMMON_CODE.equals(application)) {
            // common应用查找所有1.0的租户
            authAppTenantIdList = new ArrayList(getAllTenants(Constant.TEST_VERSION));
            // 租户断开和切版应用的2.0数据关系
            List<Cql> deleteCqls = deleteRelationBetweenAppAndTenant(application, Constant.TEST_VERSION, Constant.PROD_VERSION, authAppTenantIdList);
            cqlList.addAll(deleteCqls);
            // 和1.0应用数据重新绑定
            List<Cql> allTenantAndCommonRelation = createAllTenantAndCommonRelation(Constant.TEST_VERSION);
            cqlList.addAll(allTenantAndCommonRelation);
        } else {
            Map<String, Object> param = new HashMap<>();
            param.put("application", application);
            param.put("version", Constant.TEST_VERSION);
            String cql = "match(n:TenantEntity{version:$version}) -[]-> (m:AppEntity{code:$application}) return n.tenantId as tenantId";
            List<Map<String, Object>> authAppTenantList = new Neo4jManager(driver1).ExecuteQuery(cql, param);
            authAppTenantIdList = authAppTenantList.stream().map(authAppTenant -> String.valueOf(authAppTenant.get("tenantId"))).collect(Collectors.toList());
            // 租户断开和切版应用的2.0数据关系
            List<Cql> deleteCqls = deleteRelationBetweenAppAndTenant(application, Constant.TEST_VERSION, Constant.PROD_VERSION, authAppTenantIdList);
            cqlList.addAll(deleteCqls);
            // 和1.0应用数据重新绑定
            List<Cql> mergeCqls = getMergeTenantAndAppDataRelationCql(application, Constant.TEST_VERSION, Constant.TEST_VERSION, authAppTenantIdList);
            cqlList.addAll(mergeCqls);
        }
        // 2.断开正在切版的2.0租户和所有其他1.0数据的联系
        cqlList.add(deleteDiffVersionRelaBetweenTenantAndAppData(Constant.PROD_VERSION, Constant.TEST_VERSION, tenantIdList));
        Neo4jMultipleUtil.executeCqlTrans(cqlList, driver1, driver2);
    }

    // 租户断开和切版应用的2.0数据关系
    @Override
    public List<Cql> deleteRelationBetweenAppAndTenant(String application, String tenantVersion, String appVersion, List<String> authAppTenantIdList) {
        List<Cql> executeCqlList = new ArrayList<>();
        Map<String, Object> param = new HashMap<>();
        param.put("application", application);
        param.put("tenantVersion", tenantVersion);
        param.put("appVersion", appVersion);
        param.put("tenantIdList", authAppTenantIdList);
        String cql1 = "match (n:TenantEntity{version:$tenantVersion})-[r:ACTIVITY]->(m:Activity{version:$appVersion}) where n.tenantId in $tenantIdList and (m.athena_namespace=$application or m.nameSpace=$application) delete r";
        executeCqlList.add(new Cql().setCql(cql1).setParams(param));
        String cql2 = "match (n:TenantEntity{version:$tenantVersion})-[r:USE]->(m:MonitorRule{version:$appVersion}) where n.tenantId in $tenantIdList and (m.athena_namespace=$application or m.nameSpace=$application) delete r";
        executeCqlList.add(new Cql().setCql(cql2).setParams(param));
        String cql3 = "match (n:TenantEntity{version:$tenantVersion})-[r:ACTION]->(m:Action{version:$appVersion}) where n.tenantId in $tenantIdList and (m.athena_namespace=$application or m.nameSpace=$application) delete r";
        executeCqlList.add(new Cql().setCql(cql3).setParams(param));
        String cql4 = "match (n:TenantEntity{version:$tenantVersion})-[r:TASK]->(m:TASK{version:$appVersion}) where n.tenantId in $tenantIdList and (m.athena_namespace=$application or m.nameSpace=$application) delete r";
        executeCqlList.add(new Cql().setCql(cql4).setParams(param));
        return executeCqlList;
    }

    @Override
    public void modifyTenantRelaWithNotMatchAppDataInPublish(String tenantVersion, String appVersion, List<String> tenantIdList) {
        List<Cql> cqlList = new ArrayList<>();
        // 断开正在发版的1.0租户和所有其他2.0数据的联系
        cqlList.add(deleteDiffVersionRelaBetweenTenantAndAppData(tenantVersion, appVersion, tenantIdList));
        Neo4jMultipleUtil.executeCqlTrans(cqlList, driver1, driver2);
    }

    @Override
    public void modifyTenantConfig(String version, List<String> tenantIdList) {
        String tenantConfigVersion = version.split("\\.")[0] + ".x";
        for (String tenantId : tenantIdList) {
            Bson filterFind = Filters.eq("tenantId", tenantId);
            FindIterable<Document> documents = mongoTemplate.getMongoDatabaseFactory().getMongoDatabase("datamap").getCollection("tenantConfig").find(filterFind);
            Document document = documents.first();
            if (document == null) {
                // 新增数据
                Document newDocument = new Document();
                newDocument.put("version", tenantConfigVersion);
                newDocument.put("tenantId", tenantId);
                mongoTemplate.getMongoDatabaseFactory().getMongoDatabase("datamap").getCollection("tenantConfig").insertOne(newDocument);
            } else {
                boolean needUpdate = false;
                for (Document doc : documents) {
                    if (!tenantConfigVersion.equals(doc.get("version"))) {
                        needUpdate = true;
                        break;
                    }
                }
                if (needUpdate) {
                    Bson update = new Document("$set", new Document().append("version", tenantConfigVersion));
                    mongoTemplate.getMongoDatabaseFactory().getMongoDatabase("datamap").getCollection("tenantConfig").updateMany(filterFind, update);
                }
            }
        }
    }

    @Override
    public String grantUserAuth(TenantUser tenantUser, String application, String strategyCode, String token) {
        String tenantId = tenantUser.getTenantId();
        JSONObject paramJson = new JSONObject();
        paramJson.put("tenantId", tenantId);
        paramJson.put("userIds", tenantUser.getUserIdList());
        paramJson.put("appId", application);
        paramJson.put("strategyCode", strategyCode);

        String url = iamUrl + authUserUrl;

        HttpResponse response = HttpUtil.createPost(url).header("digi-middleware-auth-app", appToken).header("digi-middleware-auth-user", token).header("digi-middleware-auth-app", appToken).body(paramJson.toJSONString()).execute();

        log.info("调用iam接口,赋予用户应用权限#post,url:{},response:{}", url, JSON.toJSONString(response));

        if (response != null) {
            if (200 == response.getStatus()) {
                return "SUCCESS";
            } else {
                return response.body();
            }
        } else {
            return "";
        }
    }

    @Override
    public String getRealTimeTenantToken(String tenantId) {
        JSONObject param = new JSONObject();
        param.put("tenantId", tenantId);
        param.put("userId", "integration");
        param.put("passwordHash", "6826CC688C4AF1BD0A8DDA2DBDF8897B");

        String body = JSON.toJSONString(param);
        String url = iamUrl + URL_GET_INTERNAL_TOKEN;

        HttpResponse response = HttpUtil.createPost(url).header("digi-middleware-auth-app", appToken).header("Client-Agent", "mobileplatform-2.0.1.1")
                .header("Content-Type", "application/json").body(body).execute();

        log.info("调用iam接口,获取租户token#post,url:{},response:{}", url, JSON.toJSONString(response));

        String result = response.body();
        JSONObject resultInfo = JSON.parseObject(result);
        String tenantToken = resultInfo.getString("token");
        if (org.apache.commons.lang3.StringUtils.isEmpty(tenantToken)) {
            log.error("调用iam接口,获取租户token异常");
            throw new BusinessException(String.format("获取IAM集成账号租户token:%s接口异常", "getTenantToken"));
        }
        return tenantToken;
    }


    @Override
    public void updateTenantVersion(List<String> tenantIdList, String application, String version) {
        log.info("更新租户版本：租户:{},版本:{} ...", tenantIdList, version);
        List<Cql> cqlList = new ArrayList<>();
        // 更新TenantEntity版本
        cqlList.add(CqlMapper.updateTenantVersion(tenantIdList, version));
        // 更新TenantConfig版本，没有就新增
        modifyTenantConfig(version, tenantIdList);
        // 新增租户和应用AppEntity的关系
        cqlList.add(CqlMapper.mergeTenantAppEntityRelation(tenantIdList, application));
        // 非公共类型应用-merge租户与实体之间的relation TODO 娜娜暂时无该逻辑，后续若添加侦测、行动等功能时需再放开
//        List<Cql> executeCqlList = getMoreTenantAndOneAppDataRelationCql(application, version, version, tenantIdList);
//        cqlList.addAll(executeCqlList);
        log.info("更新租户--开始执行cql：" + JSON.toJSONString(cqlList));
        new Neo4jManager(driver1).ExecuteTransactionNoQuery(cqlList);
        if (driver2 != null) {
            new Neo4jManager(driver2).ExecuteTransactionNoQuery(cqlList);
        }
        try {
            // 更新租户版本
            Bson filterBson = Filters.and(Filters.in("tenantId", tenantIdList));
            Bson updateBson = new Document("$set", new Document().append("version", version));
            mongoTemplate.getMongoDatabaseFactory().getMongoDatabase(Constant.db_kg_sys).getCollection("tenantEntity").updateMany(filterBson, updateBson);
            // 增加租户和应用关系
            for (String tenantId : tenantIdList) {
                Document document = new Document();
                document.put("tenantId", tenantId);
                document.put("appCode", application);
                Bson filter = Filters.and(Filters.eq("tenantId", tenantId), Filters.eq("appCode", application));
                mongoTemplate.getMongoDatabaseFactory().getMongoDatabase(Constant.db_kg_sys).getCollection("tenantAppRelation").updateOne(
                        filter,
                        new Document("$set", document),
                        new UpdateOptions().upsert(true)
                );
            }
        } catch (Exception e) {
            log.warn("娜娜发版修改mongoDB数据异常：", e);
        }
        log.info("更新租户版本完成：租户:{},版本:{} ...", tenantIdList, version);
    }

    @Override
    public Set<String> getNotCommonAppCodeRelaToTenantId(String tenantId) {
        Map<String, Object> param = new HashMap<>();
        param.put("tenantId", tenantId);
        String cql = "match(n:TenantEntity{tenantId:$tenantId})-[]->(m:AppEntity) where m.commonApp is null or m.commonApp<>true return m.code as code";
        List<Map<String, Object>> authAppCodeList = new Neo4jManager(driver1).ExecuteQuery(cql, param);
        Set<String> appCode = authAppCodeList.stream().map(authAppCode -> String.valueOf(authAppCode.get("code"))).collect(Collectors.toSet());
        return appCode;
    }

    @Override
    public void createOneTenantAndMoreAppRelation(Set<String> appCodeList, String tenantVersion, String appVersion, String tenantId) {
        List<Cql> executeCqlList = mergeSpecificTenantAndAppDataRelation(tenantId, tenantVersion, appVersion, appCodeList);
        Neo4jMultipleUtil.executeCqlTrans(executeCqlList, driver1, driver2);
    }

    @Override
    public void updateAppDataVersion(String application, String oldVersion, String newVersion, List<String> tenantIdList) {
        List<Cql> executeCqlList = new ArrayList<>();
        List<Cql> updateAppDataVersionCqlList = getUpdateAppDataVersion(application, oldVersion, newVersion, tenantIdList);
        executeCqlList.addAll(updateAppDataVersionCqlList);
        Neo4jMultipleUtil.executeCqlTrans(updateAppDataVersionCqlList, driver1, driver2);
        // 更新TenantConfig版本，没有就新增
        modifyTenantConfig(newVersion, tenantIdList);
        try {
            // 修改mongo数据版本号
            Bson filterBson = Filters.and(Filters.in("tenantId", tenantIdList), Filters.ne("version", newVersion));
            Bson updateBson = new Document("$set", new Document().append("version", newVersion));
            mongoTemplate.getMongoDatabaseFactory().getMongoDatabase(Constant.db_kg_sys).getCollection("tenantEntity").updateMany(filterBson, updateBson);
        } catch (Exception e) {
            log.warn("修改mongoDB tenantEntity数据异常：", e);
        }
    }

    @Override
    public void copyAppVersionData(String application, String oldVersion, String newVersion) {
        long startTime = System.currentTimeMillis();
        log.info(application + "应用切版-开始复制{}版本数据至{}版本......", oldVersion, newVersion);
        // common应用当作正常应用复制数据，和普通应用的切版拆开
        Neo4jManager neo4jManager1 = new Neo4jManager(driver1);
        copyNeo4jData(application, oldVersion, newVersion, neo4jManager1);
        if (driver2 != null) {
            Neo4jManager neo4jManager2 = new Neo4jManager(driver2);
            copyNeo4jData(application, oldVersion, newVersion, neo4jManager2);
        }
        long neoT = System.currentTimeMillis();
        log.info(application + "应用切版-进入复制应用到复制完neo4j数据的耗时：" + (neoT - startTime));
        Update update = new Update();
        update.set("copied", true);
        mongoTemplate.upsert(new Query(), update, "application");
        long mongoT = System.currentTimeMillis();
        log.info(application + "应用切版-复制完mongo数据的耗时：" + (mongoT - neoT));
        log.info(application + "应用切版-复制完成");
    }

    /**
     * 复制neo4j数据
     *
     * @param application
     * @param oldVersion
     * @param newVersion
     * @param neo4jManager
     */
    private void copyNeo4jData(String application, String oldVersion, String newVersion, Neo4jManager neo4jManager) {
        long t0 = System.currentTimeMillis();
        // 作为查询
        ConcurrentHashMap<Long, Long> nodeNeo4jIdMap = new ConcurrentHashMap<>();
        // 收集所有的node节点
        HashMap<String, Object> param = new HashMap<>();
        param.put("version", oldVersion);
        param.put("application", application);
        List<Map<String, Object>> nodeResultList = neo4jManager.ExecuteQuery("match (node) where node.version = $version and (node.athena_namespace = $application or node.nameSpace = $application) and not any(label in labels(node) WHERE label in ['TenantEntity','AppEntity']) return node", param);
        // 收集所有的relation以及前后nodeId(排除租户对外的relation)
        List<Map<String, Object>> relationList = neo4jManager.ExecuteQuery("match (startNode)-[relation]->(endNode) where startNode.version = $version and (startNode.athena_namespace = $application or startNode.nameSpace = $application) and endNode.version = $version and (endNode.athena_namespace = $application or endNode.nameSpace = $application) return id(startNode) as startNodeId,type(relation) as relationType,id(endNode) as endNodeId", param);
        // 遍历nodeResultList组装新的节点属性，添加oldNodeId属性
        List<Cql> cqlList = new ArrayList<>();
        for (Map<String, Object> nodeResult : nodeResultList) {
            StringBuilder nodeCypher = new StringBuilder("create (node");
            Collection<String> labels = ((InternalNode) nodeResult.get("node")).labels();
            long nodeId = ((InternalNode) nodeResult.get("node")).id();
            for (String label : labels) {
                nodeCypher.append(String.format(":%s", label));
            }
            nodeCypher.append("{");
            Map<String, Object> properties = ((InternalNode) nodeResult.get("node")).asMap();
            // 取到属性组装cql
            combineNodeProperties(properties, nodeCypher);
            nodeCypher.append("oldNodeId:" + nodeId + ",");
            nodeCypher.append(String.format("version:'%s', publishTime:'%s'}) return id(node) as nodeId", newVersion, DateUtil.now()));
            cqlList.add(new Cql().setCql(StringEscapeUtils.escapeJava(nodeCypher.toString())));
        }
        // 执行所有复制操作
        neo4jManager.ExecuteTransactionNoQuery(cqlList);

        param.put("version", newVersion);
        param.put("application", application);
        List<Map<String, Object>> newNodeList = neo4jManager.ExecuteQuery("match (node) where node.version = $version and (node.athena_namespace = $application or node.nameSpace = $application) and not any(label in labels(node) WHERE label in ['TenantEntity','AppEntity']) return node", param);
        for (Map<String, Object> nodeResult : newNodeList) {
            long newNodeId = ((InternalNode) nodeResult.get("node")).id();
            Map<String, Object> properties = ((InternalNode) nodeResult.get("node")).asMap();
            Long oldNodeId = Convert.toLong(properties.get("oldNodeId"));
            nodeNeo4jIdMap.put(oldNodeId,newNodeId);
        }

        relationList.forEach(relation -> {
            relation.put("startNodeId", nodeNeo4jIdMap.get(Long.valueOf(relation.get("startNodeId").toString())));
            relation.put("endNodeId", nodeNeo4jIdMap.get(Long.valueOf(relation.get("endNodeId").toString())));
        });

        List<Cql> cqlList1 = new ArrayList<>();
        for (Map<String, Object> relation : relationList) {
            String relationCypher = String.format("match (startNode),(endNode) WHERE id(startNode)=%s and id(endNode)=%s merge (startNode)-[relation:%s]->(endNode)", relation.get("startNodeId"), relation.get("endNodeId"), relation.get("relationType"));
            cqlList1.add(new Cql().setParams(new HashMap<>()).setCql(relationCypher));
        }
        neo4jManager.ExecuteTransactionNoQuery(cqlList1);
        long t1 = System.currentTimeMillis();
        log.info(application + "应用复制neo4j数据耗时（ms）：" + (t1 - t0));
    }

    private void combineNodeProperties(Map<String, Object> properties, StringBuilder nodeCypher) {
        properties.forEach((k, v) -> {
            if (!"version".equals(k) && !"oldNodeId".equals(k)) {
                if (k.contains(".")) {
                    nodeCypher.append(String.format("`%s`:", k));
                } else {
                    nodeCypher.append(String.format("%s:", k));
                }
                if (v instanceof String) {
                    String propertyValue = (String) v;
                    propertyValue = propertyValue.replace("'", "\\\"");
                    nodeCypher.append(String.format("'%s',", propertyValue));
                } else if (v instanceof Collection) {
                    nodeCypher.append("[");
                    List propertyValueList = (List) v;
                    propertyValueList.forEach(propertyValue -> {
                        if (propertyValue instanceof String) {
                            nodeCypher.append(String.format("'%s'", propertyValue)).append(",");
                        } else {
                            nodeCypher.append(propertyValue).append(",");
                        }
                    });
                    if (!propertyValueList.isEmpty()) {
                        nodeCypher.deleteCharAt(nodeCypher.length() - 1);
                    }
                    nodeCypher.append("],");
                } else {
                    nodeCypher.append(v).append(",");
                }
            }
        });
    }

    public List<Cql> getUpdateAppDataVersion(String application, String oldVersion, String newVersion, List<String> tenantIdList) {
        //更新neo4j数据version,先删除新版本数据，再将旧版本数据的version更新成新版本
        List<Cql> cqlList = new ArrayList<>();

        HashMap<String, Object> param = new HashMap<>();
        param.put("oldVersion", oldVersion);
        param.put("newVersion", newVersion);
        param.put("application", application);

        Cql deleteCql = new Cql();
        deleteCql.setCql("match (node) where node.version = $newVersion and node.athena_namespace = $application detach delete node");
        deleteCql.setParams(param);

        Cql updateCql = new Cql();
        updateCql.setCql("match (node) where node.version = $oldVersion and node.athena_namespace = $application set node.version=$newVersion");
        updateCql.setParams(param);

        Cql updateTenantCql = new Cql();
        param.put("tenantIdList", tenantIdList);
        updateTenantCql.setCql("match (n:TenantEntity) where n.tenantId in $tenantIdList and n.version<>$newVersion set n.version=$newVersion");
        updateTenantCql.setParams(param);

        cqlList.add(deleteCql);
        cqlList.add(updateCql);
        cqlList.add(updateTenantCql);

        Cql updateAppCql = new Cql();
        updateAppCql.setCql("match (n:AppEntity) where n.code = $application and n.version<>$newVersion set n.version=$newVersion");
        updateAppCql.setParams(param);
        cqlList.add(updateAppCql);

        return cqlList;
    }

    // 断开正在切版的2.0租户和所有其他1.0数据的联系
    private Cql deleteDiffVersionRelaBetweenTenantAndAppData(String tenantVersion, String appVersion, List<String> tenantIdList) {
        Map<String, Object> param = new HashMap<>();
        param.put("tenantVersion", tenantVersion);
        param.put("appVersion", appVersion);
        param.put("tenantIdList", tenantIdList);
        String cqlStr = "match (n:TenantEntity{version:$tenantVersion})-[r]->(m) where n.tenantId in $tenantIdList and m.version=$appVersion and NOT m:AppEntity delete r";
        Cql cql = new Cql().setCql(cqlStr).setParams(param);
        return cql;
    }

    // 获取与租户有关系的appEntity的code
    private Set<String> getAppCodeRelaToTenantId(String tenantId) {
        Map<String, Object> param = new HashMap<>();
        param.put("tenantId", tenantId);
        String cql = "match(n:TenantEntity{tenantId:$tenantId})-[]->(m:AppEntity) return m.code as code";
        List<Map<String, Object>> authAppCodeList = new Neo4jManager(driver1).ExecuteQuery(cql, param);
        Set<String> appCode = authAppCodeList.stream().map(authAppCode -> String.valueOf(authAppCode.get("code"))).collect(Collectors.toSet());
        return appCode;
    }


    /**
     * 流程引擎中可能会用到的节点有Task/Activity/MonitorRule/Action，发版或者切版过程中需要将租户和发版应用
     * （流程引擎的节点nameSpace是应用code的全称）中的Task节点关联起来
     *
     * @param application   应用
     * @param tenantVersion 租户版本
     * @param appVersion    应用数据版本
     * @param tenantIdList  租户list
     * @return
     */
    private List<Cql> addTenantRelaWithProcess(Set<String> application, String tenantVersion, String appVersion, List<String> tenantIdList) {
        List<Cql> executeCqlList = new ArrayList<>();
        // 如果租户版本和应用数据版本不一致，暂不考虑关联流程引擎数据，因为流程引擎的节点不会因为切版删除，可能会导致2.0的租户关联1.0的流程引擎节点
        if (!tenantVersion.equals(appVersion)) {
            return executeCqlList;
        }
        // 如果应用对应AppEntity不存在fullNameSpace就直接返回
        Set<String> fullNameSpace = getAppEntityFullNamespace(application);
        if (CollUtil.isEmpty(fullNameSpace)) {
            return executeCqlList;
        }

        Map<String, Object> param = new HashMap<>();
        param.put("application", application);
        param.put("tenantVersion", tenantVersion);
        param.put("appVersion", appVersion);
        param.put("tenantIdList", tenantIdList);
        param.put("fullNameSpace", fullNameSpace);

        executeCqlList = addTenantRelaWithNodeInProcess(param);

        return executeCqlList;
    }

    private List<Cql> addTenantRelaWithNodeInProcess(Map<String, Object> param) {
        List<Cql> executeCqlList = new ArrayList<>();
        // Task
        String cql1 = "match(n:TenantEntity{version:$tenantVersion}) where n.tenantId in $tenantIdList match(m:Task{version:$appVersion}) " +
                "where (m.nameSpace in $fullNameSpace or m.athena_namespace in $fullNameSpace) and " +
                "((m.inclusionTenant is null and m.notInclusionTenant is null) or " +
                "(m.inclusionTenant is not null and n.tenantId in m.inclusionTenant) or " +
                "(m.notInclusionTenant is not null and not n.tenantId in m.notInclusionTenant)) merge (n)-[:TASK]->(m)";
        executeCqlList.add(new Cql().setCql(cql1).setParams(param));
        // Activity
        String cql2 = "match(n:TenantEntity{version:$tenantVersion}) where n.tenantId in $tenantIdList match(m:Activity{version:$appVersion}) " +
                "where (m.nameSpace in $fullNameSpace or m.athena_namespace in $fullNameSpace) and " +
                "((m.inclusionTenant is null and m.notInclusionTenant is null) or " +
                "(m.inclusionTenant is not null and n.tenantId in m.inclusionTenant) or " +
                "(m.notInclusionTenant is not null and not n.tenantId in m.notInclusionTenant)) merge (n)-[:ACTIVITY]->(m)";
        executeCqlList.add(new Cql().setCql(cql2).setParams(param));
        // MonitorRule
        String cql3 = "match(n:TenantEntity{version:$tenantVersion}) where n.tenantId in $tenantIdList match(m:MonitorRule{version:$appVersion}) " +
                "where (m.nameSpace in $fullNameSpace or m.athena_namespace in $fullNameSpace) and " +
                "((m.inclusionTenant is null and m.notInclusionTenant is null) or " +
                "(m.inclusionTenant is not null and n.tenantId in m.inclusionTenant) or " +
                "(m.notInclusionTenant is not null and not n.tenantId in m.notInclusionTenant)) merge (n)-[:USE]->(m)";
        executeCqlList.add(new Cql().setCql(cql3).setParams(param));
        // Action
        String cql4 = "match(n:TenantEntity{version:$tenantVersion}) where n.tenantId in $tenantIdList match(m:Action{version:$appVersion}) " +
                "where (m.nameSpace in $fullNameSpace or m.athena_namespace in $fullNameSpace) and " +
                "((m.inclusionTenant is null and m.notInclusionTenant is null) or " +
                "(m.inclusionTenant is not null and n.tenantId in m.inclusionTenant) or " +
                "(m.notInclusionTenant is not null and not n.tenantId in m.notInclusionTenant)) merge (n)-[:ACTION]->(m)";
        executeCqlList.add(new Cql().setCql(cql4).setParams(param));
        return executeCqlList;
    }

}
