package com.digiwin.athena.semc.common;

import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

/**
 * 分页返回包装类
 *
 * @author: sungqz
 * @date: 2023-08-16
 */

@Data
@AllArgsConstructor
@NoArgsConstructor
public class PageInfoResp<T> implements Serializable {
    private static final long serialVersionUID = 2405172041950251807L;

    /**
     * 每页记录数
     */
    private int pageSize;

    /**
     * 当前页
     */
    private int pageNo;

    /**
     * 总页数
     */
    private int totalPages;

    /**
     * 总记录数
     */
    private int totalRecords;

    /**
     * 数据列表
     */
    private List<T> list;

    public PageInfoResp(int pageNo, int pageSize, int totalRecords, List<T> list) {
        this.pageNo = pageNo;
        this.pageSize = pageSize;
        this.totalRecords = totalRecords;
        this.list = list;
        this.totalPages = ((int) Math.ceil(totalRecords * 1.0 / pageSize));
    }

    public int getPageSize() {
        return pageSize;
    }

    /**
     * 设置每页大小
     *
     * @param pageSize
     */
    public void setPageSize(int pageSize) {
        if (pageSize > 0) {
            this.pageSize = pageSize;
        } else {
            this.pageSize = 10;
        }
    }

    public int getPageNo() {
        return pageNo;
    }

    /**
     * 设置当前页
     *
     * @param pageNo
     */
    public void setPageNo(int pageNo) {
        if (pageNo <= 1) {
            this.pageNo = 1;
        } else if (pageNo > this.totalPages) {
            this.pageNo = this.totalPages;
        } else {
            this.pageNo = pageNo;
        }
    }

    public int getTotalPages() {
        return totalPages;
    }

    /**
     * 设置总页数
     */
    public void setTotalPages() {
        this.totalPages = this.totalRecords % this.pageSize == 0 ? this.totalRecords / this.pageSize
                : this.totalRecords / this.pageSize + 1;
    }

    public int getTotalRecords() {
        return totalRecords;
    }

    /**
     * 设置总记录数
     *
     * @param totalRecords
     */
    public void setTotalRecords(int totalRecords) {
        if (totalRecords >= 0) {
            this.totalRecords = totalRecords;
        } else {
            this.totalRecords = 0;
        }
    }

    public List<T> getList() {
        return list;
    }

    public void setList(List<T> list) {
        this.list = list;
    }

    /**
     * 当前页记录列表
     *
     * @param pageNo
     * @param pageSize
     * @param list
     */
    public void queryPager(int pageNo, int pageSize, List<T> list) {
        // 设置总记录数
        if (list != null && list.size() != 0) {
            this.setTotalRecords(list.size());
        } else {
            this.setTotalRecords(0);
        }
        // 设置每页记录数
        this.setPageSize(pageSize);
        // 设置总页数
        this.setTotalPages();
        // 设置当前页
        this.setPageNo(pageNo);
        // 设置当前页记录数
        if (list != null && list.size() != 0) {
            if (pageNo == this.totalPages) {
                this.list = list.subList((pageNo - 1) * pageSize, this.totalRecords);
            } else {
                this.list = list.subList((pageNo - 1) * pageSize, pageNo * pageSize);
            }
        } else {
            this.list = new ArrayList<>();
        }
    }

    /**
     * 是否有前一页
     *
     * @return
     */
    public boolean isPrePage() {
        if (this.pageNo > 1) {
            return true;
        }
        return false;
    }

    /**
     * 是否有下一页
     *
     * @return
     */
    public boolean isNextPage() {
        if (this.pageNo < this.totalPages) {
            return true;
        }
        return false;
    }

    /**
     * 前一页页码
     *
     * @return
     */
    public int getPrePageNo() {
        if (isPrePage()) {
            return this.pageNo - 1;
        }
        return this.pageNo;
    }

    /**
     * 下一页页码
     *
     * @return
     */
    public int getNextPageNo() {
        if (isNextPage()) {
            return this.pageNo + 1;
        }
        return this.pageNo;
    }
}