package com.digiwin.athena.semc.configuration;

import com.digiwin.athena.appcore.constant.ErrorTypeEnum;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.ResponseEntityWrapper;
import com.digiwin.athena.semc.common.BizException;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.ServiceException;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.validation.ObjectError;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;

import javax.servlet.http.HttpServletRequest;


@ControllerAdvice(annotations = {RestController.class, Controller.class})
public class RestControllerExceptionHandler {

    private static Logger logger = LoggerFactory.getLogger(RestControllerExceptionHandler.class);



    @ExceptionHandler(Exception.class)
    @ResponseBody
    ResponseEntity<?> handleControllerException(HttpServletRequest request, Throwable ex) {
        String requestUrl = request.getRequestURL().toString();
        if (ex instanceof ServiceException) {
            ServiceException le = (ServiceException) ex;
            logger.warn("全局ServiceException异常：{}", requestUrl, ex);
            return ResponseEntityWrapper.wrapperFail(le.getCode(), le.getMessage());
        }
        if (ex instanceof BizException) {
            BizException le = (BizException) ex;
            logger.warn("全局BizException异常：{}", requestUrl, ex);
            return ResponseEntityWrapper.wrapperFail(le.getCode(), le.getMessage());
        }
        if (ex instanceof MethodArgumentNotValidException) {
            MethodArgumentNotValidException e = (MethodArgumentNotValidException) ex;
            List<ObjectError> errorList = e.getBindingResult().getAllErrors();
            StringBuffer errorMsg = new StringBuffer();
            errorList.forEach(x -> errorMsg.append(x.getDefaultMessage()).append(";"));
            String substring = StringUtils.isNotBlank(errorMsg) ? errorMsg.substring(0, errorMsg.length() - 1) : "";
            logger.warn("全局参数校验异常：{}，{}", requestUrl, substring, e);
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR,substring);
        }
        logger.error("全局Exception异常：{}", requestUrl, ex);
        BaseResultDTO result = new BaseResultDTO();
        result.setErrorType(ErrorTypeEnum.BUSINESS.getValue());
        result.setErrorCode(String.valueOf(ErrorCodeConstant.SYSTEM_ERROR));
        result.setErrorMessage("系统异常");
        result.setStatus(HttpStatus.INTERNAL_SERVER_ERROR.value());
        result.setStatusDescription(HttpStatus.INTERNAL_SERVER_ERROR.getReasonPhrase());
        result.setPath(requestUrl);
        result.setServerTime(System.currentTimeMillis());
        result.setDebugInfo(ex.getMessage());
        return ResponseEntity.ok(result);

    }


    private HttpStatus getStatus(HttpServletRequest request) {
        Integer statusCode = (Integer) request.getAttribute("javax.servlet.error.status_code");
        if (statusCode == null) {
            return HttpStatus.INTERNAL_SERVER_ERROR;
        }
        return HttpStatus.valueOf(statusCode);
    }


}
