package com.digiwin.athena.semc.controller;

import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.I18NKey;
import com.digiwin.athena.semc.dto.config.TenantAttributeDto;
import com.digiwin.athena.semc.service.tenant.TenantAttributeService;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;
import io.swagger.v3.oas.annotations.Operation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.*;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;

/**
 * @author guijf
 * create: 2024-03-11 17:18
 * Description: 租户属性配置控制器
 */
@Slf4j
@RestController
@RequestMapping("/semc/tenant/attribute")
public class TenantAttributeController{

    @Resource
    private TenantAttributeService tenantAttributeService;

    @Resource
    private MessageUtils messageUtils;

       /**
     * 获取租户属性值接口
     * 此方法专注于处理租户级别的属性查询，确保只有授权的应用可以访问特定的租户属性
     *
     * @param attributeCode 属性代码，用于标识特定的租户属性，是查询的关键参数
     * @return 返回一个响应实体，包含查询结果或错误信息
     */
    @Operation(method = "获取租户属性值接口",description = "获取租户属性值接口")
    @GetMapping("")
    public ResponseEntity<BaseResultDTO<Object>> queryAttributeValue(@RequestParam(value = "attributeCode") String attributeCode) {
        try {
            // 尝试根据属性代码查询租户属性，并通过ResponseEntityWrapperUtil封装查询结果
            return ResponseEntityWrapperUtil.wrapperOk(tenantAttributeService.queryTenantAttributeByCode(attributeCode));
        } catch (Exception e) {
            // 当查询过程中发生异常时，记录错误日志，并返回系统错误的响应实体
            log.error("TenantAttributeController.queryAttributeValue is error, attributeCode:{}", attributeCode, e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 更新租户属性值
     *
     * 该接口用于更新特定租户的属性值它接受一个 TenantAttributeDto 对象作为请求体，
     * 该对象中包含了需要更新的属性信息以及新的属性值
     *
     * @param attribute 包含了租户属性信息和新值的 TenantAttributeDto 对象
     * @return 返回一个 ResponseEntity 对象，其中包含了更新操作的结果信息
     */
    @Operation(method = "更新租户属性值",description = "更新租户属性值")
    @PostMapping("/update")
    public ResponseEntity<BaseResultDTO<Object>> updateAttributeValue(@RequestBody TenantAttributeDto<Object> attribute) {
        try {
            // 尝试根据属性代码查询租户属性，并通过ResponseEntityWrapperUtil封装查询结果
            return tenantAttributeService.updateAttributeValue(attribute);
        } catch (Exception e) {
            // 当查询过程中发生异常时，记录错误日志，并返回系统错误的响应实体
            log.error("TenantAttributeController.queryAttributeValue is error, attributeCode:{}", null, e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

}
