package com.digiwin.athena.semc.controller.bench;

import com.alibaba.excel.util.StringUtils;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.appcore.util.ResponseEntityWrapper;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.I18NKey;
import com.digiwin.athena.semc.entity.bench.JobBench;
import com.digiwin.athena.semc.service.bench.JobBenchService;
import com.digiwin.athena.semc.service.portal.LabelSystemAuthService;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;
import com.digiwin.athena.semc.vo.bench.JobBenchSaveVO;
import com.digiwin.athena.semc.vo.bench.OrderBenchLaneVO;
import com.digiwin.athena.semc.vo.bench.OrderLaneJobVO;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import javax.validation.Valid;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;

/**
 * @description 作业工作台
 * @author: dengkaiwen
 */
@RestController
@RequestMapping("/semc/job/bench/")
public class JobBenchController {

    private static final Logger logger = LoggerFactory.getLogger(JobBenchController.class);

    @Autowired
    JobBenchService jobBenchService;

    @Resource
    private MessageUtils messageUtils;

    @Autowired
    LabelSystemAuthService labelSystemAuthService;

    /**
     * 查询所有作业工作台接口
     * @return
     */
    @PostMapping("/queryBenchList")
    public ResponseEntity<BaseResultDTO<List<JobBench>>> queryBenchList() {
        try {
            //组件中的数据源ID集合
            List<Long> allSystemIdList=new ArrayList<>();
            return ResponseEntityWrapperUtil.wrapperOk(jobBenchService.queryAllBench(null,allSystemIdList));
        } catch (Exception e) {
            logger.error("queryBenchList exceptiona", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/job/bench/queryBenchList");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 查询模版中所有作业工作台接口
     * @return
     */
    @PostMapping("/queryTempBenchList")
    public ResponseEntity<BaseResultDTO<List<JobBench>>> queryTempBenchList() {
        try {
            return ResponseEntityWrapperUtil.wrapperOk(jobBenchService.queryTempBenchList());
        } catch (Exception e) {
            logger.error("queryTempBenchList exceptiona", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/job/bench/queryTempBenchList");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 查询权限内作业工作台接口
     * @return
     */
    @PostMapping("/queryAuthBenchList")
    public ResponseEntity<BaseResultDTO<List<JobBench>>> queryAuthBenchList() {
        try {
            //获取用户权限内的自定义组件id
            List<Long> customIdList =  labelSystemAuthService.getSystemCustom();
            return ResponseEntityWrapperUtil.wrapperOk(jobBenchService.queryAuthAllBench(customIdList));
        } catch (Exception e) {
            logger.error("queryAuthBenchList exceptiona", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/job/bench/queryAuthBenchList");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }


    /**
     * 保存工作台和泳道
     *
     * @return
     */
    @PostMapping("/saveBenchLane")
    public ResponseEntity<BaseResultDTO<Long>> saveBenchLane(@RequestBody JobBench req) {
        try {
            if (null == req.getLabelIndex()) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "labelIndex is not empty");
            }
            //校验名称是否必填
            if (StringUtils.isEmpty(req.getName())) {
                String error = messageUtils.getMessage("error.message.job.bench.name.empty");
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, error);
            }
            if (req.getName().length()>12) {
                String error = messageUtils.getMessage("error.message.job.bench.name.empty.length");
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, error);
            }
            //校验名称是否唯一
            List<JobBench> jobBenchList = jobBenchService.selectByName(req);
            if (!CollectionUtils.isEmpty(jobBenchList)) {
                String error = messageUtils.getMessage("error.message.job.bench.name.repeat");
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
            }
            return ResponseEntityWrapperUtil.wrapperOk(jobBenchService.saveBench(req));
        } catch (Exception e) {
            logger.error("saveBenchLane exceptiona", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/job/bench/saveBenchLane");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }


    /**
     * 保存作业
     *
     * @return
     */
    @PostMapping("/saveJobList")
    public ResponseEntity<?> saveJobList(@RequestBody  @Valid JobBenchSaveVO req) {
        try {

            jobBenchService.saveJobList(req);
            return ResponseEntityWrapper.wrapperOk();
        } catch (Exception e) {
            logger.error("saveJobList exceptiona", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/job/bench/saveJobList");
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }



    /**
     * 删除工作台/泳道
     *
     * @return
     */
    @PostMapping("/delBeanLane")
    public ResponseEntity<?> delBeanLane(@RequestBody JobBenchSaveVO req) {
        try {

            if (null == req.getId()) {
                return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "id is not empty");
            }
            jobBenchService.delBeanLane(req.getId());
            return ResponseEntityWrapper.wrapperOk();
        } catch (Exception e) {
            logger.error("delBeanLane exceptiona", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/job/bench/delBeanLane");
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }
    /**
     * 删除泳道下的作业
     *
     * @return
     */
    @PostMapping("/delLaneJob")
    public ResponseEntity<?> delLaneJob(@RequestBody  @Valid JobBenchSaveVO req) {
        try {

            if (CollectionUtils.isEmpty(req.getSystemIdList())) {
                String error = messageUtils.getMessage("error.message.system.data.job.empty");
                return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
            }
            jobBenchService.delLaneJob(req);
            return ResponseEntityWrapper.wrapperOk();
        } catch (Exception e) {
            logger.error("delLaneJob exceptiona", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/job/bench/delLaneJob");
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }



    /**
     * 工作台/泳道排序接口
     *
     * @return
     */
    @PostMapping("/orderBenchLane")
    public ResponseEntity<?> orderBenchLane(@RequestBody  @Valid OrderBenchLaneVO req) {
        try {
            jobBenchService.orderBenchLane(req);
            return ResponseEntityWrapper.wrapperOk();
        } catch (Exception e) {
            logger.error("orderBenchLane exceptiona", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/job/bench/orderBenchLane");
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }


    /**
     * 泳道下组件排序接口
     *
     * @return
     */
    @PostMapping("/orderLaneJob")
    public ResponseEntity<?> orderLaneJob(@RequestBody  @Valid OrderLaneJobVO req) {
        try {
            
            jobBenchService.orderLaneJob(req);
            return ResponseEntityWrapper.wrapperOk();
        } catch (Exception e) {
            logger.error("orderLaneJob exceptiona", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/job/bench/orderLaneJob");
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }
}
