package com.digiwin.athena.semc.controller.bench;

import com.alibaba.fastjson.JSON;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.appcore.util.ResponseEntityWrapper;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.I18NKey;
import com.digiwin.athena.semc.common.PageInfoResp;
import com.digiwin.athena.semc.controller.BasicController;
import com.digiwin.athena.semc.dto.message.*;
import com.digiwin.athena.semc.entity.bench.ThirdJobConfig;
import com.digiwin.athena.semc.service.bench.ThirdJobConfigService;
import com.digiwin.athena.semc.service.menu.AuthService;
import com.digiwin.athena.semc.service.sso.IErpSsoInfoService;
import com.digiwin.athena.semc.service.sso.IThirdSsoInfoService;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;
import com.digiwin.athena.semc.util.SecurityLogUtil;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.validation.Valid;
import java.time.LocalDateTime;

/**
 * @description: 三方异构系统作业接入
 * @createDate: 2024/3/4
 * @author: sungqz
 */
@Slf4j
@RestController
@RequestMapping("/semc/third/job")
public class ThirdJobConfigController extends BasicController<ThirdJobConfigService, ThirdJobConfig> {

    @Resource
    private ThirdJobConfigService thirdJobConfigService;

    @Autowired
    IErpSsoInfoService erpSsoInfoService;

    @Autowired
    IThirdSsoInfoService thirdSsoInfoService;

    @Resource
    private MessageUtils messageUtils;

    @Resource
    private AuthService authService;

    /**
     * 查询异构系统的作业配置
     *
     * @param queryImportRecordReq 入参
     * @return 返回
     */
    @PostMapping("/queryJobConfigPage")
    public ResponseEntity<BaseResultDTO<PageInfoResp<ThirdJobConfig>>> queryJobConfigPage(@RequestBody @Valid QueryThirdMessageConfigReq queryImportRecordReq) {
        try {
            PageInfoResp<ThirdJobConfig> pageInfoResp = thirdJobConfigService.queryConfigPage(queryImportRecordReq);
            return ResponseEntityWrapperUtil.wrapperOk(pageInfoResp);
        } catch (Exception e) {
            log.error("query job config error, param:{}", queryImportRecordReq, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/third/job/queryJobConfigPage");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 保存异构系统的作业配置
     *
     * @param req 入参
     * @return 返回
     */
    @PostMapping("/saveApplication")
    public ResponseEntity<BaseResultDTO<Boolean>> saveApplication(@RequestBody @Valid ThirdJobConfig req, HttpServletRequest  request) {
        try {
            //集成管理/作业接入配置
            //暂时不阻断，只是写入安全日志
            if (!authService.hasAuth("third-work")) {
                //写入安全日志
                SecurityLogUtil.writeSecurityLog(request, JSON.toJSONString(req));
            }
            // 数据获取方式如果是拉取方式
            if (Constants.DataGetTypeEnum.PULL.getFlag().equals(req.getDataGetType())) {
                if (req.getAppAccessModel() == null) {
                    return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "the appAccessModel cannot be null");
                }
                // 混合云模式需要填地中台产品名称，系统UID
                if (Constants.DataModelEnum.MODEL_HYBRID_CLOUD.getVal().equals(req.getAppAccessModel())) {
                    if (StringUtils.isBlank(req.getMiddleSystemName())) {
                        return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "the middleSystemName cannot be null");
                    }
                    if (StringUtils.isBlank(req.getMiddleSystemUid())) {
                        return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "the middleSystemUid cannot be null");
                    }
                } else { // 非混合云
                    if (StringUtils.isBlank(req.getDomain())) {
                        return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "the domain cannot be null");
                    }

                }
            }
            // 判断应用是否存在
            boolean flag = thirdJobConfigService.validJobConfigExist(req);
            if (flag) {
                String error = messageUtils.getMessage("error.message.app.repeat");
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, error);
            }
            thirdJobConfigService.saveThirdJob(req);
            return ResponseEntityWrapperUtil.wrapperOk(true);
        } catch (Exception e) {
            log.error("saveApplication error:{}", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/third/job/saveApplication");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }


    /**
     * 删除作业的配置
     *
     * @param deleteMessageConfigReq 入参
     * @return 返回
     */
    @PostMapping("/delJobConfig")
    public ResponseEntity<BaseResultDTO<Boolean>> delJobConfig(@RequestBody DeleteMessageConfigReq deleteMessageConfigReq) {
        try {
            if (CollectionUtils.isEmpty(deleteMessageConfigReq.getIdList())) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, "idList is not empty");
            }
            thirdJobConfigService.delThirdJob(deleteMessageConfigReq);
            return ResponseEntityWrapperUtil.wrapperOk(true);
        } catch (Exception e) {
            log.error("delJobConfig error. e:{}", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/third/job/delJobConfig");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }


    /**
     * 更新作业配置状态
     *
     * @param req 入参
     * @return 返回
     */
    @PostMapping("/updateStatus")
    public ResponseEntity<?> updateStatus(@RequestBody ThirdJobConfig req) {
        try {
            if (null == req.getId()) {
                return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, "id is Empty");
            }
            if (null == req.getValidStatus()) {
                return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, "validStatus is Empty");
            }
            ThirdJobConfig config=thirdJobConfigService.getById(req.getId());
            config.setValidStatus(req.getValidStatus());
            thirdJobConfigService.saveThirdJob(config);
            return ResponseEntityWrapper.wrapperOk(true);
        } catch (Exception e) {
            log.error("/semc/third/job/updateStatus error:{}", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/third/job/updateStatus");
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }
}