package com.digiwin.athena.semc.controller.device;

import com.alibaba.excel.EasyExcel;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.semc.annotate.MenuPermissionAuth;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.I18NKey;
import com.digiwin.athena.semc.common.PageInfoResp;
import com.digiwin.athena.semc.common.enums.FolderTypeEnum;
import com.digiwin.athena.semc.dto.device.*;
import com.digiwin.athena.semc.dto.portal.QueryImportRecordReq;
import com.digiwin.athena.semc.service.device.DeviceBindDataWriteHandler;
import com.digiwin.athena.semc.service.device.IDeviceAuthService;
import com.digiwin.athena.semc.service.device.IUserBindDeviceFileService;
import com.digiwin.athena.semc.service.device.IUserBindDeviceService;
import com.digiwin.athena.semc.service.portal.IImportRecordService;
import com.digiwin.athena.semc.util.DateUtils;
import com.digiwin.athena.semc.service.device.*;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;
import com.digiwin.athena.semc.util.Utils;
import com.digiwin.athena.semc.vo.common.ImportRecordResp;
import com.digiwin.athena.semc.vo.portal.ImportExcelResp;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.http.ResponseEntity;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;
import javax.validation.Valid;
import java.nio.charset.StandardCharsets;
import java.time.LocalDate;
import java.util.List;
import java.util.Objects;

/**
 * @Author jf gui
 * @Description 设备绑定记录控制器
 */
@Slf4j
@RestController
@RequestMapping("/semc/device/bind")
public class DeviceBindController {

    public static final List<String> OS_TYPES= Lists.newArrayList("Windows", "macOS", "iOS", "Android", "Other");

    @Resource
    private MessageUtils messageUtils;

    @Resource
    private IUserBindDeviceService iUserBindDeviceService;

    @Resource
    private IDeviceAuthService iDeviceAuthService;

    @Resource
    private IUserBindDeviceFileService iUserBindDeviceFileService;

    @Resource
    private IImportRecordService importRecordService;

    /**
     * 查询设备绑定列表
     */
    @PostMapping("/list")
    public ResponseEntity<BaseResultDTO<PageInfoResp<DeviceBindRecordVO>>> list(@RequestBody DeviceBindQueryDTO request) {
        return ResponseEntityWrapperUtil.wrapperOk(iUserBindDeviceService.list(request));
    }

    /**
     * 添加设备绑定
     */
    @MenuPermissionAuth(code = "account-device-bind")
    @PostMapping("/save")
    public ResponseEntity<BaseResultDTO<DeviceBindRecordVO>> save(@Valid @RequestBody DeviceBindSaveDTO record) {
        return ResponseEntityWrapperUtil.wrapperOk(iUserBindDeviceService.save(record));
    }

    /**
     * 修改设备绑定
     */
    @MenuPermissionAuth(code = "account-device-bind")
    @PostMapping("/update")
    public ResponseEntity<BaseResultDTO<DeviceBindRecordVO>> update(@Valid @RequestBody DeviceBindUpdateDTO update) {
        return ResponseEntityWrapperUtil.wrapperOk(iUserBindDeviceService.update(update));
    }

    /**
     * 修改设备绑定有效状态
     */
    @MenuPermissionAuth(code = "account-device-bind")
    @PostMapping("/updateValidStatus")
    public ResponseEntity<BaseResultDTO<Boolean>> updateValidStatus(@Valid @RequestBody DeviceBindUpdateStatusDTO update) {
        return ResponseEntityWrapperUtil.wrapperOk(iUserBindDeviceService.updateValidStatus(update));
    }

    /**
     * 删除设备绑定
     */
    @MenuPermissionAuth(code = "account-device-bind")
    @PostMapping("/delete")
    public ResponseEntity<BaseResultDTO<Boolean>> delete(@RequestBody List<Long> ids) {
        return ResponseEntityWrapperUtil.wrapperOk(iUserBindDeviceService.delete(ids));
    }

    /**
     * 导入设备绑定
     */
    @MenuPermissionAuth(code = "account-device-bind")
    @PostMapping("/importReport")
    public ResponseEntity<BaseResultDTO<ImportExcelResp>> importRecords(@RequestBody DeviceBindImportDTO req) {
        return ResponseEntityWrapperUtil.wrapperOk(iUserBindDeviceFileService.importRecords(req));
    }

    /**
     * 导出设备绑定
     */
    @PostMapping("/exportReport")
    public void exportReport(HttpServletResponse response, @RequestBody DeviceBindExportQueryDTO req) {
        try {
            List<DeviceBindExportVO> deviceBindRecords = iUserBindDeviceService.exportReport(req);
            //设置文件名
            String filename = "设备绑定配置_" + LocalDate.now().toString().replace("-", "") + ".xlsx";
            //设置下载信息
            response.setContentType("application/vnd.ms-excel");
            response.setCharacterEncoding("utf-8");
            response.setHeader("Content-disposition", "attachment;filename=" + new String(filename.getBytes(StandardCharsets.UTF_8), StandardCharsets.ISO_8859_1));
            EasyExcel.write(response.getOutputStream(), DeviceBindExportVO.class)
                    .sheet("设备绑定配置")
                    .registerWriteHandler(new DeviceBindHeadWriteHandler())
                    .relativeHeadRowIndex(1)
                    .registerWriteHandler(Utils.initHorizontalCell())
                    .registerWriteHandler(new DeviceBindDataWriteHandler())
                    .doWrite(deviceBindRecords);
        } catch (Exception e) {
            log.error("FineReporterController export report excel exception", e);
            throw new RuntimeException(e);
        }
    }

    /**
     * 查询导入记录
     *
     * @param queryImportRecordReq 入参
     * @return
     */
    @PostMapping("/queryImportRecord")
    public ResponseEntity<BaseResultDTO<PageInfoResp<ImportRecordResp>>> queryImportRecord(@RequestBody @Valid QueryImportRecordReq queryImportRecordReq) {
        try {
            // 校验时间格式
            if ((org.apache.commons.lang.StringUtils.isNotBlank(queryImportRecordReq.getStartTime()) && !DateUtils.validateDate(queryImportRecordReq.getStartTime(), DateUtils.DATE_TIME_NORMAL_FORMATTER)) || (org.apache.commons.lang.StringUtils.isNotBlank(queryImportRecordReq.getEndTime()) && !DateUtils.validateDate(queryImportRecordReq.getEndTime(), DateUtils.DATE_TIME_NORMAL_FORMATTER))) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.COMMON_PARAM_ILLEGAL));
            }
            PageInfoResp<ImportRecordResp> pageInfoResp = importRecordService.queryImportRecord(queryImportRecordReq, FolderTypeEnum.DEVICE_REPORT.getValue());
            return ResponseEntityWrapperUtil.wrapperOk(pageInfoResp);
        } catch (Exception e) {
            log.error("DeviceBindImportRecordController queryImportRecord  exception", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 查询账号绑定设备
     */
    @GetMapping("/query")
    public ResponseEntity<BaseResultDTO<List<DeviceBindRecordVO>>> query() {
        return ResponseEntityWrapperUtil.wrapperOk(iUserBindDeviceService.queryUserDevice());
    }

    /**
     * 账号登陆设备认证
     */
    @PostMapping("/authorize")
    public ResponseEntity<BaseResultDTO<DeviceAuthorizeVO>> authorize(@RequestBody @Validated DeviceDTO device) {
        Pair<Integer, String> check = check(device);
        if (Objects.nonNull(check)) {
            return ResponseEntityWrapperUtil.wrapperFail(check.getKey(), check.getRight());
        }
        return ResponseEntityWrapperUtil.wrapperOk(iDeviceAuthService.authorize(device));
    }

    /**
     * 账号设备登陆首次绑定
     */
    @PostMapping("/auto")
    public ResponseEntity<BaseResultDTO<Boolean>> autoBind(@RequestBody @Validated DeviceDTO device) {
        Pair<Integer, String> check = check(device);
        if (Objects.nonNull(check)) {
            return ResponseEntityWrapperUtil.wrapperFail(check.getKey(), check.getRight());
        }
        Boolean autoBind = iDeviceAuthService.autoBind(device);
        if (BooleanUtils.isTrue(autoBind)) {
            return ResponseEntityWrapperUtil.wrapperOk(autoBind);
        }
        return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
    }

    public Pair<Integer, String> check(DeviceDTO device) {
        if (StringUtils.isNotBlank(device.getHardwareUuid()) && device.getHardwareUuid().length() > 255) {
            device.setHardwareUuid(null);
        }
        if (StringUtils.isNotBlank(device.getMacAddress()) && device.getMacAddress().length() > 255) {
            device.setDeviceModel(null);
        }
        if (StringUtils.isNotBlank(device.getOsType()) && !OS_TYPES.contains(device.getOsType())) {
            device.setOsType(null);
        }
        return null;
    }
}    