package com.digiwin.athena.semc.controller.event;


import com.alibaba.excel.EasyExcel;
import com.baomidou.mybatisplus.core.toolkit.StringUtils;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.semc.common.I18NKey;
import com.digiwin.athena.semc.common.PageInfoResp;
import com.digiwin.athena.semc.dto.event.EventTrackingReq;
import com.digiwin.athena.semc.dto.event.PageQueryEventTrackingDTO;
import com.digiwin.athena.semc.service.event.EventTrackingService;
import com.digiwin.athena.semc.service.portal.impl.SystemCustomWriteHandler;
import com.digiwin.athena.semc.util.DateUtils;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;
import com.digiwin.athena.semc.util.Utils;
import com.digiwin.athena.semc.vo.event.PageQueryEventTrackingVO;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.stream.Collectors;

/**
 * Description: 事件埋点
 */
@Slf4j
@RestController
@RequestMapping("/semc/event")
public class EventTrackingController {
    private static final Logger logger = LoggerFactory.getLogger(EventTrackingController.class);

    @Resource
    private EventTrackingService eventTrackingService;

    @Resource
    private MessageUtils messageUtils;

    @PostMapping("/saveTracking")
    public ResponseEntity<?> saveTracking(@RequestBody EventTrackingReq req) {
        ResponseEntity<BaseResultDTO<Integer>> responseEntity = ResponseEntityWrapperUtil.wrapperOk(0);
        try {
            if (req == null || (req.getTs() == null && req.getCommon() == null && req.getUser() == null && req.getEvents() == null)) {
                responseEntity.getBody().setErrorMessage("参数不能为空");
                return responseEntity;
            }
            if (StringUtils.isBlank(req.getTs())) {
                responseEntity.getBody().setErrorMessage("事件时间不能为空");
                return responseEntity;
            }
            // 校验时间参数
            if (StringUtils.isNotBlank(req.getTs()) && !DateUtils.validateDate(req.getTs(),
                    DateUtils.DATE_TIME_NORMAL_FORMATTER)) {
                responseEntity.getBody().setErrorMessage("事件时间格式不正确");
                return responseEntity;
            }
            if (CollectionUtils.isEmpty(req.getEvents())) {
                responseEntity.getBody().setErrorMessage("事件不能为空");
                return responseEntity;
            }
            if (CollectionUtils.isNotEmpty(req.getEvents())) {
                List<EventTrackingReq.EventsDTO> eventCodes = req.getEvents().stream().filter(x -> StringUtils.isBlank(x.getEventCode())).collect(Collectors.toList());
                if (CollectionUtils.isNotEmpty(eventCodes)) {
                    responseEntity.getBody().setErrorMessage("事件id不能为空");
                    return responseEntity;
                }
            }
            // 最后成功
            responseEntity.getBody().setResponse(eventTrackingService.saveTracking(req));
        } catch (Exception e) {
            logger.error("saveTracking exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/saveTracking");
            responseEntity.getBody().setErrorMessage(error);
        }
        return responseEntity;
    }

    /**
     * 分页查询用户日志
     *
     * @param pageQueryEventTrackingDTO 查询参数
     * @return 响应结果
     */
    @PostMapping("pageQuery")
    public ResponseEntity<BaseResultDTO<PageInfoResp<PageQueryEventTrackingVO>>> pageQuery(@RequestBody PageQueryEventTrackingDTO pageQueryEventTrackingDTO) {
        return ResponseEntityWrapperUtil.wrapperOk(eventTrackingService.pageQuery(pageQueryEventTrackingDTO));
    }

    /**
     * 导出所有按条件查出的结果
     *
     * @param response 响应
     */
    @PostMapping("exportReport")
    public void exportReport(@RequestBody PageQueryEventTrackingDTO pageQueryEventTrackingDTO, HttpServletResponse response) throws IOException {
        List<PageQueryEventTrackingVO> pageQueryEventTrackingVOList = eventTrackingService.exportReport(pageQueryEventTrackingDTO);
        String fileName = "用户日志_" + LocalDateTime.now().format(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss")) + ".xlsx";
        response.setContentType("application/vnd.ms-excel");
        response.setCharacterEncoding("utf-8");
        response.setHeader("Content-disposition", "attachment;filename=" + new String(fileName.getBytes(StandardCharsets.UTF_8), StandardCharsets.ISO_8859_1));
        EasyExcel.write(response.getOutputStream(), PageQueryEventTrackingVO.class)
                .registerWriteHandler(Utils.initHorizontalCell())
                .sheet("用户日志").relativeHeadRowIndex(1).registerWriteHandler(new SystemCustomWriteHandler()).doWrite(pageQueryEventTrackingVOList);
    }
}
