package com.digiwin.athena.semc.controller.fineReport;


import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.semc.common.*;
import com.digiwin.athena.semc.dto.fineReport.FineReportConfigDTO;
import com.digiwin.athena.semc.dto.fineReport.SaveFineReportUserMappingConfigReq;
import com.digiwin.athena.semc.dto.portal.FineReportQueryReq;
import com.digiwin.athena.semc.entity.fineReport.FineReportUserMapping;
import com.digiwin.athena.semc.service.fineReport.FineReportManageService;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;
import io.github.linpeilie.Converter;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.validation.Valid;

/**
 * 三方应用-帆软报表controller
 *
 * @author sunqgz
 * @since 2025-08-11
 */
@Slf4j
@RestController
@RequestMapping("/semc/tripartite/fineReport")
public class FineReportManageController {

    @Resource
    private FineReportManageService fineReportManageService;

    @Resource
    private Converter converter;

    @Resource
    private MessageUtils messageUtils;

    /**
     * 保存租户的帆软参数配置
     *
     * @return 查询结果
     */
    @PostMapping("/saveFineReportConfig")
    public ResponseEntity<?> saveFineReportConfig(@RequestBody @Valid FineReportConfigDTO fineReportConfigDTO) {
        fineReportManageService.saveFineReportConfig(fineReportConfigDTO);
        return ResponseEntityWrapperUtil.wrapperOk(true);
    }

    /**
     * 查询雅典娜用户和帆软用户的映射关系
     *
     * @return 查询结果
     */
    @PostMapping("/queryUserMappingList")
    public ResultPageBean queryUserMappingList(@Valid @RequestBody FineReportQueryReq req) {
        ResultPageBean resultPageBean;
        String tenantId = AppAuthContextHolder.getContext().getAuthoredUser().getTenantId();
        try {
            return fineReportManageService.queryUserMappings(tenantId, req);
        } catch (BizException bizException) {
            return ResultPageBean.bizException(BizException.getDefaultBizException(ErrorCodeConstant.ADAPT_SYSTEM_ERROR,
                    messageUtils.getMessage(I18NKey.REPORT_OPEN_ERROR)));
        } catch (Exception e) {
            log.error("/semc/fineReport/pageQuery exception", e);
            resultPageBean = ResultPageBean.sysException(new ServiceException(ErrorCodeConstant.SYSTEM_ERROR,
                    messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR)));

        }
        return resultPageBean;
    }

    /**
     * 新增/编辑雅典娜用户和帆软用户的映射关系
     *
     * @return 查询结果
     */
    @PostMapping("/saveUserMapping")
    public ResponseEntity<?> saveUserMapping(@RequestBody @Valid SaveFineReportUserMappingConfigReq req) {
        try {
            FineReportUserMapping fineReportUserMapping = converter.convert(req, FineReportUserMapping.class);
            String tenantId = AppAuthContextHolder.getContext().getAuthoredUser().getTenantId();
            fineReportUserMapping.setTenantId(tenantId);
            FineReportUserMapping result = fineReportManageService.saveUserMapping(fineReportUserMapping);
            return ResponseEntityWrapperUtil.wrapperOk(result);
        } catch (Exception e) {
            log.error("/semc/fineReport/saveUserMapping exception", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, e.getMessage());
        }
    }

    /**
     * 查询帆软用户映射关系详情
     *
     * @return 帆软用户映射关系详情
     */
    @GetMapping("/userMapping/{id}")
    public ResponseEntity<?> queryUserMappingDetail(@PathVariable Long id) {
        return ResponseEntityWrapperUtil.wrapperOk(fineReportManageService.getById(id));
    }

    /**
     * 删除帆软用户映射关系
     *
     * @return 删除结果
     */
    @DeleteMapping("/userMapping/{id}")
    public ResponseEntity<?> deleteUserMapping(@PathVariable Long id) {
        try {
            fineReportManageService.deleteUserMapping(id);
            return ResponseEntityWrapperUtil.wrapperOk();
        } catch (Exception e) {
            log.error("delete /semc/fineReport/userMapping exception", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, e.getMessage());
        }
    }

    /**
     * 查询雅典娜用户
     *
     * @return 查询结果
     */
    @GetMapping("/queryAthenaUserList")
    public ResponseEntity<?> queryAthenaUserList() {
        return ResponseEntityWrapperUtil.wrapperOk(fineReportManageService.queryAthenaUserList());
    }

    /**
     * 查询帆软用户数量
     *
     * @return 查询结果
     */
    @GetMapping("/queryFrUser/{frUserId}")
    public ResponseEntity<?> queryFrUser(@PathVariable String frUserId) {
        long count = fineReportManageService.count(new LambdaQueryWrapper<FineReportUserMapping>().eq(FineReportUserMapping::getFrUserId, frUserId)
                .eq(FineReportUserMapping::getTenantId, AppAuthContextHolder.getContext().getAuthoredUser().getTenantId()));
        return ResponseEntityWrapperUtil.wrapperOk(count);
    }
}

