package com.digiwin.athena.semc.controller.homepage.upgrade;

import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.I18NKey;
import com.digiwin.athena.semc.common.PageInfoResp;
import com.digiwin.athena.semc.dto.homepage.AdvertisementDTO;
import com.digiwin.athena.semc.dto.homepage.ReadAdvertisementReq;
import com.digiwin.athena.semc.entity.bench.JobBenchImport;
import com.digiwin.athena.semc.entity.homepage.ReadAdvertisement;
import com.digiwin.athena.semc.service.homepage.IAdvertisementService;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;
import io.swagger.v3.oas.annotations.Operation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.validation.Valid;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @description 首页tbb消息通知列表
 * @author: sungqz
 * @create: 2023-8-23
 */
@RestController
@RequestMapping("/semc/advertisement")
public class AdvertisementController {

    private static final Logger logger = LoggerFactory.getLogger(AdvertisementController.class);

    @Autowired
    IAdvertisementService advertisementService;

    @Resource
    private MessageUtils messageUtils;

    /**
     * 重要性 important:重要 unimportant:不重要
     */
    public static final String AD_IMPORTANT_FLAG = "important";

    /**
     * 查询未读数量
     *
     * @param accessToken accessToken
     * @return
     */
    @Operation(summary = "查询未读数量")
    @GetMapping("/queryUnreadCount")
    public ResponseEntity<BaseResultDTO<JSONObject>> queryUnreadCount(@RequestParam String accessToken) {
        try {
            List<AdvertisementDTO> advertisementDTOList = advertisementService.queryAdListFromEs(accessToken);
            // 未读数量
            long unReadCount = advertisementDTOList.stream().filter(x -> Boolean.FALSE.equals(x.getReadStatus())).count();
            // 重要的未读数量
            long importUnReadCount = advertisementDTOList.stream().filter(x -> AD_IMPORTANT_FLAG.equals(x.getTriggerEvent()) && Boolean.FALSE.equals(x.getReadStatus())).count();
            JSONObject jsonObject = new JSONObject();
            jsonObject.put("unRead", unReadCount);
            jsonObject.put("importUnRead", importUnReadCount);
            return ResponseEntityWrapperUtil.wrapperOk(jsonObject);
        } catch (Exception e) {
            logger.error("AdvertisementController | query unread count occur error. accessToken:{}", accessToken, e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 查询广告列表
     *
     * @param accessToken accessToken
     * @return
     */
    @Operation(summary = "查询广告列表")
    @GetMapping("/queryAdList")
    public ResponseEntity<BaseResultDTO<List<AdvertisementDTO>>> queryAdList(@RequestParam String accessToken) {
        try {
            List<AdvertisementDTO> advertisementDTOList = advertisementService.queryAdListFromEs(accessToken);
            return ResponseEntityWrapperUtil.wrapperOk(advertisementDTOList);
        } catch (Exception e) {
            logger.error("AdvertisementController | query advertisement list occur error. accessToken:{}", accessToken, e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 记录阅读的广告
     *
     * @param readAdvertisementReq 记录广告请求对象
     * @return
     */
    @Operation(summary = "记录阅读的广告")
    @PostMapping("/readAdNotice")
    public ResponseEntity<BaseResultDTO<Boolean>> readAdNotice(@RequestBody @Valid ReadAdvertisementReq readAdvertisementReq) {
        try {
            if (!Arrays.asList(Constants.ReadAdPlatformEnum.ENTRANCE.getFlag(), Constants.ReadAdPlatformEnum.TBB.getFlag()).contains(readAdvertisementReq.getReadPlatform())) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.COMMON_PARAM_ILLEGAL));
            }
            Boolean readFlag = advertisementService.insertAdReadRecord(readAdvertisementReq);
            if (!readFlag) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.UPDATE_FAILED, messageUtils.getMessage(I18NKey.COMMON_UPDATE_ERROR));
            }
            return ResponseEntityWrapperUtil.wrapperOk(readFlag);
        } catch (Exception e) {
            logger.error("AdvertisementController | read advertisement occur error. param:{}", readAdvertisementReq, e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 查询已读的广告列表
     *
     * @return
     */
    @Operation(summary = "查询已读的广告列表")
    @GetMapping("/queryReadList")
    public ResponseEntity<BaseResultDTO<List<ReadAdvertisement>>> queryReadList() {
        try {
            // 查询当前租户下当前用户的已读列表
            List<ReadAdvertisement> readAdvertisementList = advertisementService.queryReadList();
            return ResponseEntityWrapperUtil.wrapperOk(readAdvertisementList);
        } catch (Exception e) {
            logger.error("AdvertisementController | query read list occur error.", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }
}