package com.digiwin.athena.semc.controller.homepage.upgrade;

import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.I18NKey;
import com.digiwin.athena.semc.dto.homepage.SaveTenantPortalReq;
import com.digiwin.athena.semc.dto.homepage.SaveUserPortalReq;
import com.digiwin.athena.semc.entity.homepage.TenantPortalRecord;
import com.digiwin.athena.semc.entity.homepage.UserPortalRecord;
import com.digiwin.athena.semc.service.homepage.IUserPortalService;
import com.digiwin.athena.semc.service.homepage.TenantPortalRecordService;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;
import io.swagger.v3.oas.annotations.Operation;
import lombok.extern.slf4j.Slf4j;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.validation.Valid;
import java.time.LocalDateTime;
import java.util.Arrays;

/**
 * @description 首页用户门户风格
 * @author: sungqz
 * @create: 2023-3-27
 */
@Slf4j
@RestController
@RequestMapping("/semc/portal")
public class UserPortalController {

    private static final Logger logger = LoggerFactory.getLogger(UserPortalController.class);

    @Autowired
    IUserPortalService userPortalService;

    /**
     * 服务对象
     */
    @Resource
    private TenantPortalRecordService tenantPortalRecordService;

    @Resource
    private MessageUtils messageUtils;

    /**
     * 保存切换的门户类型
     *
     * @param saveUserPortalReq 请求实体
     * @return
     * @throws Exception
     */
    @Operation(summary = "保存切换的门户类型")
    @PostMapping("/saveLatest")
    public ResponseEntity<BaseResultDTO<Long>> saveLatestPortal(@RequestBody SaveUserPortalReq saveUserPortalReq) {
        logger.info("saveLatest param:"+ JSONObject.toJSONString(saveUserPortalReq));
        // 参数校验
        if (saveUserPortalReq.getPortalType() == null) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR,
                    messageUtils.getMessage(I18NKey.COMMON_PARAM_MISSING));
        }
        if (!Arrays.asList(Constants.PortalTypeEnum.SIMPLE.getFlag(),
                Constants.PortalTypeEnum.CONVENTIONAL.getFlag()).contains(saveUserPortalReq.getPortalType())) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR,
                    messageUtils.getMessage(I18NKey.COMMON_PARAM_ILLEGAL));
        }

        // 保存
        try {
            Long id = userPortalService.saveLatestUserPortal(saveUserPortalReq);
            return ResponseEntityWrapperUtil.wrapperOk(id);
        } catch (Exception e) {
            logger.error("save latest portal type error. saveUserPortalReq:{},e:{}", saveUserPortalReq, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/portal/saveLatest, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 查询用户当前使用的门户类型
     *
     * @return ResponseEntity
     */
    @Operation(summary = "查询用户当前使用的门户类型")
    @GetMapping("/queryLatest")
    public ResponseEntity<BaseResultDTO<UserPortalRecord>> queryLatestPortal() {
        try {
            TenantPortalRecord tenantPortalRecord = tenantPortalRecordService.queryTenantPortal();
            if (tenantPortalRecord == null || Constants.USE_LATEST_FLAG_YES.equals(tenantPortalRecord.getUseLatestFlag())) {
                UserPortalRecord userPortalRecord = userPortalService.queryLatestUserPortal();
                return ResponseEntityWrapperUtil.wrapperOk(userPortalRecord);
            } else {
                UserPortalRecord userPortalRecord = new UserPortalRecord();
                userPortalRecord.setPortalType(tenantPortalRecord.getPortalType());
                return ResponseEntityWrapperUtil.wrapperOk(userPortalRecord);
            }
        } catch (Exception e) {
            logger.error("query latest portal type error.", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/portal/queryLatest, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 查询当前租户的门户类型
     *
     * @return 返回
     */
    @PostMapping("/tenant/queryTenantPortal")
    public ResponseEntity<BaseResultDTO<TenantPortalRecord>> queryTenantPortal() {
        try {
            TenantPortalRecord tenantPortalRecord = tenantPortalRecordService.queryTenantPortal();
            return ResponseEntityWrapperUtil.wrapperOk(tenantPortalRecord);
        } catch (Exception e) {
            logger.error("query tenant portal error.", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/tenant/portal/queryTenantPortal, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 保存当前租户的门户类型
     *
     * @return 返回
     */
    @PostMapping("/tenant/saveTenantPortal")
    public ResponseEntity<BaseResultDTO<Boolean>> saveTenantPortal(@RequestBody @Valid SaveTenantPortalReq saveTenantPortalReq) {
        try {
            tenantPortalRecordService.saveTenantPortal(saveTenantPortalReq);
            return ResponseEntityWrapperUtil.wrapperOk(true);
        } catch (Exception e) {
            logger.error("save tenant portal error.", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/tenant/portal/saveTenantPortalReq, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }
}
