package com.digiwin.athena.semc.controller.jacoco;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.servlet.http.HttpServletResponse;
import java.io.*;
import java.lang.reflect.Method;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Enumeration;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

/**
 * jacoco 相关的接口
 * 1.生成解析问题
 * 2.生成解析文件并导出报告
 */
@RestController
@Slf4j
@RequestMapping(value = "/api/semc/jacoco")
public class JacocoController {

    // 工作目录
    @Value("${jacoco.workingDir:/usr/local/semc}")
    private String workingDir;
    // JaCoCo jar包位置
    @Value("${jacoco.jarDir:/usr/local/semc/jacoco/lib}")
    private String jacocoJarDir;
    // 目录下必须包含源码编译过的class文件,用来统计覆盖率。所以这里用server打出的jar包地址即可,运行的jar或者Class目录
    @Value("${jacoco.workingDir:/usr/local/semc}/${jacoco.classesJarDir:semc-1.0.0.jar}")
    private String classesJarDir;
    // 源码jar包位置
    @Value("${jacoco.workingDir:/usr/local/semc}/${jacoco.sourceJarDir:semc-1.0.0-sources.jar}")
    private String sourceJarDir;

    private void dumpJacocoExec() {
        try {
            Class<?> rt = Class.forName("org.jacoco.agent.rt.RT");
            Method getAgent = rt.getMethod("getAgent");
            Method dump = getAgent.getReturnType().getMethod("dump", boolean.class);
            Object agent = getAgent.invoke(null);
            //  if true, the current execution data is cleared
            dump.invoke(agent, false);
        } catch (ClassNotFoundException e) {
            log.error("Jacoco not enabled", e);
            throw new RuntimeException("Jacoco not enabled");
        } catch (Exception e) {
            log.error("Dump exec failed", e);
            throw new RuntimeException("Dump exec failed");
        }
    }

    /**
     * 将目录下所有文件打包为 ZIP，写入 outputStream。
     */
    public void zipDirectory(String name, Path sourceDir, OutputStream outputStream) throws IOException {
        try (ZipOutputStream zipOut = new ZipOutputStream(outputStream)) {
            Files.walk(sourceDir)
                    .forEach(path -> {
                        Path rel = sourceDir.relativize(path);
                        String entryName = (name + "/" + rel.toString().replace(File.separatorChar, '/')
                                + (Files.isDirectory(path) ? "/" : "")).replace("//", "/");
                        try {
                            zipOut.putNextEntry(new ZipEntry(entryName));
                            if (Files.isRegularFile(path)) {
                                Files.copy(path, zipOut);
                            }
                            zipOut.closeEntry();
                        } catch (IOException e) {
                            log.error("An error occurred in packaging file : {}", path, e);
                        }
                    });
            zipOut.finish();
        }
    }

    private void extractClassesJar() {
        File classesDirectory = new File(workingDir, "jacoco_classes");
        if (!classesDirectory.exists()) {
            try {
                extractJar(classesJarDir, workingDir + "/jacoco_classes");
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
    }

    private void extractSourceJar() {
        File sourceDirectory = new File(workingDir, "src/main/java");
        if (!sourceDirectory.exists()) {
            try {
                extractJar(sourceJarDir, workingDir + "/src/main/java");
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
    }

    private void extractJar(String jarPath, String destDir) throws IOException {
        JarFile jarFile = new JarFile(jarPath);
        Enumeration<JarEntry> entries = jarFile.entries();

        while (entries.hasMoreElements()) {
            JarEntry entry = entries.nextElement();
            File entryFile = new File(destDir, entry.getName());

            if (entry.isDirectory()) {
                entryFile.mkdirs();
            } else {
                // 确保父目录存在
                entryFile.getParentFile().mkdirs();

                try (InputStream in = jarFile.getInputStream(entry);
                     OutputStream out = new FileOutputStream(entryFile)) {

                    byte[] buffer = new byte[4096];
                    int bytesRead;

                    while ((bytesRead = in.read(buffer)) != -1) {
                        out.write(buffer, 0, bytesRead);
                    }
                }
            }
        }

        jarFile.close();
        log.info("Decompression completed: " + destDir);
    }

    private void execCmd(String coverageReportDir) throws InterruptedException, IOException {
        int exitCode = new ProcessBuilder()
                .command("java", "-jar",
                        jacocoJarDir + "/jacococli.jar", "report", workingDir + "/platform/jacoco/jacoco.exec",
                        "--classfiles", workingDir + "/jacoco_classes/BOOT-INF/classes",
                        "--sourcefiles", sourceJarDir,
                        "--html", workingDir + coverageReportDir)
                .redirectErrorStream(true)
                .start()
                .waitFor();
        if (exitCode == 0) {
            log.info("HTML report generation successfully!");
        } else {
            log.error("Report generation failed. exitCode: {}", exitCode);
        }
    }

    /**
     * 获取Jacoco exec 文件
     *
     * @return
     */
    @GetMapping("/getJacocoExec")
    public void getJacocoExec(HttpServletResponse response) throws IOException {
        dumpJacocoExec();

        Path execPath = Paths.get(workingDir, "platform", "jacoco", "jacoco.exec");
        // 使用缓冲流传输大文件
        try (OutputStream os = response.getOutputStream();
             InputStream is = Files.newInputStream(execPath)) {
            byte[] buffer = new byte[4096];
            int bytesRead;
            while ((bytesRead = is.read(buffer)) != -1) {
                os.write(buffer, 0, bytesRead);
            }
            os.flush();
        }
    }

    /**
     * 获取Jacoco HTML报告
     *
     * @param response
     * @throws Exception
     */
    @GetMapping("/downloadJacocoReport")
    public void getReport(HttpServletResponse response) throws Exception {
        dumpJacocoExec();
        extractClassesJar();
        extractSourceJar();
        String coverageReportDir = "/coveragereport";
        execCmd(coverageReportDir);
        String filename = "semc-jacoco-report_" + LocalDateTime.now().format(DateTimeFormatter.ofPattern("yyyyMMddHHmmss")) + ".zip";
        // 设置响应头
        response.setContentType("application/octet-stream");
        response.setHeader("Content-Disposition", "attachment; filename=\"" + filename + "\"");
        try (OutputStream os = response.getOutputStream()) {
            // 写入你的数据
            zipDirectory(filename, Paths.get(workingDir + coverageReportDir), os);
            os.flush();
        }
    }
}
