package com.digiwin.athena.semc.controller.menu;

import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.semc.common.*;
import com.digiwin.athena.semc.dto.menu.DeleteMenuTemplateReq;
import com.digiwin.athena.semc.dto.menu.QueryMenuTemplateReq;
import com.digiwin.athena.semc.dto.menu.UpdateMenuTemplateReq;
import com.digiwin.athena.semc.entity.menu.CustomizedMenuTemplate;
import com.digiwin.athena.semc.entity.portal.PortalInfo;
import com.digiwin.athena.semc.service.menu.CustomizedMenuTemplateService;
import com.digiwin.athena.semc.service.portal.PortalInfoService;
import com.digiwin.athena.semc.service.tenant.TenantAppService;
import com.digiwin.athena.semc.service.workbench.constant.PublishSourceEnum;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;
import com.digiwin.athena.semc.vo.portal.PortalUserAuthResp;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.validation.Valid;
import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * 自定义菜单模板表(TCustomizedMenuTemplate)表控制层
 *
 * @author sungqz
 * @since 2023-11-08
 */
@Slf4j
@RestController
@RequestMapping("/semc/customized/template")
public class CustomizedMenuTemplateController {

    @Resource
    private CustomizedMenuTemplateService customizedMenuTemplateService;

    @Resource
    private PortalInfoService portalInfoService;

    @Resource
    private TenantAppService tenantAppService;

    @Resource
    private MessageUtils messageUtils;

    /**
     * 自定义菜单列表配置查询菜单模板列表
     *
     * @return 查询结果
     */
    @PostMapping("/queryMenuTemplateList")
    public ResponseEntity<BaseResultDTO<PageInfoResp<CustomizedMenuTemplate>>> queryMenuTemplateList(@RequestBody @Valid QueryMenuTemplateReq queryMenuTemplateReq) {
        try {
            PageInfoResp<CustomizedMenuTemplate> templateRespList = customizedMenuTemplateService.queryMenuTemplateList(queryMenuTemplateReq);
            return ResponseEntityWrapperUtil.wrapperOk(templateRespList);
        } catch (Exception e) {
            log.error("CustomizedMenuTemplateController query menu template list error. param:{}", queryMenuTemplateReq, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/customized/template/queryMenuTemplateList, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 更新菜单模板状态
     *
     * @param updateMenuTemplateReq 入参
     * @return 返回
     */
    @PostMapping("/updateTemplateStatus")
    public ResponseEntity<BaseResultDTO<Boolean>> updateMenuTemplateStatus(@RequestBody @Valid UpdateMenuTemplateReq updateMenuTemplateReq) {
        try {
            if (CollectionUtils.isEmpty(updateMenuTemplateReq.getIdList())) {
                return ResponseEntityWrapperUtil.wrapperOk(true);
            }

            String tenantId = AppAuthContextHolder.getContext().getAuthoredUser().getTenantId();
            boolean isOnlyAsa = tenantAppService.isOnlyAsaSalesPlan(tenantId);

            // 取消发布，判断是否都关联了门户
            // 用户只购买娜娜销售包时，没有门户概念，此时不做门户关联限制
            if (Constants.VALID_STATUS_UNUSABLE.equals(updateMenuTemplateReq.getTemplateStatus()) && !isOnlyAsa) {
                // 单个更新
                if (updateMenuTemplateReq.getIdList().size() == Constants.NUMBER_ONE) {
                    if (queryPortalByMenuTempId(updateMenuTemplateReq.getIdList())) {
                        return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.MENU_TEMPLATE_NOT_CANCEL));
                    }
                } else { // 批量更新
                    if (queryPortalByMenuTempId(updateMenuTemplateReq.getIdList())) {
                        return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.MENU_TEMPLATE_NOT_CANCEL_BATCH));
                    }
                }
            }

            // 娜娜销售包下，只允许有一个菜单模板处于发布状态，发布时，需要同时将其他已发布模板更改为未发布状态
            if (isOnlyAsa && Constants.VALID_STATUS_ENABLE.equals(updateMenuTemplateReq.getTemplateStatus())) {
                customizedMenuTemplateService.unusableOtherTemplate(updateMenuTemplateReq, tenantId);
            }

            return ResponseEntityWrapperUtil.wrapperOk(customizedMenuTemplateService.updateStatus(updateMenuTemplateReq));
        } catch (Exception e) {
            log.error("CustomizedMenuTemplateController update menu template status error. param:{}", updateMenuTemplateReq, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/customized/template/updateMenuTemplateStatus, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 删除模板
     *
     * @param deleteMenuTemplateReq 入参
     * @return 删除是否成功
     */
    @PostMapping("/deleteMenuTemplate")
    public ResponseEntity<BaseResultDTO<Boolean>> deleteTemplate(@RequestBody DeleteMenuTemplateReq deleteMenuTemplateReq) {
        try {
            if (CollectionUtils.isEmpty(deleteMenuTemplateReq.getIdList())) {
                return ResponseEntityWrapperUtil.wrapperOk(true);
            }
            // ISV菜单，报错提示
            Optional<CustomizedMenuTemplate> optional = customizedMenuTemplateService.queryTemplateByIds(deleteMenuTemplateReq.getIdList()).stream()
                    .filter(template -> PublishSourceEnum.ISV.name().equals(template.getPublishSource()))
                    .findAny();
            if (optional.isPresent()) {
                String error = messageUtils.getMessage("error.unsupported.edit.isv.menu");
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, error);
            }
            String tenantId = AppAuthContextHolder.getContext().getAuthoredUser().getTenantId();
            boolean isOnlyAsa = tenantAppService.isOnlyAsaSalesPlan(tenantId);

            // 娜娜销售包下，用户没有门户概念，不做相关校验
            if (!isOnlyAsa) {
                // 判断是否关联了门户
                if (deleteMenuTemplateReq.getIdList().size() == Constants.NUMBER_ONE) {
                    if (queryPortalByMenuTempId(deleteMenuTemplateReq.getIdList())) {
                        return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.MENU_TEMPLATE_NOT_DELETE));
                    }
                } else {
                    if (queryPortalByMenuTempId(deleteMenuTemplateReq.getIdList())) {
                        return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.MENU_TEMPLATE_NOT_DELETE_BATCH));
                    }
                }
            }
            return ResponseEntityWrapperUtil.wrapperOk(customizedMenuTemplateService.deleteTemplateById(deleteMenuTemplateReq));
        } catch (Exception e) {
            log.error("CustomizedMenuTemplateController delete menu template error. param:{}", deleteMenuTemplateReq, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/customized/template/deleteMenuTemplate, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 查询菜单模板关联的门户
     *
     * @param menuTempIdList 菜单模板id
     * @return 返回
     */
    private boolean queryPortalByMenuTempId(List<Long> menuTempIdList) {
        List<PortalInfo> portalInfoList = portalInfoService.queryPortalByMenuTempId(menuTempIdList);
        if (CollectionUtils.isNotEmpty(portalInfoList)) {
            List<Long> idList = portalInfoList.stream().map(PortalInfo::getMenuTemplateId).distinct().collect(Collectors.toList());
            return menuTempIdList.size() == idList.size();
        }
        return false;
    }

    /**
     * 门户配置管理页面查询菜单模板列表
     *
     * @return 查询结果
     */
    @PostMapping("/queryPortalTempList")
    public ResultPageBean queryPortalTempList(@RequestBody @Valid QueryMenuTemplateReq queryMenuTemplateReq) {
        try {
            // 修改门户时有门户id
            Long portalMenuTempId = null;
            if (ObjectUtils.isNotEmpty(queryMenuTemplateReq.getPortalId())) {
                PortalInfo tempPortal = portalInfoService.getTempPortal(queryMenuTemplateReq.getPortalId());
                if (ObjectUtils.isEmpty(tempPortal)) {
                    return ResultPageBean.sysException(new ServiceException(ErrorCodeConstant.PARAM_MISSING_ERROR, "portal is not exist"));
                }
                portalMenuTempId = tempPortal.getMenuTemplateId();
            }
            return customizedMenuTemplateService.queryPortalTempList(queryMenuTemplateReq, portalMenuTempId);
        } catch (Exception e) {
            log.error("CustomizedMenuTemplateController query portal temp list error. param:{}", queryMenuTemplateReq, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/customized/template/queryPortalTemplateList, message:" + e.getMessage());
            return ResultPageBean.sysException(new ServiceException(ErrorCodeConstant.SYSTEM_ERROR, error));
        }
    }

    /**
     * 校验模板名称是否存在
     *
     * @param map 模板名称
     * @return 返回
     */
    @PostMapping("/validMenuTempName")
    public ResponseEntity<BaseResultDTO<Boolean>> validMenuTempName(@RequestBody Map<String, String> map) {
        String templateName = map.get("templateName");
        try {
            if (StringUtils.isBlank(templateName)) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "templateName can not be null");
            }
            Boolean existFlag = customizedMenuTemplateService.validExist(null, templateName);
            return ResponseEntityWrapperUtil.wrapperOk(existFlag);
        } catch (Exception e) {
            log.error("CustomizedMenuWorkController valid menu temp name error. templateName:{}", templateName, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/customized/menu/validMenuTempName, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 查询当前用户使用的菜单类型（通用菜单还是自定义菜单）
     *
     * @return 返回
     */
    @GetMapping("/queryUserMenuFlag")
    public ResponseEntity<BaseResultDTO<Integer>> queryUserMenuFlag() {
        try {
            List<PortalUserAuthResp> portalUserAuthRespList = portalInfoService.selectUserPortal(Constants.ClientTypeEnum.PC.getValue());
            List<PortalUserAuthResp> portalUserList = portalUserAuthRespList.stream().filter(x -> Boolean.TRUE.equals(x.getIsRecently())).collect(Collectors.toList());
            if (CollectionUtils.isEmpty(portalUserList)) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage("error.message.use.portal.recovery"));
            }
//            if (Constants.MenuTypeSourceEnum.ISV.getFlag().equals(portalUserList.get(0).getMenuTypeSource())) {
//                // isv菜单 -- 非通用菜单
//                return ResponseEntityWrapperUtil.wrapperOk(Constants.GENERAL_MENU_FLAG_NO);
//            }
            Long menuTemplateId = portalUserList.get(0).getMenuTemplateId();
            // 判断门户关联的菜单模板
            if (Constants.GENERAL_MENU_TEMPLATE_ID.equals(menuTemplateId)) {
                return ResponseEntityWrapperUtil.wrapperOk(Constants.GENERAL_MENU_FLAG_YES);
            }
//            CustomizedMenuTemplate template = customizedMenuTemplateService.queryTemplateById(menuTemplateId);
//            if (ObjectUtils.isEmpty(template)) {
//                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "template is not exist");
//            }
//            return ResponseEntityWrapperUtil.wrapperOk(template.getGeneralFlag());
            return ResponseEntityWrapperUtil.wrapperOk(Constants.GENERAL_MENU_FLAG_NO);
        } catch (Exception e) {
            log.error("CustomizedMenuWorkController query user menu flag error.", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/customized/menu/queryUserMenuFlag, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 自定义菜单列表配置查询菜单模板列表
     *
     * @return 查询结果
     */
    @GetMapping("/queryEnableMenuTemplateList")
    public ResponseEntity<BaseResultDTO<List<CustomizedMenuTemplate>>> queryEnableMenuTemplateList() {
        try {
            List<CustomizedMenuTemplate> templateRespList = customizedMenuTemplateService.queryEnableMenuTemplateList();
            return ResponseEntityWrapperUtil.wrapperOk(templateRespList);
        } catch (Exception e) {
            log.error("CustomizedMenuTemplateController query menu template list error.", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/customized/template/queryEnableMenuTemplateList, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }
}