package com.digiwin.athena.semc.controller.menu;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.appcore.util.SnowflakeIdWorker;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.I18NKey;
import com.digiwin.athena.semc.dto.menu.SaveMenuReq;
import com.digiwin.athena.semc.entity.menu.CustomizedMenuTemplate;
import com.digiwin.athena.semc.entity.menu.CustomizedMenuWork;
import com.digiwin.athena.semc.entity.portal.LabelSystemCustom;
import com.digiwin.athena.semc.entity.portal.PortalInfo;
import com.digiwin.athena.semc.proxy.athena.service.audc.AudcService;
import com.digiwin.athena.semc.proxy.athena.service.audc.model.SystemMenuDTO;
import com.digiwin.athena.semc.proxy.km.service.KmService;
import com.digiwin.athena.semc.proxy.km.service.model.AppJobDTO;
import com.digiwin.athena.semc.service.menu.CustomizedMenuTemplateService;
import com.digiwin.athena.semc.service.menu.CustomizedMenuWorkService;
import com.digiwin.athena.semc.service.portal.LabelSystemCustomService;
import com.digiwin.athena.semc.service.portal.PortalInfoService;
import com.digiwin.athena.semc.service.workbench.constant.PublishSourceEnum;
import com.digiwin.athena.semc.util.InterceptorIgnoreUtil;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;
import com.digiwin.athena.semc.vo.menu.CustomizedMenuWorkResp;
import com.digiwin.athena.semc.vo.menu.ValidCustomMenuRelVO;
import com.digiwin.athena.semc.vo.portal.LabelSystemAllVo;
import com.google.common.collect.Lists;
import io.github.linpeilie.Converter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.validation.Valid;
import java.util.*;
import java.util.stream.Collectors;

import static com.digiwin.athena.semc.common.Constants.SYSTEM_MENU_FLAG_YES;

/**
 * 自定义菜单作业表(TCustomizedMenuWork)表控制层
 *
 * @author sungqz
 * @since 2023-11-10
 */
@Slf4j
@RestController
@RequestMapping("/semc/customized/menu")
public class CustomizedMenuWorkController {

    /**
     * 服务对象
     */
    @Resource
    private CustomizedMenuWorkService customizedMenuWorkService;

    @Resource
    private CustomizedMenuTemplateService customizedMenuTemplateService;

    @Resource
    private LabelSystemCustomService labelSystemCustomService;

    @Resource
    private PortalInfoService portalInfoService;

    @Resource
    private AudcService audcService;

    @Autowired
    private KmService kmService;

    @Resource
    private MessageUtils messageUtils;

    @Resource
    private Converter converter;

    /**
     * 模板数量上限
     */
    public static final int MENU_TEMPLATE_COUNT_MAX = 99;

    /**
     * 菜单/作业名称的长度限制
     */
    public static final int MENU_NAME_LENGTH_MAX = 50;

    /**
     * 菜单数量限制
     */
    public static final int MENU_COUNT_MAX = 50;

    /**
     * 查询管理后台模板的菜单树列表
     *
     * @param templateId 模板id
     * @return 查询结果
     */
    @GetMapping("/queryManageMenu")
    public ResponseEntity<BaseResultDTO<JSONObject>> queryManageMenu(@RequestParam Long templateId, @RequestParam Integer portalTypeSource) {
        CustomizedMenuTemplate template = null;
        if (Constants.LabelOrPoralSourceEnum.TENANT.getFlag().equals(portalTypeSource)) {
            template = customizedMenuTemplateService.queryTemplateById(templateId);
        } else if (Constants.LabelOrPoralSourceEnum.ISV.getFlag().equals(portalTypeSource)) {
            LambdaQueryWrapper<CustomizedMenuTemplate> wrapper = new LambdaQueryWrapper<>();
            String tenantVersion = kmService.queryTenantVersion();
            wrapper.eq(CustomizedMenuTemplate::getMenuDesignerId, templateId);
            wrapper.eq(CustomizedMenuTemplate::getVersion, tenantVersion);
            wrapper.eq(CustomizedMenuTemplate::getPublishSource, PublishSourceEnum.ISV.name());
            template = InterceptorIgnoreUtil.handler(() -> customizedMenuTemplateService.getBaseMapper().selectOne(wrapper));
        }
        if (ObjectUtils.isEmpty(template)) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, "模板不存在");
        }

        // 查询自定义菜单树
        JSONObject jsonObject = new JSONObject();
        List<CustomizedMenuWorkResp> menuWorkRespList = customizedMenuWorkService.queryManageMenu(template.getId());
        jsonObject.put("customList", menuWorkRespList);

        // 展示系统菜单
        List<SystemMenuDTO> systemMenuDTOList = Lists.newArrayList();
        if (SYSTEM_MENU_FLAG_YES.equals(template.getSystemMenuFlag())) {
            systemMenuDTOList = audcService.querySystemMenu();
        }
        jsonObject.put("systemList", systemMenuDTOList);
        return ResponseEntityWrapperUtil.wrapperOk(jsonObject);
    }

    /**
     * 校验自定义菜单是否关联停用的自定义组件
     *
     * @param saveMenuReq 实体
     * @return 新增结果
     */
    @PostMapping("/validCustomMenuRel")
    public ResponseEntity<BaseResultDTO<ValidCustomMenuRelVO>> validCustomMenuRel(@RequestBody @Valid SaveMenuReq saveMenuReq) {
        ValidCustomMenuRelVO validCustomMenuRelVO = new ValidCustomMenuRelVO();
        if (CollectionUtils.isEmpty(saveMenuReq.getMenuList())) {
            return ResponseEntityWrapperUtil.wrapperOk(validCustomMenuRelVO);
        }

        // 判断是否存在停用的自定义组件
        List<LabelSystemCustom> disabledList = queryDisabledListByIdList(saveMenuReq.getMenuList());
        if (CollectionUtils.isNotEmpty(disabledList)) {
            List<String> labelNameList = disabledList.stream().map(LabelSystemCustom::getNameZh).collect(Collectors.toList());
            validCustomMenuRelVO.setSuccessFlag(false);
            validCustomMenuRelVO.setDisabledCustomList(labelNameList);
        }
        return ResponseEntityWrapperUtil.wrapperOk(validCustomMenuRelVO);
    }

    /**
     * 过滤出自定义组件菜单
     *
     * @param menu                菜单对象
     * @param customLabelMenuList 自定义组件菜单列表
     */
    private void filterCustomLabelList(CustomizedMenuWorkResp menu, List<CustomizedMenuWork> customLabelMenuList) {
        if (Constants.CustomizedMenuTypeEnum.LABEL.getType().equals(menu.getMenuType())
                && Constants.LabelTypeEnum.SYSTEM_CUSTOM.getVal().equals(menu.getMenuLabelType())) {
            CustomizedMenuWork menuWork = converter.convert(menu, CustomizedMenuWork.class);
            customLabelMenuList.add(menuWork);
        }
        // 遍历子节点
        if (CollectionUtils.isEmpty(menu.getChildren())) {
            return;
        }
        for (CustomizedMenuWorkResp child : menu.getChildren()) {
            filterCustomLabelList(child, customLabelMenuList);
        }
    }

    /**
     * 判断是否存在停用的自定义组件
     *
     * @param menuTreeReqList 菜单数据
     * @return 返回
     */
    private List<LabelSystemCustom> queryDisabledListByIdList(List<CustomizedMenuWorkResp> menuTreeReqList) {
        List<LabelSystemCustom> labelSystemCustomList = Lists.newArrayList();
        if (CollectionUtils.isEmpty(menuTreeReqList)) {
            return labelSystemCustomList;
        }

        // 过滤出所有自定义组件，校验是否存在停用的组件
        List<CustomizedMenuWork> customLabelMenuList = Lists.newArrayList();
        for (CustomizedMenuWorkResp menu : menuTreeReqList) {
            filterCustomLabelList(menu, customLabelMenuList);
        }
        if (CollectionUtils.isEmpty(customLabelMenuList)) {
            return labelSystemCustomList;
        }
        // isv部件，不可操作, 过滤掉
        List<Long> labelIdList = customLabelMenuList.stream().filter(x -> Constants.LabelTypeSourceEnum.TENANT.getFlag().equals(x.getLabelTypeSource())).map(CustomizedMenuWork::getMenuLabelId).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(labelIdList)) {
            return labelSystemCustomList;
        }
        LabelSystemAllVo req = new LabelSystemAllVo();
        req.setIdList(labelIdList);
        req.setValidStatus(Constants.VALID_STATUS_UNUSABLE);
        return labelSystemCustomService.queryCustomList(req);
    }

    /**
     * 保存菜单树
     *
     * @param saveMenuReq 实体
     * @return 新增结果
     */
    @PostMapping("/saveMenu")
    public ResponseEntity<BaseResultDTO<Boolean>> saveMenuList(@RequestBody @Valid SaveMenuReq saveMenuReq) {
        // 新增模板，需判断模板数量是否超过100个
        if (ObjectUtils.isEmpty(saveMenuReq.getTemplateId())) {
            long count = customizedMenuTemplateService.count();
            if (count > MENU_TEMPLATE_COUNT_MAX) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.MENU_TEMPLATE_COUNT_OVER_LIMIT));
            }
        }
        // 判断菜单模板名称是否重复
        Boolean existFlag = customizedMenuTemplateService.validExist(saveMenuReq.getTemplateId(), saveMenuReq.getTemplateName());
        if (existFlag) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.MENU_TEMPLATE_NAME_DUPLICATE));
        }
        // ISV菜单，不可修改，报错提示
        if (saveMenuReq.getTemplateId() != null) {
            boolean isIsv = Optional.ofNullable(customizedMenuTemplateService.queryTemplateById(saveMenuReq.getTemplateId()))
                    .map(template -> PublishSourceEnum.ISV.name().equals(template.getPublishSource()))
                    .orElse(false);
            if (isIsv) {
                String error = messageUtils.getMessage("error.unsupported.edit.isv.menu");
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, error);
            }
        }

        // 遍历菜单树
        List<CustomizedMenuWork> parallelMenuList = Lists.newArrayList();
        List<CustomizedMenuWorkResp> menuTreeReqList = saveMenuReq.getMenuList();
        for (CustomizedMenuWorkResp menu : menuTreeReqList) {
            traverseTree(menu, parallelMenuList, saveMenuReq.getTemplateId());
        }

        // 判断是否存在停用的自定义组件，过滤掉停用的组件
        List<LabelSystemCustom> disabledList = queryDisabledListByIdList(menuTreeReqList);
        List<CustomizedMenuWork> finalMenuList = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(disabledList)) {
            List<Long> labelIdList = disabledList.stream().map(LabelSystemCustom::getId).collect(Collectors.toList());
            for (CustomizedMenuWork menuWork : parallelMenuList) {
                if (!Constants.LabelTypeEnum.SYSTEM_CUSTOM.getVal().equals(menuWork.getMenuLabelType())) {
                    finalMenuList.add(menuWork);
                    continue;
                }
                if (!labelIdList.contains(menuWork.getMenuLabelId())) {
                    finalMenuList.add(menuWork);
                }
            }
        } else {
            finalMenuList.addAll(parallelMenuList);
        }

        // 校验参数
        ResponseEntity<BaseResultDTO<Boolean>> responseEntity = validSaveMenuParam(finalMenuList, menuTreeReqList.size());
        if (ObjectUtils.isNotEmpty(responseEntity)) {
            return responseEntity;
        }
        // 保存菜单
        Boolean flag = customizedMenuWorkService.saveMenu(saveMenuReq, finalMenuList);
        return ResponseEntityWrapperUtil.wrapperOk(flag);
    }

    /**
     * 参数校验
     *
     * @param menuList  菜单列表
     * @param menuCount 菜单数量
     * @return 返回
     */
    private ResponseEntity<BaseResultDTO<Boolean>> validSaveMenuParam(List<CustomizedMenuWork> menuList, Integer menuCount) {
        if (CollectionUtils.isEmpty(menuList)) {
            return null;
        }
        // 菜单层级不能超过三级
        boolean validLevel = menuList.stream().anyMatch(x -> Constants.CustomizedMenuLevelEnum.THIRD.getLevel() < x.getLevel());
        if (validLevel) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "The menu level cannot exceed three levels");
        }
        // 菜单名称不能为空
        boolean validNameNull = menuList.stream().anyMatch(x -> StringUtils.isBlank(x.getMenuName()));
        if (validNameNull) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.MENU_NAME_CANNOT_BE_NULL));
        }
        // 菜单/作业名称长度不能超过40个字符
        boolean validNameLength = menuList.stream().anyMatch(x -> x.getMenuName().length() > MENU_NAME_LENGTH_MAX);
        if (validNameLength) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "The menu name cannot exceed 50 characters");
        }
        // 菜单数量不能超过50个
        if (menuCount > MENU_COUNT_MAX) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "The menu count cannot exceed 50");
        }
        Map<Long, Long> parentIdMap = menuList.stream().filter(x -> x.getParentId() != null).collect(Collectors.groupingBy(CustomizedMenuWork::getParentId, Collectors.counting()));
        for (Map.Entry<Long, Long> parentIdEntry : parentIdMap.entrySet()) {
            if (parentIdEntry.getValue() > MENU_COUNT_MAX) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "The menu count cannot exceed 50");
            }
        }

        // 菜单之间不能重名
        List<String> menuNameList = menuList.stream().filter(x -> Constants.CustomizedMenuTypeEnum.MENU.getType().equals(x.getMenuType())).map(CustomizedMenuWork::getMenuName).collect(Collectors.toList());
        HashSet<String> disMenuNameSet = new HashSet<>(menuNameList);
        if (menuNameList.size() != disMenuNameSet.size()) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "The menu cannot have duplicate names");
        }
        // 菜单和作业之间不能重名
        List<String> workNameList = menuList.stream().filter(x -> Arrays.asList(Constants.CustomizedMenuTypeEnum.WORK.getType(),
                Constants.CustomizedMenuTypeEnum.LABEL.getType()).contains(x.getMenuType())).map(CustomizedMenuWork::getMenuName).collect(Collectors.toList());
        boolean disjoint = Collections.disjoint(menuNameList, workNameList);
        if (!disjoint) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "The menu and work cannot have duplicate names");
        }
        // 同一个作业不能多次放在同一个菜单下
        for (CustomizedMenuWork menu : menuList) {
            if (Arrays.asList(Constants.CustomizedMenuTypeEnum.WORK.getType(), Constants.CustomizedMenuTypeEnum.LABEL.getType()).contains(menu.getMenuType())) {
                continue;
            }
            List<String> workList = menuList.stream().filter(x -> Arrays.asList(Constants.CustomizedMenuTypeEnum.WORK.getType(),
                    Constants.CustomizedMenuTypeEnum.LABEL.getType()).contains(x.getMenuType()) && ObjectUtils.isNotEmpty(x.getParentId()) &&
                    x.getParentId().equals(menu.getId())).map(CustomizedMenuWork::getMenuName).collect(Collectors.toList());
            HashSet<String> disWorkNameSet = new HashSet<>(workList);
            if (workList.size() != disWorkNameSet.size()) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "The same job cannot be placed multiple times under the same menu");
            }
        }
        // 作业不允许还有下级作业/菜单
        for (CustomizedMenuWork menu : menuList) {
            if (Constants.CustomizedMenuTypeEnum.MENU.getType().equals(menu.getMenuType())) {
                continue;
            }
            boolean validWorkChild = menuList.stream().anyMatch(x -> x.getParentId() != null && x.getParentId().equals(menu.getId()));
            if (validWorkChild) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "The job cannot have sub level job or menu");
            }
        }
        return null;
    }

    /**
     * 设置菜单父子级关系
     *
     * @param menu             菜单对象
     * @param parallelMenuList 菜单列表
     */
    private void traverseTree(CustomizedMenuWorkResp menu, List<CustomizedMenuWork> parallelMenuList, Long templateId) {
        // 如果模板id为空，则说明是新菜单（复制操作，前端会传入原模板中的菜单id，需要置空）
        if (ObjectUtils.isEmpty(templateId) && ObjectUtils.isNotEmpty(menu.getId())) {
            menu.setId(null);
        }
        Long id = menu.getId();
        if (id == null) {
            id = SnowflakeIdWorker.getInstance().newId();
            menu.setId(id);
            // id为空，则初始化作业名称，此处预留字段，暂时没有用
            // menu.setOriginalName(menu.getOriginalName());
        }
        CustomizedMenuWork menuWork = converter.convert(menu, CustomizedMenuWork.class);
        parallelMenuList.add(menuWork);
        // 遍历子节点
        if (CollectionUtils.isEmpty(menu.getChildren())) {
            return;
        }
        for (CustomizedMenuWorkResp child : menu.getChildren()) {
            child.setParentId(id);
            child.setParentName(menu.getMenuName());
            traverseTree(child, parallelMenuList, templateId);
        }
    }

    /**
     * 查询前台用户展示的菜单树
     *
     * @param templateId 模板id
     * @param portalId   门户id
     * @return 新增结果
     */
    @GetMapping("/queryDisplayMenu")
    public ResponseEntity<BaseResultDTO<JSONObject>> queryDisplayMenu(@RequestParam(required = false) Long templateId,
                                                                      @RequestParam(required = false) Long portalId,
                                                                      @RequestParam Integer portalTypeSource
    ) {
        if (ObjectUtils.isEmpty(templateId) && ObjectUtils.isEmpty(portalId)) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "param is not exist");
        }
        Integer menuTypeSource = Constants.SourceEnum.TENANT.getFlag();
        // 如果菜单模板id为空，则根据门户id查询门户配置中的菜单模板id
        if (ObjectUtils.isEmpty(templateId)) {
            PortalInfo tempPortal = null;
            if (Constants.PortalTypeSourceEnum.ISV.getFlag().equals(portalTypeSource)) {
                tempPortal = portalInfoService.queryPortalInfoByDesignerId(portalId);
            } else {
                tempPortal = portalInfoService.getTempPortal(portalId);
            }

            if (ObjectUtils.isEmpty(tempPortal)) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "portal is not exist, portalId:" + portalId);
            }
            templateId = tempPortal.getMenuTemplateId();
            menuTypeSource = tempPortal.getMenuTypeSource();
        }

        // 如果是通用菜单则直接返回，前端渲染
        JSONObject jsonObject = new JSONObject();
        if (Constants.GENERAL_MENU_TEMPLATE_ID.equals(templateId)) {
            jsonObject.put("generalFlag", Constants.GENERAL_MENU_FLAG_YES);
            return ResponseEntityWrapperUtil.wrapperOk(jsonObject);
        }

        // 自定义菜单
        CustomizedMenuTemplate template = null;
        if (Constants.SourceEnum.ISV.getFlag().equals(menuTypeSource)) {
            template = customizedMenuTemplateService.queryTemplateByDesignerId(templateId);
        } else {
            template = customizedMenuTemplateService.queryTemplateById(templateId);
        }
        if (ObjectUtils.isEmpty(template)) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "template is not exist, templateId:" + templateId + ", portalId:" + portalId);
        }
        jsonObject.put("generalFlag", template.getGeneralFlag());
        JSONObject menuList = customizedMenuWorkService.queryDisplayMenu(template.getId(), template.getSystemMenuFlag());
        jsonObject.putAll(menuList);
        return ResponseEntityWrapperUtil.wrapperOk(jsonObject);
    }

    /**
     * 查询前台用户展示的菜单树
     *
     * @param templateId 模板id
     * @return 新增结果
     */
    @GetMapping("/queryNoPortalDisplayMenu")
    public ResponseEntity<BaseResultDTO<JSONObject>> queryNoPortalDisplayMenu(@RequestParam(required = false) Long templateId) {
        CustomizedMenuTemplate template = null;

        if (ObjectUtils.isEmpty(templateId)) {
            List<CustomizedMenuTemplate> customizedMenuTemplate = customizedMenuTemplateService.queryEnableMenuTemplateList();

            if (CollectionUtils.isNotEmpty(customizedMenuTemplate)) {
                template = customizedMenuTemplate.get(0);
                templateId = template.getId();
            } else {
                // 娜娜销售包下，没有通用菜单模板，因此初始化时查不到任何菜单模板数据，直接返回成功
                return ResponseEntityWrapperUtil.wrapperOk();
            }
        }

        // 如果是通用菜单则直接返回，前端渲染
        JSONObject jsonObject = new JSONObject();
        if (Constants.GENERAL_MENU_TEMPLATE_ID.equals(templateId)) {
            jsonObject.put("generalFlag", Constants.GENERAL_MENU_FLAG_YES);
            return ResponseEntityWrapperUtil.wrapperOk(jsonObject);
        }

        // 自定义菜单
        if (ObjectUtils.isEmpty(template)) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "template is not exist, templateId:" + templateId);
        }
        jsonObject.put("generalFlag", template.getGeneralFlag());
        JSONObject menuList = customizedMenuWorkService.queryDisplayMenu(templateId, template.getSystemMenuFlag());
        jsonObject.putAll(menuList);
        return ResponseEntityWrapperUtil.wrapperOk(jsonObject);
    }

    /**
     * 查询作业列表
     *
     * @return 查询结果
     */
    @GetMapping("/queryWorkList")
    public ResponseEntity<BaseResultDTO<List<AppJobDTO>>> queryWorkList() {
        List<AppJobDTO> jobList = customizedMenuWorkService.queryAppJobList();
        return ResponseEntityWrapperUtil.wrapperOk(jobList);
    }

    /**
     * 查询所有系统菜单
     *
     * @return 查询结果
     */
    @GetMapping("/querySystemMenuList")
    public ResponseEntity<BaseResultDTO<List<SystemMenuDTO>>> querySystemMenuList() {
        List<SystemMenuDTO> systemMenuDTOList = audcService.querySystemMenu();
        return ResponseEntityWrapperUtil.wrapperOk(systemMenuDTOList);
    }

    /**
     * 数据初始化：为租户来源(TENANT)的菜单模板补齐缺失的“待办”预设组件
     * 路径：/semc/customized/menu/initData
     */
    @GetMapping("/initData")
    public ResponseEntity<BaseResultDTO<Boolean>> initCustomizedMenuData() {
        Boolean result = customizedMenuWorkService.initData();
        return ResponseEntityWrapperUtil.wrapperOk(result);
    }
}