package com.digiwin.athena.semc.controller.menu.manage;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.I18NKey;
import com.digiwin.athena.semc.dto.menu.manage.SaveMenuAuthReq;
import com.digiwin.athena.semc.entity.menu.manage.ManageMenu;
import com.digiwin.athena.semc.entity.menu.manage.ManageMenuAuth;
import com.digiwin.athena.semc.env.EnvProperties;
import com.digiwin.athena.semc.service.menu.ManageMenuAuthService;
import com.digiwin.athena.semc.service.menu.ManageMenuService;
import com.digiwin.athena.semc.service.menu.AuthService;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;
import com.digiwin.athena.semc.util.SecurityLogUtil;
import com.digiwin.athena.semc.util.Utils;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.validation.Valid;
import java.time.LocalDateTime;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 管理后台菜单权限配置
 *
 * @author sungqz
 * @since 2024-08-26
 */
@Slf4j
@RestController
@RequestMapping("/manage/menu/auth")
public class ManageMenuAuthController {

    @Resource
    private ManageMenuAuthService manageMenuAuthService;

    @Resource
    private ManageMenuService manageMenuService;

    @Resource
    private MessageUtils messageUtils;

    @Resource
    EnvProperties envProperties;

    @Resource
    private AuthService authService;


    /**
     * 管理后台-查询权限对象关联的菜单
     *
     * @return 查询结果
     */
    @GetMapping("/queryAuthRelMenu")
    public ResponseEntity<BaseResultDTO<JSONObject>> queryAuthRelMenu(@RequestParam(value = "authId") String authId) {
        JSONObject resp = new JSONObject();
        List<ManageMenuAuth> menuAuthList = manageMenuAuthService.queryAuthRelMenu(authId);
        List<String> menuKeyList = menuAuthList.stream().map(ManageMenuAuth::getMenuKey).collect(Collectors.toList());
        resp.put("menuAuthList", menuKeyList);
        if (CollectionUtils.isEmpty(menuAuthList)) {
            resp.put("allMenuAuthFlag", Constants.ALL_AUTH_FLAG_NO);
            return ResponseEntityWrapperUtil.wrapperOk(resp);
        }
        boolean flag = menuAuthList.stream().anyMatch(x -> Constants.ALL_AUTH_FLAG_YES.equals(x.getAllMenuAuthFlag()));
        resp.put("allMenuAuthFlag", flag ? Constants.ALL_AUTH_FLAG_YES : Constants.ALL_AUTH_FLAG_NO);
        return ResponseEntityWrapperUtil.wrapperOk(resp);
    }

    /**
     * 管理后台-保存权限对象关联的菜单
     *
     * @return 查询结果
     */
    @PostMapping("/saveAuthRelMenu")
    public ResponseEntity<BaseResultDTO<Boolean>> saveAuthRelMenu(@RequestBody @Valid SaveMenuAuthReq saveMenuAuthReq, HttpServletRequest request) {
        try {
            //用户及权限配置/工作台权限配置/菜单权限配置（权限判断）
            if (authService.hasAuth("permission-menu")) {
                manageMenuAuthService.saveAuthRelMenu(saveMenuAuthReq);
                return ResponseEntityWrapperUtil.wrapperOk(true);
            }
            else {
                //写入安全日志
                SecurityLogUtil.writeSecurityLog(request, JSON.toJSONString(saveMenuAuthReq));
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "无权限");
            }

        } catch (Exception e) {
            log.error("ManageMenuAuthController save auth rel menu error. param:{}", saveMenuAuthReq, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/manage/menu/auth/saveAuthRelMenu, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 查询当前登录用户关联的菜单
     *
     * @param environmentFlag 所属环境标识 0:全部 1:云上 2:下地
     * @return 查询结果
     */
    @GetMapping("/queryUserMenuList")
    public ResponseEntity<BaseResultDTO<List<String>>> queryUserMenuList(@RequestParam Integer environmentFlag) {
        // 查询全部菜单
        String version = manageMenuService.getTenantVersion();
        List<ManageMenu> manageMenuList = manageMenuService.queryBaseManageMenu(environmentFlag, true, version);
        // 特殊账号的菜单，即ISV菜单
        List<String> specialMenuKeyList = manageMenuList.stream().filter(x -> StringUtils.isNotBlank(x.getTenantId()) && x.getTenantId().equals(Utils.getTenantId())
                && x.getBelongingUsers().contains(Utils.getUserId())).map(ManageMenu::getMenuKey).collect(Collectors.toList());

        // 查询当前用户有权限的菜单
        List<ManageMenuAuth> menuAuthList = manageMenuAuthService.queryUserMenuList();
        // 如果权限配置为空，则只返回特殊账号的菜单，即ISV菜单
        if (CollectionUtils.isEmpty(menuAuthList)) {
            return ResponseEntityWrapperUtil.wrapperOk(specialMenuKeyList);
        }
        // 如果权限配置不为空，则判断是否有全部菜单权限
        List<String> menuKeyList;
        boolean allFlag = menuAuthList.stream().anyMatch(x -> Constants.ALL_AUTH_FLAG_YES.equals(x.getAllMenuAuthFlag()));
        if (allFlag) {
            menuKeyList = manageMenuList.stream().filter(x -> StringUtils.isBlank(x.getTenantId())).map(ManageMenu::getMenuKey).collect(Collectors.toList());
        } else {
            menuKeyList = menuAuthList.stream().map(ManageMenuAuth::getMenuKey).collect(Collectors.toList());
        }
        menuKeyList.addAll(specialMenuKeyList);
        return ResponseEntityWrapperUtil.wrapperOk(menuKeyList);
    }

    /**
     * 获取特殊账号，获取88888888租户和泽城账号
     *
     * @return 查询结果
     */
    @GetMapping("/querySpecialUser")
    public ResponseEntity<BaseResultDTO<JSONObject>> querySpecialUser() {
        try {
            String specialTenantId = envProperties.getTemplateTenantId();
            String specialUserId = envProperties.getTemplateUserId();
            JSONObject jsonObject = new JSONObject();
            jsonObject.put("specialTenantId", specialTenantId);
            jsonObject.put("specialUserId", specialUserId);
            return ResponseEntityWrapperUtil.wrapperOk(jsonObject);
        } catch (Exception e) {
            log.error("ManageMenuAuthController query special user error.", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/manage/menu/auth/querySpecialUser, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }
}