package com.digiwin.athena.semc.controller.menu.manage;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.CollectionUtils;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.I18NKey;
import com.digiwin.athena.semc.dto.menu.manage.DelMenuListReq;
import com.digiwin.athena.semc.dto.menu.manage.SaveSpecialTenantIdReq;
import com.digiwin.athena.semc.entity.menu.manage.ManageMenu;
import com.digiwin.athena.semc.service.cache.CommonConfigService;
import com.digiwin.athena.semc.service.menu.ManageMenuService;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;

import com.digiwin.athena.semc.vo.menu.UserManageMenulVO;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.time.LocalDateTime;
import java.util.List;

import javax.annotation.Resource;
import javax.validation.Valid;

import lombok.extern.slf4j.Slf4j;

/**
 * 管理后台基础菜单配置表(TManageMenu)表控制层
 *
 * @author sungqz
 * @since 2024-09-30
 */
@Slf4j
@RestController
@RequestMapping("/manage/menu")
public class ManageMenuController {

    @Resource
    private ManageMenuService manageMenuService;

    @Resource
    private MessageUtils messageUtils;

    @Resource
    private CommonConfigService commonConfigService;

    /**
     * 查询基础菜单配置列表数据
     *
     * @return 查询结果
     */
    @GetMapping("/queryManageMenuConfigList")
    public ResponseEntity<BaseResultDTO<List<ManageMenu>>> queryManageMenuConfigList() {
        try {
            List<ManageMenu> manageMenuList = manageMenuService.queryManageMenuConfigList();
            return ResponseEntityWrapperUtil.wrapperOk(manageMenuList);
        } catch (Exception e) {
            log.error("ManageMenuController query manage menu page error.", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/manage/menu/queryManageMenuConfigList, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 新增/编辑基础菜单
     *
     * @param manageMenu 实体
     * @return 新增结果
     */
    @PostMapping("/saveManageMenu")
    public ResponseEntity<BaseResultDTO<Boolean>> saveManageMenu(@RequestBody @Valid ManageMenu manageMenu) {
        try {
            if (Boolean.FALSE.equals(commonConfigService.getBackendMenuOperateFlag())) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "无权限");
            }
            if (manageMenu.getOnlyPreArea()==null) {
                manageMenu.setOnlyPreArea(0);
            }
            // menuKey和parentMenuKey不能重复
            if (StringUtils.isNotBlank(manageMenu.getParentMenuKey()) && manageMenu.getMenuKey().equals(manageMenu.getParentMenuKey())) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "menuKey and parentMenuKey cannot be duplicated");
            }
            // 校验menuKey是否重复
            List<ManageMenu> manageMenuList = manageMenuService.existMenuKey(manageMenu);
            if (CollectionUtils.isNotEmpty(manageMenuList)) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "menuKey is exist:" + manageMenu.getMenuKey());
            }
            //处理版本号
            if(StringUtils.isNotEmpty(manageMenu.getVersion())){
               Boolean isNeedEmpty= manageMenuService.handMenuVersion(manageMenu);
               //是否需要将当前菜单版本号置空。true：置空，false：不置空
               if(isNeedEmpty){
                   manageMenu.setVersion(null);
               }
            }
            if (ObjectUtils.isEmpty(manageMenu.getId())) {
                manageMenuService.insertManageMenu(manageMenu);
            } else {
                manageMenuService.updateManageMenu(manageMenu);
            }
            return ResponseEntityWrapperUtil.wrapperOk(true);
        } catch (Exception e) {
            log.error("ManageMenuController save manage menu error. param:{}", manageMenu, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/manage/menu/saveManageMenu, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }


    /**
     * 批量删除基础菜单
     *
     * @param delMenuListReq 菜单标识列表
     * @return 删除是否成功
     */
    @PostMapping("/delManageMenu")
    public ResponseEntity<BaseResultDTO<Boolean>> delManageMenu(@RequestBody @Valid DelMenuListReq delMenuListReq) {
        try {
            if (Boolean.FALSE.equals(commonConfigService.getBackendMenuOperateFlag())) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "无权限");
            }
            // 查询所有菜单列表
            List<ManageMenu> manageMenuList = manageMenuService.getBaseMapper().selectList(Wrappers.emptyWrapper());
            // 删除菜单及子菜单
            manageMenuService.deleteByMenuKeyList(delMenuListReq.getMenuKeyList(), manageMenuList);
            return ResponseEntityWrapperUtil.wrapperOk(true);
        } catch (Exception e) {
            log.error("ManageMenuController del manage menu error. param:{}", delMenuListReq.getMenuKeyList(), e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/manage/menu/delManageMenu, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 管理后台-查询权限设置页面的菜单列表数据
     *
     * @param environmentFlag 所属环境标识 0:全部 1:云上 2:下地
     * @return 查询结果
     */
    @GetMapping("/queryManageMenuList")
    public ResponseEntity<BaseResultDTO<List<ManageMenu>>> queryManageMenuList(@RequestParam Integer environmentFlag) {
        try {
            List<ManageMenu> menuList = manageMenuService.queryManageMenuList(environmentFlag);
            return ResponseEntityWrapperUtil.wrapperOk(menuList);
        } catch (Exception e) {
            log.error("ManageMenuController query manage menu list error. environmentFlag:{}", environmentFlag, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/manage/menu/queryManageMenuList, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 管理后台-查询用户的左侧菜单树
     *
     * @param environmentFlag 所属环境标识 0:全部 1:云上 2:下地
     * @return 查询结果
     */
    @GetMapping("/queryManageMenuTree")
    public ResponseEntity<BaseResultDTO<List<ManageMenu>>> queryManageMenuTree(@RequestParam Integer environmentFlag) {
        try {
            List<ManageMenu> menuList = manageMenuService.queryManageMenuTree(environmentFlag);
            return ResponseEntityWrapperUtil.wrapperOk(menuList);
        } catch (Exception e) {
            log.error("ManageMenuController query manage menu tree error. environmentFlag:{}", environmentFlag, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/manage/menu/queryManageMenuTree, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 校验当前登录用户是否有菜单权限
     *
     * @param environmentFlag 所属环境标识 0:全部 1:云上 2:下地
     * @param menuKey         菜单Key
     * @return 查询结果
     */
    @GetMapping("/validMenuAuth")
    public ResponseEntity<BaseResultDTO<Boolean>> validMenuAuth(@RequestParam Integer environmentFlag,
                                                        @RequestParam String menuKey) {
        try {
            // 先判断menuKey是否存在
            QueryWrapper<ManageMenu> condition = new QueryWrapper<>();
            condition.eq("menu_key", menuKey);
            List<ManageMenu> manageMenuList = manageMenuService.getBaseMapper().selectList(condition);
            if (CollectionUtils.isEmpty(manageMenuList)) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "the menuKey is not exist:" + menuKey);
            }
            // 校验
            boolean flag = manageMenuService.validMenuAuth(environmentFlag, manageMenuList.get(0));
            return ResponseEntityWrapperUtil.wrapperOk(flag);
        } catch (Exception e) {
            log.error("ManageMenuController valid menu auth error. environmentFlag:{}, menuKey:{}", environmentFlag, menuKey, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/manage/menu/validMenuAuth, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 设置特殊租户和账号
     *
     * @param saveSpecialTenantIdReq 请求入参
     * @return 返回
     */
    @PostMapping("/saveTenantId")
    public ResponseEntity<BaseResultDTO<Boolean>> saveTenantId(@RequestBody SaveSpecialTenantIdReq saveSpecialTenantIdReq) {
        try {
            // 根据主键id设置租户和账号
            manageMenuService.saveTenantId(saveSpecialTenantIdReq);
            return ResponseEntityWrapperUtil.wrapperOk(true);
        } catch (Exception e) {
            log.error("ManageMenuController save tenantId error. param:{}", saveSpecialTenantIdReq, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/manage/menu/saveTenantId, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }


    /**
     * 查询用户管理菜单标识
     *
     * @param req  所属环境标识 0:全部 1:云上 2:下地
     * @return 查询结果
     */
    @PostMapping("/queryUserMenuFlag")
    public ResponseEntity<BaseResultDTO<UserManageMenulVO>> queryUserMenuFlag(@RequestBody  @Valid  UserManageMenulVO req) {
        try {
            return ResponseEntityWrapperUtil.wrapperOk(manageMenuService.queryUserMenuFlag(req));
        } catch (Exception e) {
            log.error("queryUserMenuFlag  environmentFlag:{}, error:{}", req.getEnvironmentFlag(), e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/manage/menu/queryUserMenuFlag, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }
}

