package com.digiwin.athena.semc.controller.message;

import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.I18NKey;
import com.digiwin.athena.semc.dto.message.SaveBlackWhiteReq;
import com.digiwin.athena.semc.entity.message.MessageBlackWhite;
import com.digiwin.athena.semc.proxy.iam.service.IamService;
import com.digiwin.athena.semc.proxy.iam.service.model.RoleDTO;
import com.digiwin.athena.semc.service.message.MessageBlackWhiteService;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;
import lombok.extern.slf4j.Slf4j;
import net.sf.json.JSONObject;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.validation.Valid;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 黑白名单(TMessageChannel)表控制层
 *
 * @author sungqz
 * @since 2024-01-10
 */
@Slf4j
@RestController
@RequestMapping("/semc/message/blackwhite")
public class MessageBlackWhiteController {

    @Resource
    private MessageBlackWhiteService messageBlackWhiteService;

    @Resource
    private MessageUtils messageUtils;

    @Resource
    private IamService iamService;

    /**
     * 查询黑白名单用户列表
     *
     * @param type 名单类型 0:黑名单 1:白名单
     * @param key  关键字
     * @return 查询结果
     */
    @GetMapping("/queryUserList")
    public ResponseEntity<BaseResultDTO<List<MessageBlackWhite>>> queryUserList(@RequestParam(required = false) Integer type,
                                                        @RequestParam(required =
            false) String key, @RequestParam(required = false) String tenantId) {
        try {
            if (type != null && !Arrays.asList(Constants.BLACK_TYPE, Constants.WHITE_TYPE).contains(type)) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "the type is is illegal");
            }
            List<MessageBlackWhite> blackWhiteList = messageBlackWhiteService.queryUserList(type, key, tenantId);
            return ResponseEntityWrapperUtil.wrapperOk(blackWhiteList);
        } catch (Exception e) {
            log.error("MessageBlackWhiteController | query user list error.", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 根据主键id删除用户
     *
     * @param id 主键id
     * @return 结果
     */
    @GetMapping("/delUser")
    public ResponseEntity<BaseResultDTO<Boolean>> delUser(@RequestParam Long id) {
        try {
            MessageBlackWhite blackWhite = messageBlackWhiteService.getBaseMapper().selectById(id);
            if (ObjectUtils.isEmpty(blackWhite)) {
                return ResponseEntityWrapperUtil.wrapperOk(true);
            }
            // 只能删除当前租户下的用户
            List<RoleDTO> userList = iamService.queryAllUserList();
            List<String> userIdList = userList.stream().map(RoleDTO::getId).collect(Collectors.toList());
            if (!userIdList.contains(blackWhite.getUserId())) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "No permission to delete the current user");
            }

            messageBlackWhiteService.delUser(id);
            return ResponseEntityWrapperUtil.wrapperOk(true);
        } catch (Exception e) {
            log.error("MessageBlackWhiteController | delete user list error.", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 校验用户是否存在对立的名单中
     *
     * @param type   名单类型 0:黑名单 1:白名单
     * @param userId 用户id
     * @return 结果
     */
    @GetMapping("/validUser")
    public ResponseEntity<BaseResultDTO<JSONObject>> validUser(@RequestParam Integer type, @RequestParam String userId) {
        try {
            // 校验用户
            Integer oppositeType = Constants.BLACK_TYPE.equals(type) ? Constants.WHITE_TYPE : Constants.BLACK_TYPE;
            List<MessageBlackWhite> blackWhiteList = messageBlackWhiteService.queryUserList(oppositeType, null, null);
            Boolean flag = Boolean.TRUE;
            String errorMsg = StringUtils.EMPTY;
            if (CollectionUtils.isNotEmpty(blackWhiteList)) {
                List<String> userIdList = blackWhiteList.stream().map(MessageBlackWhite::getUserId).collect(Collectors.toList());
                // 判断保存的黑名单用户是否在白名单中
                if (userIdList.contains(userId) && Constants.BLACK_TYPE.equals(type)) {
                    flag = Boolean.FALSE;
                    errorMsg = messageUtils.getMessage(I18NKey.WHITE_USER_IS_EXIST);
                }
                // 保存的白名单用户是否在黑名单中
                if (userIdList.contains(userId) && Constants.WHITE_TYPE.equals(type)) {
                    flag = Boolean.FALSE;
                    errorMsg = messageUtils.getMessage(I18NKey.BLACK_USER_IS_EXIST);
                }
            }
            JSONObject jsonObject = new JSONObject();
            jsonObject.put("flag", flag);
            jsonObject.put("errorMsg", errorMsg);
            return ResponseEntityWrapperUtil.wrapperOk(jsonObject);
        } catch (Exception e) {
            log.error("MessageBlackWhiteController | valid user error.", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 保存用户
     *
     * @param saveBlackWhiteReq 请求对象
     * @return 结果
     */
    @PostMapping("/saveUser")
    public ResponseEntity<BaseResultDTO<Boolean>> saveUser(@RequestBody @Valid SaveBlackWhiteReq saveBlackWhiteReq) {
        try {
            // 判断保存的黑名单是否已存在白名单中/保存的白名单是否已存在黑名单中
            Integer oppositeType = Constants.BLACK_TYPE.equals(saveBlackWhiteReq.getType()) ? Constants.WHITE_TYPE : Constants.BLACK_TYPE;
            List<MessageBlackWhite> blackWhiteList = messageBlackWhiteService.queryUserList(oppositeType, null, null);
            if (CollectionUtils.isNotEmpty(blackWhiteList)) {
                List<String> userIdList = blackWhiteList.stream().map(MessageBlackWhite::getUserId).collect(Collectors.toList());
                List<String> reqUserList = saveBlackWhiteReq.getUserList().stream().map(SaveBlackWhiteReq.UserDTO::getUserId).collect(Collectors.toList());
                if (CollectionUtils.containsAny(userIdList, reqUserList)) {
                    return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "there are users on the opposing list");
                }
            }

            // 获取所当前租户下所有用户，判断是否是当前租户下的用户
            List<RoleDTO> userList = iamService.queryAllUserList();
            List<String> userIdList = userList.stream().map(RoleDTO::getId).collect(Collectors.toList());
            for (SaveBlackWhiteReq.UserDTO userDTO : saveBlackWhiteReq.getUserList()) {
                if (!userIdList.contains(userDTO.getUserId())) {
                    return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "No permission to add user:" + userDTO.getUserName());
                }
            }

            messageBlackWhiteService.saveUser(saveBlackWhiteReq);
            return ResponseEntityWrapperUtil.wrapperOk(true);
        } catch (Exception e) {
            log.error("MessageBlackWhiteController |save user list error.", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }
}
