package com.digiwin.athena.semc.controller.message;

import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.appcore.util.ResponseEntityWrapper;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.I18NKey;
import com.digiwin.athena.semc.dto.message.CopyTemplateReq;
import com.digiwin.athena.semc.dto.message.QueryTemplateListReq;
import com.digiwin.athena.semc.dto.message.QueryTemplateListResp;
import com.digiwin.athena.semc.dto.message.QueryTemplateRelResp;
import com.digiwin.athena.semc.dto.message.SaveTemplateReq;
import com.digiwin.athena.semc.dto.message.TemplateRelReq;
import com.digiwin.athena.semc.entity.message.MessageChannelBase;
import com.digiwin.athena.semc.entity.message.MessageChannelTemplate;
import com.digiwin.athena.semc.service.message.MessageChannelBaseService;
import com.digiwin.athena.semc.service.message.MessageChannelService;
import com.digiwin.athena.semc.service.message.MessageChannelTemplateService;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;
import com.digiwin.athena.semc.util.SpecialValidator;

import net.sf.json.JSONObject;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.Resource;
import javax.validation.Valid;

import lombok.extern.slf4j.Slf4j;

/**
 * 消息发送渠道表(TMessageChannel)表控制层
 *
 * @author sungqz
 * @since 2023-11-30
 */
@Slf4j
@RestController
@RequestMapping("/semc/message/channel")
public class MessageChannelController {

    /**
     * 联系方式长度校验
     */
    public static final int CONTACT_MAX_LENGTH = 100;

    /**
     * 人员数量最大限制
     */
    public static final int MESSAGE_REL_PERSONNEL_COUNT_MAX = 1000;

    /**
     * 模板复制的最大限制数量
     */
    public static final int CHILD_TEMPLATE_MAX_COUNT = 10;

    /**
     * 服务对象
     */
    @Resource
    private MessageChannelBaseService messageChannelBaseService;

    @Resource
    private MessageChannelService messageChannelService;

    @Resource
    private MessageChannelTemplateService messageChannelTemplateService;

    @Resource
    private MessageUtils messageUtils;

    /**
     * 查询消息发送渠道基础配置
     *
     * @param sceneType 场景类型 1：统一消息管理
     * @return 查询结果
     */
    @GetMapping("/queryBaseChannel")
    public ResponseEntity<BaseResultDTO<List<MessageChannelBase>>> queryBaseChannel(@RequestParam(required = false) Integer sceneType) {
        try {
            List<MessageChannelBase> channelBaseList = messageChannelBaseService.queryBaseChannel();
            // 不传表示企微一期的功能，只针对乐普生，所以只返回企微
            if (sceneType == null && CollectionUtils.isNotEmpty(channelBaseList)) {
                channelBaseList = channelBaseList.stream().filter(x -> x.getChannelKey().equals("wecom")).collect(Collectors.toList());
            }
            return ResponseEntityWrapperUtil.wrapperOk(channelBaseList);
        } catch (Exception e) {
            log.error("MessageChannelController | query message base channel error. param:{}", sceneType, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/message/channel/queryBaseChannel");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 查询租户设置的消息渠道
     *
     * @return 查询结果
     */
    @GetMapping("/queryChannel")
    public ResponseEntity<BaseResultDTO<String>> queryChannel(@RequestParam(required = false, value = "tenantId") String tenantId) {
        try {
            return ResponseEntityWrapperUtil.wrapperOk(messageChannelService.queryChannelFlag(null, tenantId));
        } catch (Exception e) {
            log.error("MessageChannelController | query message channel error. param:{} ", tenantId, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/message/channel/queryChannel");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 保存租户设置的消息渠道
     *
     * @param channelStr 渠道
     * @return 编辑结果
     */
    @GetMapping("/saveChannel")
    public ResponseEntity<BaseResultDTO<Boolean>> saveChannel(@RequestParam String channelStr) {
        try {
            messageChannelService.saveChannel(channelStr);
            return ResponseEntityWrapperUtil.wrapperOk(true);
        } catch (Exception e) {
            log.error("MessageChannelController | save message channel error. param：{}", channelStr, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/message/channel/saveChannel");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 查询各渠道模板列表
     *
     * @param queryTemplateListReq 请求对象
     * @return 返回结果
     */
    @PostMapping("/queryTemplateList")
    public ResponseEntity<BaseResultDTO<List<QueryTemplateListResp>>> queryTemplateList(@RequestBody @Valid QueryTemplateListReq queryTemplateListReq) {
        try {
            List<QueryTemplateListResp> templateListRespList = messageChannelTemplateService.queryTemplateList(queryTemplateListReq);
            return ResponseEntityWrapperUtil.wrapperOk(templateListRespList);
        } catch (Exception e) {
            log.error("MessageChannelController | query template list error. param：{}", queryTemplateListReq, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/message/channel/queryTemplateList");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 修改模板状态
     *
     * @param saveTemplateReq 请求对象
     * @return 返回结果
     */
    @PostMapping("/updateStatus")
    public ResponseEntity<BaseResultDTO<Boolean>> updateStatus(@RequestBody @Valid SaveTemplateReq saveTemplateReq) {
        try {
            if (saveTemplateReq.getStatus() == null) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "the status can not be null");
            }
            messageChannelTemplateService.updateStatus(saveTemplateReq);
            return ResponseEntityWrapperUtil.wrapperOk(true);
        } catch (Exception e) {
            log.error("MessageChannelController | update status error. param：{}", saveTemplateReq, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/message/channel/updateStatus");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 更新模板
     *
     * @param saveTemplateReq 请求对象
     * @return 返回结果
     */
    @PostMapping("/updateTemplate")
    public ResponseEntity<BaseResultDTO<Boolean>> updateTemplate(@RequestBody @Valid SaveTemplateReq saveTemplateReq) {
        try {
            if (!Constants.MessageChannelEnum.EMAIL.getFlag().equals(saveTemplateReq.getChannelFlag())) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "the channelFlag is illegal");
            }
            if (StringUtils.isBlank(saveTemplateReq.getRemark())) {
                saveTemplateReq.setRemark("");
            }
            messageChannelTemplateService.updateTemplate(saveTemplateReq);
            return ResponseEntityWrapperUtil.wrapperOk(true);
        } catch (Exception e) {
            log.error("MessageChannelController | update template error. param：{}", saveTemplateReq, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/message/channel/updateTemplate");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 查看模板接收人
     *
     * @param templateRelReq 请求对象
     * @return 返回结果
     */
    @PostMapping("/queryRelPerson")
    public ResponseEntity<BaseResultDTO<QueryTemplateRelResp>> queryRelPerson(@RequestBody @Valid TemplateRelReq templateRelReq) {
        try {
            if (!Arrays.asList(Constants.MessageChannelEnum.EMAIL.getFlag(), Constants.MessageChannelEnum.SMS.getFlag(),
                    Constants.MessageChannelEnum.WECOM.getFlag(), Constants.MessageChannelEnum.DINGTALK.getFlag()).contains(templateRelReq.getChannelFlag())) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "the channelFlag is illegal");
            }
            QueryTemplateRelResp relResp = messageChannelTemplateService.queryRelPerson(templateRelReq);
            return ResponseEntityWrapperUtil.wrapperOk(relResp);
        } catch (Exception e) {
            log.error("MessageChannelController | query rel person error. param：{}", templateRelReq, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/message/channel/queryRelPerson");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 添加模板接收人
     *
     * @param templateRelReq 请求对象
     * @return 返回结果
     */
    @PostMapping("/addRelPerson")
    public ResponseEntity<?> addRelPerson(@RequestBody @Valid TemplateRelReq templateRelReq) {
        try {
            // 参数校验
            if (!Arrays.asList(Constants.MessageChannelEnum.EMAIL.getFlag(), Constants.MessageChannelEnum.SMS.getFlag(),
                    Constants.MessageChannelEnum.WECOM.getFlag(), Constants.MessageChannelEnum.DINGTALK.getFlag()).contains(templateRelReq.getChannelFlag())) {
                return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "the channelFlag is illegal");
            }
            if (StringUtils.isBlank(templateRelReq.getLanguage())) {
                return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "the language can not be null");
            }
            if (CollectionUtils.isEmpty(templateRelReq.getContactList())) {
                return ResponseEntityWrapper.wrapperOk(true);
            }
            for (String contact : templateRelReq.getContactList()) {
                if (contact.length() >= CONTACT_MAX_LENGTH) {
                    return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "The length of the contact cannot over" + CONTACT_MAX_LENGTH);
                }
            }
            // 邮箱校验
            if (Constants.MessageChannelEnum.EMAIL.getFlag().equals(templateRelReq.getChannelFlag())) {
                // 邮箱类型不能为空
                if (templateRelReq.getEmailType() == null) {
                    return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "the emailType can not be null");
                }
                // 邮箱格式
                for (String email : templateRelReq.getContactList()) {
                    if (!SpecialValidator.validateEmail(email)) {
                        return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "the email format error:" + email);
                    }
                }
            }
            // 手机号校验
            if (Constants.MessageChannelEnum.SMS.getFlag().equals(templateRelReq.getChannelFlag())) {
                for (String telephone : templateRelReq.getContactList()) {
                    if (!SpecialValidator.validatePositiveNumber(telephone)) {
                        return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "The phone number is illegal:" + telephone);
                    }
                }
            }

            JSONObject jsonObject = new JSONObject();
            jsonObject.put("errorCode", "");
            // 校验人员数量
            Integer totalCount = validPersonNumber(templateRelReq);
            if (totalCount > MESSAGE_REL_PERSONNEL_COUNT_MAX) {
                jsonObject.put("errorCode", ErrorCodeConstant.PARAM_LENGTH_ERROR);
                jsonObject.put("errorMsg", "the number of person can not over " + MESSAGE_REL_PERSONNEL_COUNT_MAX);
                return ResponseEntityWrapper.wrapperOk(jsonObject);
            }
            messageChannelTemplateService.addRelPerson(templateRelReq);
            return ResponseEntityWrapper.wrapperOk(jsonObject);
        } catch (Exception e) {
            log.error("MessageChannelController | add rel person error. param：{}", templateRelReq, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/message/channel/addRelPerson, message:" + e.getMessage());
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 校验人员数量
     *
     * @param templateRelReq 入参
     * @return 返回
     */
    private Integer validPersonNumber(TemplateRelReq templateRelReq) {
        QueryTemplateRelResp relResp = messageChannelTemplateService.queryRelPerson(templateRelReq);
        Integer paramCount = CollectionUtils.isNotEmpty(templateRelReq.getContactList()) ? templateRelReq.getContactList().size() : 0;
        if (Constants.MessageChannelEnum.EMAIL.getFlag().equals(templateRelReq.getChannelFlag())) {
            if (Constants.EmailTypeEnum.CONTACT.getFlag().equals(templateRelReq.getEmailType())) {
                Integer existCount = CollectionUtils.isNotEmpty(relResp.getContactList()) ? relResp.getContactList().size() : 0;
                return existCount + paramCount;
            }
            if (Constants.EmailTypeEnum.CCCONTACT.getFlag().equals(templateRelReq.getEmailType())) {
                Integer existCount = CollectionUtils.isNotEmpty(relResp.getCcContactList()) ? relResp.getCcContactList().size() : 0;
                return existCount + paramCount;
            }
            if (Constants.EmailTypeEnum.BCCCONTACT.getFlag().equals(templateRelReq.getEmailType())) {
                Integer existCount = CollectionUtils.isNotEmpty(relResp.getBccContactList()) ? relResp.getBccContactList().size() : 0;
                return existCount + paramCount;
            }
        } else {
            Integer existCount = CollectionUtils.isNotEmpty(relResp.getContactList()) ? relResp.getContactList().size() : 0;
            return existCount + paramCount;
        }
        return 0;
    }

    /**
     * 删除模板收件人
     *
     * @param templateRelReq 请求对象
     * @return 返回结果
     */
    @PostMapping("/delRelPerson")
    public ResponseEntity<BaseResultDTO<Boolean>> delRelPerson(@RequestBody @Valid TemplateRelReq templateRelReq) {
        try {
            if (StringUtils.isBlank(templateRelReq.getLanguage())) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "the language can not be null");
            }
            if (!Arrays.asList(Constants.MessageChannelEnum.EMAIL.getFlag(), Constants.MessageChannelEnum.SMS.getFlag(),
                    Constants.MessageChannelEnum.WECOM.getFlag(), Constants.MessageChannelEnum.DINGTALK.getFlag()).contains(templateRelReq.getChannelFlag())) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "the channelFlag is illegal");
            }
            messageChannelTemplateService.delRelPerson(templateRelReq);
            return ResponseEntityWrapperUtil.wrapperOk(true);
        } catch (Exception e) {
            log.error("MessageChannelController | del rel person error. param:{}", templateRelReq, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/message/channel/delRelPerson");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 查询租户下备注对应的事件id
     *
     * @param tenantId 租户id
     * @param remark   备注
     * @return 返回结果
     */
    @GetMapping("/queryEventIdList")
    public ResponseEntity<BaseResultDTO<List<String>>> queryEventIdList(@RequestParam String tenantId, @RequestParam String remark) {
        try {
            if (StringUtils.isBlank(tenantId) || StringUtils.isBlank(remark)) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "the tenantId or remark can not be null");
            }
            List<String> eventIdList = messageChannelTemplateService.queryEventIdList(tenantId, remark);
            if(CollectionUtils.isEmpty(eventIdList)){
                return ResponseEntityWrapperUtil.wrapperOk(new ArrayList<String>());
            }
            return ResponseEntityWrapperUtil.wrapperOk(eventIdList);
        } catch (Exception e) {
            log.error("MessageChannelController | query event id list error. tenantId:{}, remark:{}", tenantId, remark, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/message/channel/queryEventIdList");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }
    /**
     * 复制模板
     *
     * @param copyTemplateReq 请求对象
     * @return 返回结果
     */
    @PostMapping("/copyTemplate")
    public ResponseEntity<BaseResultDTO<Boolean>> copyTemplate(@RequestBody @Valid CopyTemplateReq copyTemplateReq) {
        try {
            // 判断是否超过10个子模板
            if (copyTemplateReq.getId() != null) {
                Long count = messageChannelTemplateService.queryChildTemplateCount(copyTemplateReq.getId());
                if (count >= CHILD_TEMPLATE_MAX_COUNT) {
                    return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "最多允许复制10个模板");
                }
            }
            messageChannelTemplateService.copyTemplate(copyTemplateReq);
            return ResponseEntityWrapperUtil.wrapperOk(true);
        } catch (Exception e) {
            log.error("MessageChannelController | copy template error. param:{}", copyTemplateReq, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/message/channel/copyTemplate, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 删除模板
     *
     * @param templateId 模板主键id
     * @return 返回结果
     */
    @GetMapping("/delTemplate")
    public ResponseEntity<BaseResultDTO<Boolean>> delTemplate(@RequestParam Long templateId) {
        try {
            // 判断当前模板是否是子模板
            MessageChannelTemplate template = messageChannelTemplateService.getBaseMapper().selectById(templateId);
            if (template == null) {
                return ResponseEntityWrapperUtil.wrapperOk(true);
            }
            if (template.getParentId() == null) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "不允许删除原模版");
            }
            messageChannelTemplateService.getBaseMapper().deleteById(templateId);
            return ResponseEntityWrapperUtil.wrapperOk(true);
        } catch (Exception e) {
            log.error("MessageChannelController | del template error. templateId:{}", templateId, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/message/channel/delTemplate, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }
}

