package com.digiwin.athena.semc.controller.message;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.appcore.util.ResponseEntityWrapper;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.I18NKey;
import com.digiwin.athena.semc.common.PageInfoResp;
import com.digiwin.athena.semc.controller.BasicController;
import com.digiwin.athena.semc.dto.message.*;
import com.digiwin.athena.semc.entity.message.ThirdMessageConfig;
import com.digiwin.athena.semc.entity.message.ThirdMessageInfo;
import com.digiwin.athena.semc.entity.message.ThirdMessageReadInfo;
import com.digiwin.athena.semc.entity.message.ThirdMessageType;
import com.digiwin.athena.semc.entity.portal.LabelSystemCustom;
import com.digiwin.athena.semc.quartz.ThirdMessagePullTaskJob;
import com.digiwin.athena.semc.service.menu.AuthService;
import com.digiwin.athena.semc.service.message.ThirdMessageInfoService;
import com.digiwin.athena.semc.service.message.ThirdMessageInfoService;
import com.digiwin.athena.semc.service.message.ThirdMessageReadInfoService;
import com.digiwin.athena.semc.service.message.ThirdSystemMessageService;
import com.digiwin.athena.semc.service.portal.LabelSystemDataService;
import com.digiwin.athena.semc.util.*;

import com.digiwin.athena.semc.vo.message.MessageAppNumResp;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.validation.Valid;

import lombok.extern.slf4j.Slf4j;

/**
 * @description: 三方异构系统消息接入
 * @createDate: 2024/3/4
 * @author: sungqz
 */
@Slf4j
@RestController
@RequestMapping("/semc/third/message")
public class ThirdSystemMessageController extends BasicController<ThirdSystemMessageService, ThirdMessageConfig> {

    @Resource
    private ThirdSystemMessageService thirdSystemMessageService;

    @Resource
    private ThirdMessageInfoService thirdMessageInfoService;

    @Resource
    ThirdMessageReadInfoService thirdMessageReadInfoService;

    @Resource
    private MessageUtils messageUtils;

    @Resource
    private ThirdMessagePullTaskJob thirdMessagePullTaskJob;

    @Autowired
    private LabelSystemDataService labelSystemDataService;

    @Autowired
    private AuthService authService;

    /**
     * 查询异构系统的消息配置
     *
     * @param queryImportRecordReq 入参
     * @return 返回
     */
    @PostMapping("/queryMessageConfigPage")
    public ResponseEntity<BaseResultDTO<PageInfoResp<ThirdMessageConfig>>> queryMessageConfigPage(@RequestBody @Valid QueryThirdMessageConfigReq queryImportRecordReq) {
        try {
            PageInfoResp<ThirdMessageConfig> pageInfoResp = thirdSystemMessageService.queryMessageConfigPage(queryImportRecordReq);
            return ResponseEntityWrapperUtil.wrapperOk(pageInfoResp);
        } catch (Exception e) {
            log.error("query message config error, param:{}", queryImportRecordReq, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/third/message/queryMessageConfig");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 保存异构系统的消息配置
     *
     * @param saveThirdMessageConfigReq 入参
     * @return 返回
     */
    @PostMapping("/saveApplication")
    public ResponseEntity<BaseResultDTO<Boolean>> saveApplication(@RequestBody @Valid SaveThirdMessageConfigReq saveThirdMessageConfigReq, HttpServletRequest request) {
        try {
            //集成管理/异构消息接入配置
            //暂时不阻断，只是写入安全日志
            if (!authService.hasAuth("third-message")) {
                //写入安全日志
                SecurityLogUtil.writeSecurityLog(request, JSON.toJSONString(saveThirdMessageConfigReq));
            }
             // 数据获取方式如果是拉取方式
            if (Constants.DataGetTypeEnum.PULL.getFlag().equals(saveThirdMessageConfigReq.getDataGetType())) {
                if (saveThirdMessageConfigReq.getPullPeriod() == null) {
                    return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "the pullPeriod cannot be null");
                }
                if (saveThirdMessageConfigReq.getAppAccessModel() == null) {
                    return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "the appAccessModel cannot be null");
                }

                // 混合云模式需要填地中台产品名称，系统UID
                if (Constants.DataModelEnum.MODEL_HYBRID_CLOUD.getVal().equals(saveThirdMessageConfigReq.getAppAccessModel())) {
                    if (StringUtils.isBlank(saveThirdMessageConfigReq.getMiddleSystemName())) {
                        return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "the middleSystemName cannot be null");
                    }
                    if (StringUtils.isBlank(saveThirdMessageConfigReq.getMiddleSystemUid())) {
                        return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "the middleSystemUid cannot be null");
                    }
                } else { // 非混合云
                    if (StringUtils.isBlank(saveThirdMessageConfigReq.getDomain())) {
                        return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "the domain cannot be null");
                    }
                }
            }
            // 判断应用是否存在
            boolean flag = thirdSystemMessageService.validMessageConfigExist(saveThirdMessageConfigReq);
            if (flag) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "应用已存在！");
            }
            if(saveThirdMessageConfigReq.getId()!=null){
                //删除定时任务
                thirdMessagePullTaskJob.deleteJob("ThirdMessagePullTaskJob","MessageJob_"+saveThirdMessageConfigReq.getId());
            }
            ThirdMessageConfig thirdMessageConfig=thirdSystemMessageService.saveApplication(saveThirdMessageConfigReq);
            //启动定时任务
            if(saveThirdMessageConfigReq.getDataGetType()==2&&thirdMessageConfig.getValidStatus()==1) {
                Map<String,Object> params=new HashMap<>();
                params.put("id",thirdMessageConfig.getId());
                params.put("appCode",thirdMessageConfig.getAppCode());
                params.put("appPrimaryId",thirdMessageConfig.getAppPrimaryId());
                params.put("appSource",thirdMessageConfig.getAppSource());
                params.put("pullPeriod",thirdMessageConfig.getPullPeriod());
                params.put("appAccessModel",thirdMessageConfig.getAppAccessModel());
                params.put("middleSystemName",thirdMessageConfig.getMiddleSystemName());
                params.put("middleSystemUid",thirdMessageConfig.getMiddleSystemUid());
                params.put("domain",thirdMessageConfig.getDomain());
                params.put("sendChannel",thirdMessageConfig.getSendChannel());
                params.put("tenantsid",AppAuthContextHolder.getContext().getAuthoredUser().getTenantId());
                //获取appToken
                Map<String, String> ssoInfoMap = labelSystemDataService.getSSoInfo(Long.parseLong(thirdMessageConfig.getAppPrimaryId()), thirdMessageConfig.getAppSource());
                if (null != ssoInfoMap && StringUtils.isNotEmpty(ssoInfoMap.get("appId"))) {
                    params.put("appId",ssoInfoMap.get("appId").toString());
                }
                thirdMessagePullTaskJob.addJob(params);
            }
            return ResponseEntityWrapperUtil.wrapperOk(true);
        } catch (Exception e) {
            log.error("add application error. addThirdMessageConfigReq:{}", saveThirdMessageConfigReq, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/third/message/saveApplication");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 接收异构系统的消息推送 restful接口
     *
     * @param receiveThirdMessageReq 入参
     * @return 返回
     */
    @PostMapping("/receive")
    public ResponseEntity<BaseResultDTO<Boolean>> receiveMessage(@RequestBody @Valid ReceiveThirdMessageReq receiveThirdMessageReq) {
        try {
            log.info("receive message. param:{}", receiveThirdMessageReq);
            String appCode = AesUtil.decrypt(receiveThirdMessageReq.getAppCode());
            log.info("AesUtil decrypt appCode. result:{}", appCode);
            // 判断当前应用是否存在
            List<ThirdMessageConfig> thirdMessageConfigList = thirdSystemMessageService.queryConfigByAppCode(appCode,receiveThirdMessageReq.getAppId());
            if (CollectionUtils.isEmpty(thirdMessageConfigList)) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, "当前应用不在配置中，无权推送消息");
            }
            List<ThirdMessageConfig> filteredList = thirdMessageConfigList.stream().filter(x -> Constants.VALID_STATUS_ENABLE.equals(x.getValidStatus())).collect(Collectors.toList());
            if (CollectionUtils.isEmpty(filteredList)) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, "当前应用未启用，无权推送消息");
            }
            String sendChannel = filteredList.get(0).getSendChannel();
            receiveThirdMessageReq.setSendChannel(sendChannel);
            for (ThirdMessageInfo info : receiveThirdMessageReq.getMessageList()) {
                //校验消息ID
                if (StringUtils.isEmpty(info.getMessageId())) {
                    return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, "消息id不能为空");
                }
                //校验时间和格式
                if (StringUtils.isEmpty(info.getMessagePublishTime())) {
                    return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, "消息发布时间不能为空");
                }
                if (!DateUtils.validateDate(info.getMessagePublishTime(), DateUtils.DATE_TIME_NORMAL_FORMATTER)) {
                    return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, "消息发布时间格式不正确");
                }
                //校验内容
                if (StringUtils.isEmpty(info.getMessageTitle()) && StringUtils.isEmpty(info.getMessageContent())) {
                    return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, "消息标题和消息内容不能同时为空");
                }

                if (Objects.isNull(info.getSideType())){
                    return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, "消息端不能为空");
                }
            }
            receiveThirdMessageReq.setAppCode(appCode);
            thirdSystemMessageService.receiveMessage(receiveThirdMessageReq);
            return ResponseEntityWrapperUtil.wrapperOk(true);
        } catch (Exception e) {
            log.error("receive message error. param:{}", receiveThirdMessageReq, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/third/message/receive, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 查询消息的未读数量，在首页小铃铛展示
     *
     * @return 返回
     */
    @PostMapping("/queryUnreadCount")
    public ResponseEntity<BaseResultDTO<Long>> queryUnreadCount(@RequestBody QueryThirdMessageListReq queryThirdMessageListReq) {
        try {
            Long count = thirdSystemMessageService.queryUnreadCount(queryThirdMessageListReq.getClientType());
            return ResponseEntityWrapperUtil.wrapperOk(count);
        } catch (Exception e) {
            log.error("query unread count error", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/third/message/queryUnreadCount");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 分页查询消息列表
     *
     * @return 返回
     */
    @PostMapping("/queryMessageListPage")
    public ResponseEntity<BaseResultDTO<PageInfoResp<ThirdMessageInfo>>> queryMessageListPage(@RequestBody @Valid QueryThirdMessageListReq queryThirdMessageListReq) {
        if (StringUtils.isBlank(queryThirdMessageListReq.getAppId())) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "appId不能为空");
        }
        PageInfoResp<ThirdMessageInfo> messageInfoListPage = thirdSystemMessageService.queryMessageListPage(queryThirdMessageListReq);
        return ResponseEntityWrapperUtil.wrapperOk(messageInfoListPage);
    }

    /**
     * 读取消息，记录为已读消息
     *
     * @return 返回
     */
    @PostMapping("/readMessage")
    public ResponseEntity<BaseResultDTO<Boolean>> readMessage(@RequestBody ReadThirdMessageReq readThirdMessageReq) {
        try {
            thirdSystemMessageService.readMessage(readThirdMessageReq);
            return ResponseEntityWrapperUtil.wrapperOk(true);
        } catch (Exception e) {
            log.error("read message list error. readThirdMessageReq:{}", readThirdMessageReq, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/third/message/readMessage");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 删除消息的配置
     *
     * @param deleteMessageConfigReq 入参
     * @return 返回
     */
    @PostMapping("/deleteBatch")
    public ResponseEntity<BaseResultDTO<Boolean>> deleteBatch(@RequestBody DeleteMessageConfigReq deleteMessageConfigReq) {
        try {
            QueryWrapper<ThirdMessageConfig> condition = new QueryWrapper<>();
            condition.in("id", deleteMessageConfigReq.getIdList());
            condition.eq("valid_status", Constants.VALID_STATUS_ENABLE);
            List<ThirdMessageConfig> messageConfigList = thirdSystemMessageService.getBaseMapper().selectList(condition);
            if (CollectionUtils.isNotEmpty(messageConfigList)) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, "已启用的应用不能被删除");
            }
            if (CollectionUtils.isEmpty(deleteMessageConfigReq.getIdList())) {
                return ResponseEntityWrapperUtil.wrapperOk(true);
            }
            //删除定时任务
            List<Long>idList=deleteMessageConfigReq.getIdList();
            for(Long id:idList){
                thirdMessagePullTaskJob.deleteJob("ThirdMessagePullTaskJob","MessageJob_"+id);
            }
            thirdSystemMessageService.deleteBatchApp(deleteMessageConfigReq);
            return ResponseEntityWrapperUtil.wrapperOk(true);
        } catch (Exception e) {
            log.error("read message list error. param:{}", deleteMessageConfigReq, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/third/message/deleteBatch");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 同步消息
     * @return 返回
     */
    @PostMapping("/syncMessageData")
    public ResponseEntity<?> syncMessageData() {
        try {
            return thirdSystemMessageService.syncMessageData();
        } catch (Exception e) {
            log.error("read message list error. readThirdMessageReq:{}", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/third/message/syncMessageData");
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 一键已读
     *
     * @return 返回
     */
    @PostMapping("/readAllMessage")
    public ResponseEntity<BaseResultDTO<Integer>> readAllMessage(@RequestBody QueryThirdMessageListReq queryThirdMessageListReq) {
        try {
            return thirdSystemMessageService.readAllMessage(queryThirdMessageListReq);
        } catch (Exception e) {
            log.error("readAllMessage error. e:{}", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/third/message/readAllMessage");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 查询消息所属应用列表
     *
     * @param queryThirdMessageListReq 请求入参
     * @return 返回
     */
    @PostMapping("/queryMessageAppList")
    public ResponseEntity<BaseResultDTO<List<MessageAppNumResp>>> queryMessageAppList(@RequestBody QueryThirdMessageListReq queryThirdMessageListReq) {
        List<MessageAppNumResp> appNumRespList = thirdSystemMessageService.queryMessageAppList(queryThirdMessageListReq);
        return ResponseEntityWrapperUtil.wrapperOk(appNumRespList);
    }

    /**
     * 查询消息分类列表
     *
     * @return 返回
     */
    @PostMapping("/queryMessageType")
    public ResponseEntity<BaseResultDTO<List<ThirdMessageType>>> queryMessageType() {
        List<ThirdMessageType> messageTypeList = thirdSystemMessageService.queryMessageType();
        return ResponseEntityWrapperUtil.wrapperOk(messageTypeList);
    }

    /**
     * 保存用户的消息应用展示顺序
     *
     * @param appIdList 应用id列表
     * @return 返回
     */
    @PostMapping("/saveAppSort")
    public ResponseEntity<BaseResultDTO<Boolean>> saveAppSort(@RequestBody List<String> appIdList) {
        thirdSystemMessageService.saveAppSort(appIdList);
        return ResponseEntityWrapperUtil.wrapperOk(true);
    }

    /**
     * 查询消息的数量
     *
     * @return 返回
     */
    @PostMapping("/queryMessageCount")
    public ResponseEntity<BaseResultDTO<MessageAppNumResp>> queryMessageCount(@RequestBody QueryThirdMessageListReq queryThirdMessageListReq) {
        if (StringUtils.isBlank(queryThirdMessageListReq.getAppId())) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "appId不能为空");
        }
        MessageAppNumResp messageAppNumResp = thirdSystemMessageService.queryMessageCount(queryThirdMessageListReq);
        return ResponseEntityWrapperUtil.wrapperOk(messageAppNumResp);
    }

    /**
     * 删除消息的配置
     *
     * @param deleteMessageConfigReq 入参
     * @return 返回
     */
    @PostMapping("/delete")
    public ResponseEntity<BaseResultDTO<Boolean>> delete(@RequestBody DeleteMessageConfigReq deleteMessageConfigReq) {
        try {
            QueryWrapper<ThirdMessageInfo> condition = new QueryWrapper<>();
            condition.in("id", deleteMessageConfigReq.getIdList());
            InterceptorIgnoreUtil.handler(() -> thirdMessageInfoService.remove(condition));
            return ResponseEntityWrapperUtil.wrapperOk(true);
        } catch (Exception e) {
            log.error("read message list error. param:{}", deleteMessageConfigReq, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/third/message/deleteBatch");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 删除消息和已读消息（开发使用）
     *
     * @param deleteMessageConfigReq 入参
     * @return 返回
     */
    @PostMapping("/deleteMessage")
    public ResponseEntity<BaseResultDTO<Boolean>> deleteMessage(@RequestBody DeleteMessageConfigReq deleteMessageConfigReq) {
        // 删除消息
        QueryWrapper<ThirdMessageInfo> condition = new QueryWrapper<>();
        condition.in("id", deleteMessageConfigReq.getIdList());
        thirdMessageInfoService.remove(condition);

        // 删除已读的消息
        QueryWrapper<ThirdMessageReadInfo> queryWrapper = new QueryWrapper<>();
        queryWrapper.in("message_id", deleteMessageConfigReq.getIdList());
        thirdMessageReadInfoService.remove(queryWrapper);
        return ResponseEntityWrapperUtil.wrapperOk(true);
    }
}
