package com.digiwin.athena.semc.controller.mobile;


import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.I18NKey;
import com.digiwin.athena.semc.dto.mobile.MobileClassificationReqDTO;
import com.digiwin.athena.semc.dto.news.AnnouncementIdReq;
import com.digiwin.athena.semc.dto.news.QueryAnnouncementTypeReq;
import com.digiwin.athena.semc.entity.mobile.MobileDatasourceClassification;
import com.digiwin.athena.semc.service.mobile.MobileDatasourceClassificationService;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;
import com.digiwin.athena.semc.vo.mobile.MobileClassificatonResp;
import org.apache.commons.lang3.ObjectUtils;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.time.LocalDateTime;
import java.util.List;

import javax.annotation.Resource;
import javax.validation.Valid;

import lombok.extern.slf4j.Slf4j;

/**
 * 自定义应用分类控制层
 *
 * @author qutong
 * @since 2024-09-03
 */
@Slf4j
@RestController
@RequestMapping("/semc/mobile/application/class")
public class MobileDatasourceClassificationController {

    /**
     * 服务对象
     */
    @Resource
    private MobileDatasourceClassificationService mobileDatasourceClassificationService;

    @Resource
    private MessageUtils messageUtils;

    /**
     * 分类数量上限
     */
    private static final Integer MAX_COUNT = 20;

    /**
     * 查询自定义应用分类列表
     *
     * @param queryAnnouncementTypeReq 调用场景 不传表示新增页下拉框，传值表示分类列表
     * @return 查询结果
     */
    @PostMapping("/queryApplicationClassificationList")
    public ResponseEntity<BaseResultDTO<List<MobileDatasourceClassification>>> queryApplicationClassificationList(@RequestBody QueryAnnouncementTypeReq queryAnnouncementTypeReq) {
        try {
            List<MobileDatasourceClassification> mobileApplicationClassificationList = mobileDatasourceClassificationService.queryApplicationClassificationList();
            return ResponseEntityWrapperUtil.wrapperOk(mobileApplicationClassificationList);
        } catch (Exception e) {
            log.error("query news type list error. param:{}", queryAnnouncementTypeReq, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/mobile/application/class/queryApplicationClassificationList, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 保存应用分类
     *
     * @param mobileApplicationClassification 请求
     * @return 编辑结果
     */
    @PostMapping("/saveApplicationClassification")
    public ResponseEntity<BaseResultDTO<Boolean>> saveApplicationClassification(@RequestBody @Valid MobileDatasourceClassification mobileApplicationClassification) {
        try {
            // 新增时判断数量
            List<MobileDatasourceClassification> applicationClassificationList = mobileDatasourceClassificationService.queryApplicationClassificationList();
            if (ObjectUtils.isEmpty(mobileApplicationClassification.getId())) {
                if (applicationClassificationList.size() >= MAX_COUNT) {
                    return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "分类数量不能超过20个");
                }
            }

            // 判断名称是否重复
            List<MobileDatasourceClassification> mobileApplicationClassificationList = mobileDatasourceClassificationService.existNameValid(mobileApplicationClassification.getId(), mobileApplicationClassification.getClassificationName());
            if (mobileApplicationClassificationList.size() > 0) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "该应用分类名称已存在");
            }

            mobileDatasourceClassificationService.saveClassification(mobileApplicationClassification);
            return ResponseEntityWrapperUtil.wrapperOk(true);
        } catch (Exception e) {
            log.error("save news type error. param:{}", mobileApplicationClassification, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/mobile/application/class/saveApplicationClassification, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 删除应用分类
     *
     * @param announcementIdReq 主键id
     * @return 返回
     */
    @PostMapping("/delApplicationClassification")
    public ResponseEntity<BaseResultDTO<Boolean>> delApplicationClassification(@RequestBody @Valid AnnouncementIdReq announcementIdReq) {
        try {
            mobileDatasourceClassificationService.delApplicationClassification(announcementIdReq.getId());
            return ResponseEntityWrapperUtil.wrapperOk(true);
        } catch (Exception e) {
            log.error("delete news type error. param:{}", announcementIdReq, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/mobile/application/class/delApplicationClassification, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 查询移动端分类和分类下应用列表
     *
     * @return
     */
    @PostMapping("/getClassificationList")
    public ResponseEntity<BaseResultDTO<List<MobileClassificatonResp>>> getClassificationList(@RequestBody MobileClassificationReqDTO reqDTO) {
        List<MobileClassificatonResp> result = mobileDatasourceClassificationService.getClassificationList(reqDTO.getIsAll());
        return ResponseEntityWrapperUtil.wrapperOk(result);
    }

    /**
     * 查询移动端分类和分类下应用列表  默认门户
     *
     * @return
     */
    @PostMapping("/getClassificationListNew")
    public ResponseEntity<BaseResultDTO<List<MobileClassificatonResp>>> getClassificationListNew(@RequestBody MobileClassificationReqDTO reqDTO) {
        List<MobileClassificatonResp> result = mobileDatasourceClassificationService.getClassificationList(reqDTO.getIsAll());
        return ResponseEntityWrapperUtil.wrapperOk(result);
    }
}

