package com.digiwin.athena.semc.controller.mobile;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.CollectionUtils;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.I18NKey;
import com.digiwin.athena.semc.common.PageInfoResp;
import com.digiwin.athena.semc.common.enums.ApplicationTypeEnum;
import com.digiwin.athena.semc.common.enums.VirtualApplicationEnum;
import com.digiwin.athena.semc.dto.mobile.CustomApplicationIdReq;
import com.digiwin.athena.semc.dto.mobile.CustomApplicationsListPageReq;
import com.digiwin.athena.semc.dto.mobile.SaveAppcationReq;
import com.digiwin.athena.semc.dto.mobile.UpdateAppcationConfigReq;
import com.digiwin.athena.semc.dto.mobile.UpdateAppcationStatusReq;
import com.digiwin.athena.semc.dto.news.DeleteNewsOrTypeReq;
import com.digiwin.athena.semc.entity.applink.AppLinkDTO;
import com.digiwin.athena.semc.entity.mobile.MobileDatasourceInfo;
import com.digiwin.athena.semc.entity.mobile.MobilePortalInfoContent;
import com.digiwin.athena.semc.entity.portal.LabelSystemData;
import com.digiwin.athena.semc.mapper.mobile.MobileDatasourceInfoMapper;
import com.digiwin.athena.semc.service.applink.AppLinkService;
import com.digiwin.athena.semc.service.mobile.MobileDatasourceInfoService;
import com.digiwin.athena.semc.service.mobile.MobilePortalService;
import com.digiwin.athena.semc.service.portal.LabelSystemDataService;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;
import com.digiwin.athena.semc.vo.common.PermissionUserFunctionalDTO;
import com.digiwin.athena.semc.vo.common.UserApplicationModulesDTO;
import com.digiwin.athena.semc.vo.common.UserApplicationSimpleDTO;
import com.digiwin.athena.semc.vo.mobile.MobileDatasourceInfoVO;

import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.time.LocalDateTime;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.Resource;
import javax.validation.Valid;

import lombok.extern.slf4j.Slf4j;

/**
 * 移动应用/作业管理功能控制层
 *
 * @author qutong
 * @since 2024-09-03
 */
@Slf4j
@RestController
@RequestMapping("/semc/mobile/application")
public class MobileDatasourceInfoController {

    /**
     * 服务对象
     */
    @Resource
    private MobileDatasourceInfoService mobileDatasourceInfoService;

    @Autowired
    private MobilePortalService mobilePortalService;

    @Autowired
    private  MobileDatasourceInfoMapper mobileDatasourceInfoMapper;

    @Autowired
    private AppLinkService appLinkService;

    @Resource
    private MessageUtils messageUtils;

    @Autowired
    private LabelSystemDataService labelSystemDataService;
    /**
     * 数量最大限制
     */
//    public static final int MAX_NEWS_COUNT = 1000;


    /**
     * 管理后台分页查询应用列表
     *
     * @param customApplicationsListPageReq 入参
     * @return 查询结果
     */
    @PostMapping("/queryCustomApplicationsByPage")
    public ResponseEntity<BaseResultDTO<PageInfoResp<MobileDatasourceInfoVO>>> queryCustomApplicationsByPage(@RequestBody @Valid CustomApplicationsListPageReq customApplicationsListPageReq) {
        try {
            PageInfoResp<MobileDatasourceInfoVO> pageInfoResp = mobileDatasourceInfoService.queryCustomApplicationsByPage(customApplicationsListPageReq);
            return ResponseEntityWrapperUtil.wrapperOk(pageInfoResp);
        } catch (Exception e) {
            log.error("query custom applications by page error. param:{}", customApplicationsListPageReq, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/mobile/application/queryCustomApplicationsByPage, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 更新应用启用状态
     *
     * @param updateAppcationStatusReq 入参
     * @return 返回
     */
    @PostMapping("/updateStatus")
    public ResponseEntity<BaseResultDTO<Boolean>> updateStatus(@RequestBody @Valid UpdateAppcationStatusReq updateAppcationStatusReq) {
        try {

            //停用时当前应用如果被组件引用，则不支持关闭
            if(Constants.MobileDatasourceInfoStatusEnum.UNUSABLE.getFlag().equals(updateAppcationStatusReq.getStatus())){
                //停用时当前应用如果被移动门户引用，则不支持关闭
                List<MobilePortalInfoContent> mobilePortalInfoContents = mobilePortalService.checkContentApp(updateAppcationStatusReq.getIdList());
                if (CollectionUtils.isNotEmpty(mobilePortalInfoContents)){
                    return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage("error.message.mobile.custom.application.status.close"));
                }
                List<String> workCodeList=updateAppcationStatusReq.getIdList().stream().map(Object::toString)
                        .collect(Collectors.toList());
                //停用时当前应用如果被数据源引用，则不支持关闭
                List<LabelSystemData> systemDataList = labelSystemDataService.getDataByWork(ApplicationTypeEnum.VIRTUAL_APPLICATION.getType(),VirtualApplicationEnum.ATHENA_APP.getPrimaryId(),workCodeList);
                if (CollectionUtils.isNotEmpty(systemDataList)){
                    return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage("error.message.mobile.custom.application.status.close"));
                }
            }

            mobileDatasourceInfoService.updateStatus(updateAppcationStatusReq);
            return ResponseEntityWrapperUtil.wrapperOk(true);
        } catch (Exception e) {
            log.error("update status error. param:{}", updateAppcationStatusReq, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/mobile/application/updateStatus, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }


    /**
     * 删除应用信息
     *
     * @param deleteNewsOrTypeReq 请求入参
     * @return 返回
     */
    @PostMapping("/delApplicationInfo")
    public ResponseEntity<BaseResultDTO<Boolean>> delApplicationInfo(@RequestBody DeleteNewsOrTypeReq deleteNewsOrTypeReq) {
        try {
            if (CollectionUtils.isEmpty(deleteNewsOrTypeReq.getIdList())) {
                return ResponseEntityWrapperUtil.wrapperOk(true);
            }

            QueryWrapper<MobileDatasourceInfo> condition = new QueryWrapper<>();
            condition.in("id", deleteNewsOrTypeReq.getIdList());
            condition.eq("status", Constants.MobileDatasourceInfoStatusEnum.UNUSABLE.getFlag());
            List<MobileDatasourceInfo> customApplicationList = mobileDatasourceInfoService.getBaseMapper().selectList(condition);
            if (CollectionUtils.isNotEmpty(customApplicationList)) {
                deleteNewsOrTypeReq.setIdList(customApplicationList.stream().map(MobileDatasourceInfo::getId).collect(Collectors.toList()));
                mobileDatasourceInfoService.delApplicationInfo(deleteNewsOrTypeReq.getIdList());
            }

            return ResponseEntityWrapperUtil.wrapperOk(true);
        } catch (Exception e) {
            log.error("delete applicationInfo error. param:{}", deleteNewsOrTypeReq, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/mobile/application/delApplicationInfo, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }


    /**
     * 查询管理后台应用详情
     *
     * @param customApplicationIdReq 主键id
     * @return 返回
     */
    @PostMapping("/queryCustomApplicationDetail")
    public ResponseEntity<BaseResultDTO<MobileDatasourceInfoVO>> queryCustomApplicationDetail(@RequestBody @Valid CustomApplicationIdReq customApplicationIdReq) {
        try {
            MobileDatasourceInfo mobileCustomApplication = mobileDatasourceInfoService.getBaseMapper().selectById(customApplicationIdReq.getId());
            if (ObjectUtils.isEmpty(mobileCustomApplication)) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "自定义应用不存在");
            }
            MobileDatasourceInfoVO mobileCustomApplicationVO = mobileDatasourceInfoService.queryCustomApplicationDetail(customApplicationIdReq.getId());
            return ResponseEntityWrapperUtil.wrapperOk(mobileCustomApplicationVO);
        } catch (Exception e) {
            log.error("query news detail error. param:{}", customApplicationIdReq, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/mobile/application/queryCustomApplicationDetail, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 保存应用信息
     *
     * @param saveAppcationReq 入参
     * @return 返回
     */
    @PostMapping("/saveCustomApplication")
    public ResponseEntity<BaseResultDTO<Boolean>> saveCustomApplication(@RequestBody @Valid SaveAppcationReq saveAppcationReq) {
        try {

            if (Constants.MobileDataSourceTye.MOBILE_THIRD_APPLICATION.getValue().equals(saveAppcationReq.getType()) && (ObjectUtils.isEmpty(saveAppcationReq.getApplicationPrimaryId()) || ObjectUtils.isEmpty(saveAppcationReq.getApplicationSource()))){
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, messageUtils.getMessage("error.message.common.param-missing"));
            }

            //type为2-鼎捷非雅典娜新应用时 应用配置必填
            if (Constants.MobileDataSourceTye.DIGIWIN_NOT_ATHENA_APPLICATION.getValue().equals(saveAppcationReq.getType()) && StringUtils.isEmpty(saveAppcationReq.getApplicationConfig())){
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, messageUtils.getMessage("error.message.common.param-missing"));
            }

            //校验名称是否重复
            Integer isExistData = mobileDatasourceInfoService.countByParam(saveAppcationReq);
            if (isExistData > 0) {
                String error = messageUtils.getMessage("error.message.job.bench.name.repeat");
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, error);
            }

            mobileDatasourceInfoService.saveCustomApplication(saveAppcationReq);
            return ResponseEntityWrapperUtil.wrapperOk(true);
        } catch (Exception e) {
            log.error("save news error. param: {}", saveAppcationReq, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/mobile/application/saveCustomApplication, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 列表全量查询
     * @param customApplicationReq
     * @return
     */
    @PostMapping("/queryDatasourceInfo")
    public ResponseEntity<BaseResultDTO<List<MobileDatasourceInfo>>> queryDatasourceInfo(@RequestBody @Valid CustomApplicationsListPageReq customApplicationReq) {
        try {
            customApplicationReq.setStatus(Constants.VALID_STATUS_ENABLE);
            return ResponseEntityWrapperUtil.wrapperOk(mobileDatasourceInfoService.queryDatasourceInfo(customApplicationReq));
        } catch (Exception e) {
            log.error("queryDatasourceInfo error. param:{}", customApplicationReq, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/mobile/application/queryDatasourceInfo, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 根据用户查询商品信息
     * @return
     */
    @PostMapping("/queryApplicationByUser")
    public ResponseEntity<BaseResultDTO<List<UserApplicationSimpleDTO>>> queryApplicationByUser() {
        try {
            return ResponseEntityWrapperUtil.wrapperOk(mobileDatasourceInfoService.queryApplicationByUser());
        } catch (Exception e) {
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/mobile/application/queryApplicationByUser, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 根据商品信息查询模组信息
     * 该方法后台不调用了
     * @return
     */
    @Deprecated
    @PostMapping("/queryModuleByApplication")
    public ResponseEntity<BaseResultDTO<List<UserApplicationModulesDTO>>> queryModuleByApplication(@RequestBody @Valid PermissionUserFunctionalDTO permissionUserFunctionalDTO) {
        try {
            return ResponseEntityWrapperUtil.wrapperOk(mobileDatasourceInfoService.queryModuleByApplication(permissionUserFunctionalDTO));
        } catch (Exception e) {
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/mobile/application/queryModuleByApplication, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 根据商品、模组信息查询作业信息
     * @return
     */
    @PostMapping("/queryWorkByApplication")
    public ResponseEntity<BaseResultDTO<List<PermissionUserFunctionalDTO>>> queryWorkByApplication(@RequestBody @Valid PermissionUserFunctionalDTO permissionUserFunctionalDTO) {
        try {
            return ResponseEntityWrapperUtil.wrapperOk(mobileDatasourceInfoService.queryWorkByApplication(permissionUserFunctionalDTO));
        } catch (Exception e) {
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/mobile/application/queryWorkByApplication, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 更新应用启用状态
     *
     * @param updateAppcationConfigReq 入参
     * @return 返回
     */
    @PostMapping("/updateConfig")
    public ResponseEntity<BaseResultDTO<Boolean>> updateConfig(@RequestBody @Valid UpdateAppcationConfigReq updateAppcationConfigReq) {
        try {

            QueryWrapper<MobileDatasourceInfo> queryWrapper = new QueryWrapper<>();
            queryWrapper.eq("iam_work_id", updateAppcationConfigReq.getIamWorkId());
            queryWrapper.ne("id", updateAppcationConfigReq.getId());
            List<MobileDatasourceInfo> mobileDatasourceInfos = mobileDatasourceInfoMapper.selectList(queryWrapper);
            if (!CollectionUtils.isEmpty(mobileDatasourceInfos)){
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.UPDATE_FAILED, messageUtils.getMessage("error.message.mobile.work.exist"));
            }

            mobileDatasourceInfoService.updateConfig(updateAppcationConfigReq);
            return ResponseEntityWrapperUtil.wrapperOk(true);
        } catch (Exception e) {
            log.error("update status error. param:{}", updateAppcationConfigReq, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/mobile/application/updateConfig, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }


    /**
     * 移动应用作业中选择移动SSO应用的接口
     * @return
     */
    @PostMapping("/applicationDropdownList")
    public ResponseEntity<BaseResultDTO<List<AppLinkDTO>>> applicationDropdownList() {
        try {
            List<AppLinkDTO> appLinkListDTO = appLinkService.queryManageList(AppAuthContextHolder.getContext().getAuthoredUser());
            return ResponseEntityWrapperUtil.wrapperOk( appLinkListDTO.stream().filter(x -> Constants.APP_TYPE_BS.equals(x.getSystemType())).collect(Collectors.toList()));
//            return ResponseEntityWrapperUtil.wrapperOk( appLinkListDTO.stream().filter(x -> Constants.APP_TYPE_BS.equals(x.getSystemType()) && !StringUtils.isEmpty(x.getMobileCallBackUrl())).collect(Collectors.toList()));
        } catch (Exception e) {
            log.error("applicationDropdownList error:", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/mobile/application/applicationDropdownList, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }
}

