package com.digiwin.athena.semc.controller.mobile;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.appcore.util.ResponseEntityWrapper;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.I18NKey;
import com.digiwin.athena.semc.common.ResultPageBean;
import com.digiwin.athena.semc.common.enums.BizAuthTypeEnum;
import com.digiwin.athena.semc.dto.portal.PortalInfoQueryReq;
import com.digiwin.athena.semc.entity.mobile.*;
import com.digiwin.athena.semc.entity.portal.PortalInfo;
import com.digiwin.athena.semc.service.mobile.MobilePortalService;
import com.digiwin.athena.semc.service.portal.PortalInfoService;
import com.digiwin.athena.semc.service.workbench.constant.PublishSourceEnum;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;
import com.digiwin.athena.semc.util.Utils;
import com.digiwin.athena.semc.vo.mobile.MobilePortalInfoAuthResp;
import com.digiwin.athena.semc.vo.mobile.MobilePortalInfoResp;
import com.digiwin.athena.semc.vo.portal.PortalInfoAuthReq;
import io.swagger.v3.oas.annotations.Operation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import javax.validation.Valid;
import java.time.LocalDateTime;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @description 移动门户
 * @author: dengkaiwen
 */
@RestController
@RequestMapping("/semc/mobile/portal/")
@Slf4j
public class MobilePortalWebController {

    @Autowired
    MobilePortalService mobilePortalService;

    @Autowired
    PortalInfoService portalInfoService;

    @Resource
    private MessageUtils messageUtils;


    /**
     * 保存移动门户
     *
     * @param req
     * @return
     */
    @Operation(method = "savePortal", description = "保存移动门户")
    @PostMapping("/info/savePortal")
    public ResponseEntity<?> saveMobilePortal(@RequestBody @Valid MobilePortalInfo req) {
        if (null == req.getPortalType()) {
            req.setPortalType(Constants.MobilePortalTypeEnum.PORTAL.getFlag());
        }
        //校验名称重复
        List<MobilePortalInfo> portalInfoList = mobilePortalService.checkPortal(req);
        if (CollectionUtils.isNotEmpty(portalInfoList)) {
            String error = messageUtils.getMessage("error.message.job.bench.name.repeat");
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, error);
        }
        //校验入参
        ResponseEntity<?> checkResult = checkMobilePortal(req);
        if (null != checkResult) {
            return checkResult;
        }
        try {
            return ResponseEntityWrapper.wrapperOk(mobilePortalService.savePortal(req));
        } catch (Exception e) {
            log.error("mobile savePortal exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/mobile/portal/info/savePortal");
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }

    }

    /**
     * @param req
     * @return
     */
    public ResponseEntity<?> checkMobilePortal(MobilePortalInfo req) {
        List<MobilePortalInfoContent> contentList = req.getContentList();
        if (CollectionUtils.isEmpty(contentList)) {
            return null;
        }
        //任务组件判断最多只有一个
        List<MobilePortalInfoContent> taskDataList = contentList.stream().filter(item -> Constants.PrtalMobileContentTypeEnum.PRE_TASK.getVal().equals(item.getContentType())).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(taskDataList) && taskDataList.size() > 1) {
            String error = messageUtils.getMessage("error.message.mobile.portal.task.can.only.error");
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, error);
        }
        //项目追踪组件判断最多只有一个
        List<MobilePortalInfoContent> projectDataList = contentList.stream().filter(item -> Constants.PrtalMobileContentTypeEnum.PRE_PROJECT.getVal().equals(item.getContentType())).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(projectDataList) && taskDataList.size() > 1) {
            String error = messageUtils.getMessage("error.message.mobile.portal.project.can.only.error");
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, error);
        }
        //金刚区判断
        List<MobilePortalInfoContent> districtList = contentList.stream().filter(item -> Constants.PrtalMobileContentTypeEnum.PRE_KING_KONG_DISTRICT.getVal().equals(item.getContentType())).collect(Collectors.toList());
        int customDistrict = 0;
        for (MobilePortalInfoContent district : districtList) {
            if (district.getCustomType().equals(Constants.DistrictCustomTypeEnum.USER_TYPE.getValue())) {
                customDistrict++;
            }
            //用户自定义金刚区只能有一个
            if (customDistrict > 1) {
                String error = messageUtils.getMessage("error.message.mobile.portal.custom.district.can.only.error");
                return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, error);
            }
            if (Constants.DistrictinteractionTypeEnum.EXPAND_ALL.getValue().equals(district.getInteractionType())) {
                district.setLineNum(null);
            }
        }
        return null;
    }

    /**
     * 查询移动门户列表
     *
     * @param
     * @return
     */
    @Operation(method = "/manage/pageQuery", description = "查询移动门户列表")
    @PostMapping("/manage/pageQuery")
    public ResultPageBean pageQuery(@RequestBody @Valid PortalInfoQueryReq req) {
        return mobilePortalService.pageQuery(req);
    }


    /**
     * 批量发布和取消发布移动门户
     *
     * @param req
     * @return
     */
    @Operation(method = "updateStatus", description = "批量发布和取消发布移动门户")
    @PostMapping("/info/updateStatus")
    public ResponseEntity<?> updateStatus(@RequestBody PortalInfoQueryReq req) {

        if (CollectionUtils.isEmpty(req.getIdList())) {
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "idList is null");
        }
        if (null == req.getStatus()) {
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "status is null");
        }
        try {
            mobilePortalService.updateBat(req);
        } catch (Exception e) {
            log.error("mobile updateStatus exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/mobile/portal/info/updateStatus");
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
        return ResponseEntityWrapper.wrapperOk();
    }


    /**
     * 批量删除布局移动门户
     *
     * @param req
     * @return
     */
    @Operation(method = "batchDel", description = "批量删除布局移动门户")
    @PostMapping("info/batchDel")
    public ResponseEntity<?> batchDel(@RequestBody PortalInfoQueryReq req) {

        if (CollectionUtils.isEmpty(req.getIdList())) {
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "idList is null");
        }
        try {
            mobilePortalService.delBat(req);
        } catch (Exception e) {
            log.error("mobile batchDel exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/mobile/portal/info/batchDel");
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
        return ResponseEntityWrapper.wrapperOk();
    }

    /**
     * 保存移动门户组件权限
     *
     * @param req 请求实体
     * @return
     * @throws Exception
     */
    @Operation(summary = "保存移动门户组件权限")
    @PostMapping("/userAuth/savePortalAuth")
    public ResponseEntity<BaseResultDTO<Integer>> savePortalAuth(@RequestBody @Valid PortalInfoAuthReq req) {
        try {
            return ResponseEntityWrapperUtil.wrapperOk(mobilePortalService.savePortalAuth(req));
        } catch (Exception e) {
            log.error("mobile savePortalAuth exception", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 查询移动门户权限信息
     *
     * @param req 请求实体
     * @return
     * @throws Exception
     */
    @Operation(summary = "查询移动门户权限信息")
    @PostMapping("/userAuth/selectByPortal")
    public ResponseEntity<BaseResultDTO<MobilePortalInfoAuthResp>> selectByPortal(@RequestBody MobilePortalInfoAuth req) {
        if (null == req.getId()) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "id  cannot be empty");
        }
        try {
            return ResponseEntityWrapperUtil.wrapperOk(mobilePortalService.selectByPortalAuth(req.getId()));
        } catch (Exception e) {
            log.error("mobile selectByPortal exception", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 获取移动门户数据详情-管理后台使用
     *
     * @param req
     * @return
     */
    @Operation(method = "getPortalDetail", description = "获取门户数据详情")
    @PostMapping("/info/getPortalDetail")
    public ResponseEntity<BaseResultDTO<MobilePortalInfoResp>> getPortalDetail(@RequestBody MobilePortalInfo req) {
        if (null == req.getId()) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "id  cannot be empty");
        }
        MobilePortalInfo portalInfo = mobilePortalService.getById(req.getId());
        if (null == portalInfo) {
            String recentlyMsg = messageUtils.getMessage("error.message.choose.portal.recovery");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, recentlyMsg);
        }
        try {
            return ResponseEntityWrapperUtil.wrapperOk(mobilePortalService.getMobilePortalDetail(portalInfo));
        } catch (Exception e) {
            log.error("mobile getPortalDetail exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/mobile/portal/info/getPortalDetail");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 获取移动门户预览数据
     *
     * @param req
     * @return
     */
    @Operation(method = "getPreviewPortal", description = "获取门户数据详情")
    @PostMapping("/info/getPreviewPortal")
    public ResponseEntity<BaseResultDTO<MobilePortalPreview>> getPreviewPortal(@RequestBody MobilePortalInfo req) {
        if (null == req.getId()) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "id  cannot be empty");
        }
        try {
            return ResponseEntityWrapperUtil.wrapperOk(mobilePortalService.getPreviewPortal(req.getId()));
        } catch (Exception e) {
            log.error("mobile getPortalDetail exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/mobile/portal/info/getPortalDetail");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }


    /**
     * 预览移动门户
     *
     * @param req
     * @return
     */
    @Operation(method = "previewPortal", description = "预览移动门户")
    @PostMapping("/info/previewPortal")
    public ResponseEntity<BaseResultDTO<MobilePortalPreview>> previewPortal(@RequestBody @Valid MobilePortalInfo req) {
        MobilePortalPreview mobilePortalPreview;
        // 落日工作台需求中门户数据只插入pc端门户表，所以需要查询pc门户表中的移动门户
        if (ObjectUtils.isNotEmpty(req.getOldPortalFlag()) && Boolean.FALSE.equals(req.getOldPortalFlag())) {
            PortalInfo portalInfo = null;
            // 列表页面点击预览，需要查询详情
            if (null != req.getId()) {
                // 租户门户
                if (Constants.LabelOrPoralSourceEnum.TENANT.getFlag().equals(req.getPortalTypeSource())) {
                    portalInfo = portalInfoService.getById(req.getId());
                } else if (Constants.LabelOrPoralSourceEnum.ISV.getFlag().equals(req.getPortalTypeSource())) { // ISV门户
                    portalInfo = portalInfoService.queryPortalInfoByDesignerId(req.getPortalDesignerId());
                }
                if (null == portalInfo) {
                    String recentlyMsg = messageUtils.getMessage("error.message.choose.portal.recovery");
                    return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, recentlyMsg);
                }
                // 查询门户详情
                PortalInfo pcPortalDetail = portalInfoService.getMobilePortalDetail(portalInfo);
                // 保存预览数据
                pcPortalDetail.setVersionDefault(portalInfo.getVersionDefault());
                mobilePortalPreview = mobilePortalService.previewPortal(null, pcPortalDetail);
                mobilePortalPreview.setVersionDefault(portalInfo.getVersionDefault());
                return ResponseEntityWrapperUtil.wrapperOk(mobilePortalPreview);
            }
            mobilePortalPreview = mobilePortalService.previewPortal(req, null);
            mobilePortalPreview.setVersionDefault(1);
        } else {
            if (null != req.getId()) {
                MobilePortalInfo mobilePortalInfo = mobilePortalService.getById(req.getId());
                if (null == mobilePortalInfo) {
                    String recentlyMsg = messageUtils.getMessage("error.message.choose.portal.recovery");
                    return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, recentlyMsg);
                }
                req = mobilePortalInfo;
                req.setOldPortalFlag(true);
            }
            mobilePortalPreview = mobilePortalService.previewPortal(req, null);
            mobilePortalPreview.setVersionDefault(0);
        }
        return ResponseEntityWrapperUtil.wrapperOk(mobilePortalPreview);
    }

    /**
     * 查询全部自定义页面布局列表
     *
     * @return
     */
    @Operation(method = "listAllCustomPortal", description = "查询全部自定义页面布局列表")
    @PostMapping("/manage/listAllCustomPortal")
    public ResponseEntity<BaseResultDTO<List<MobilePortalInfo>>> listAllCustomPortal() {
        // 落日工作台需求中门户数据只插入pc端门户表，所以需要查询pc门户表中的自定义页面门户
        LambdaQueryWrapper<PortalInfo> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(PortalInfo::getPortalType, Constants.MobilePortalTypeEnum.CUSTOM_PORTAL.getFlag());
        queryWrapper.eq(PortalInfo::getTenantId, Utils.getTenantId());
        queryWrapper.orderByDesc(PortalInfo::getModifyTime);
        List<PortalInfo> portalList = portalInfoService.list(queryWrapper);

        // 查询移动表自定义页面门户
        LambdaQueryWrapper<MobilePortalInfo> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(MobilePortalInfo::getPortalType, Constants.MobilePortalTypeEnum.CUSTOM_PORTAL.getFlag());
        wrapper.orderByDesc(MobilePortalInfo::getModifyTime);
        List<MobilePortalInfo> portalInfoList = mobilePortalService.list(wrapper);
        for (MobilePortalInfo mobilePortalInfo : portalInfoList) {
            PortalInfo pcPortalInfo = mobilePortalConvertToPortal(mobilePortalInfo);
            portalList.add(pcPortalInfo);
        }
        return ResponseEntityWrapperUtil.wrapperOk(portalInfoList);
    }

    /**
     * 移动门户转为pc门户
     *
     * @param mobilePortalInfo 移动门户
     * @return 返回
     */
    private static PortalInfo mobilePortalConvertToPortal(MobilePortalInfo mobilePortalInfo) {
        PortalInfo pcPortalInfo = new PortalInfo();
        pcPortalInfo.setId(mobilePortalInfo.getId());
        pcPortalInfo.setPortalTypeSource(Constants.PortalTypeSourceEnum.TENANT.getFlag());
        pcPortalInfo.setClientType(Constants.ClientTypeTypeEnum.MOBILE.getFlag());
        pcPortalInfo.setPortalType(mobilePortalInfo.getPortalType());
        pcPortalInfo.setTitle(mobilePortalInfo.getTitle());
        pcPortalInfo.setBgColor(mobilePortalInfo.getBgColor());
        pcPortalInfo.setPortalDesignerId(null);
        pcPortalInfo.setWorkbenchAppCode(null);
        pcPortalInfo.setParentId(0L);
        pcPortalInfo.setTemplateId(null);
        pcPortalInfo.setDefaultFlag(mobilePortalInfo.getDefaultFlag());
        pcPortalInfo.setName(mobilePortalInfo.getName());
        pcPortalInfo.setPortalDesc(mobilePortalInfo.getPortalDesc());
        pcPortalInfo.setMenuTemplateId(null);
        pcPortalInfo.setStatus(mobilePortalInfo.getStatus());
        pcPortalInfo.setPortalContent(mobilePortalInfo.getPortalContent());
        pcPortalInfo.setPortalImgId(mobilePortalInfo.getPortalImgId());
        pcPortalInfo.setTenantId(mobilePortalInfo.getTenantId());
        pcPortalInfo.setCreateUserName(mobilePortalInfo.getName());
        pcPortalInfo.setCreateTime(mobilePortalInfo.getCreateTime());
        pcPortalInfo.setModifyUserName(mobilePortalInfo.getModifyUserName());
        pcPortalInfo.setModifyTime(mobilePortalInfo.getModifyTime());
        pcPortalInfo.setModifyUserId(mobilePortalInfo.getModifyUserId());
        pcPortalInfo.setPublishSource(PublishSourceEnum.TENANT.name());
        return pcPortalInfo;
    }

    /**
     * 复制门户
     *
     * @param req 请求实体
     * @return
     * @throws Exception
     */
    @Operation(summary = "复制门户")
    @PostMapping("info/copyPortal")
    public ResponseEntity<BaseResultDTO<Integer>> copyPortal(@RequestBody MobilePortalInfo req) {
        try {
            if (null == req.getId()) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "id  cannot be empty");
            }
            MobilePortalInfo portalInfo = mobilePortalService.getById(req.getId());
            if (null == portalInfo) {
                String recentlyMsg = messageUtils.getMessage("error.message.choose.portal.recovery");
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, recentlyMsg);
            }
            req = portalInfo;
            return ResponseEntityWrapperUtil.wrapperOk(mobilePortalService.copyPortal(req));
        } catch (Exception e) {
            log.error("mobile copyPortal exception", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

}
