package com.digiwin.athena.semc.controller.news;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.toolkit.CollectionUtils;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.I18NKey;
import com.digiwin.athena.semc.common.PageInfoResp;
import com.digiwin.athena.semc.common.enums.DataChangeTypeEnum;
import com.digiwin.athena.semc.common.enums.NewsAnnouncementPublishTypeEnum;
import com.digiwin.athena.semc.dto.news.AnnouncementIdReq;
import com.digiwin.athena.semc.dto.news.DeleteNewsOrTypeReq;
import com.digiwin.athena.semc.dto.news.MyNewsFavoritePageReq;
import com.digiwin.athena.semc.dto.news.NewLikeReqDTO;
import com.digiwin.athena.semc.dto.news.NewsCommonIdListReq;
import com.digiwin.athena.semc.dto.news.NewsCommonIdReq;
import com.digiwin.athena.semc.dto.news.NewsStatisticsReqDTO;
import com.digiwin.athena.semc.dto.news.NewsUpdateToppedStatusReq;
import com.digiwin.athena.semc.dto.news.NewsUserNewsDetailReq;
import com.digiwin.athena.semc.dto.news.QueryMobileNewsListPageReq;
import com.digiwin.athena.semc.dto.news.QueryNewsListPageReq;
import com.digiwin.athena.semc.dto.news.SaveAnnouncementReq;
import com.digiwin.athena.semc.dto.news.UpdateNewsStatusReq;
import com.digiwin.athena.semc.dto.news.WebQueryNewsListPageReq;
import com.digiwin.athena.semc.entity.news.NewsAnnouncement;
import com.digiwin.athena.semc.event.NewsAuthBizIdSetEvent;
import com.digiwin.athena.semc.event.NewsChangeEvent;
import com.digiwin.athena.semc.event.dto.NewsChangeEventDTO;
import com.digiwin.athena.semc.service.news.NewsAnnouncementFavoriteService;
import com.digiwin.athena.semc.service.news.NewsAnnouncementLikeService;
import com.digiwin.athena.semc.service.news.NewsAnnouncementReadService;
import com.digiwin.athena.semc.service.news.NewsAnnouncementService;
import com.digiwin.athena.semc.util.DateUtils;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;
import com.digiwin.athena.semc.util.TrackingUtil;
import com.digiwin.athena.semc.vo.news.MyNewsAnnouncementFavoriteVO;
import com.digiwin.athena.semc.vo.news.NewsAnnouncementStatisticsVO;
import com.digiwin.athena.semc.vo.news.NewsAnnouncementVO;
import com.digiwin.athena.semc.vo.news.ValidateNewsVO;

import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.Resource;
import javax.validation.Valid;

import lombok.extern.slf4j.Slf4j;

/**
 * 新闻公告表(TNewsAnnouncement)表控制层
 *
 * @author sunqgz
 * @since 2024-05-13
 */
@Slf4j
@RestController
@RequestMapping("/semc/news")
public class NewsAnnouncementController {

    /**
     * 服务对象
     */
    @Resource
    private NewsAnnouncementService newsAnnouncementService;

    @Resource
    private NewsAnnouncementReadService newsAnnouncementReadService;


    @Resource
    private TrackingUtil trackingUtil;

    @Resource
    private MessageUtils messageUtils;

    @Resource
    private ApplicationEventPublisher eventPublisher;


    @Resource
    private NewsAnnouncementLikeService newsAnnouncementLikeService;

    @Resource
    private NewsAnnouncementFavoriteService newsAnnouncementFavoriteService;

    /**
     * 公告数量最大限制
     */
    public static final int MAX_NEWS_COUNT = 1000;

    /**
     * 公告附件数量最大限制
     */
    public static final int MAX_NEWS_FILE_COUNT = 5;

    /**
     * 管理后台分页查询公告列表
     *
     * @param queryNewsListPageReq 入参
     * @return 查询结果
     */
    @PostMapping("/queryManagerNewsByPage")
    public ResponseEntity<BaseResultDTO<PageInfoResp<NewsAnnouncementVO>>> queryManagerNewsByPage(@RequestBody @Valid QueryNewsListPageReq queryNewsListPageReq) {
        try {
            PageInfoResp<NewsAnnouncementVO> pageInfoResp = newsAnnouncementService.queryNewsListByPage(queryNewsListPageReq);

            // 权限表业务对象id赋值事件，为了刷历史数据
            // TODO: 2025/4/10  历史数据刷完删除该事件
            NewsAuthBizIdSetEvent newsAuthBizIdSetEvent = new NewsAuthBizIdSetEvent(this, null);
            eventPublisher.publishEvent(newsAuthBizIdSetEvent);
            return ResponseEntityWrapperUtil.wrapperOk(pageInfoResp);
        } catch (Exception e) {
            log.error("query manager news by page error. param:{}", queryNewsListPageReq, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/news/queryManagerNewsByPage, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }


    /**
     * 管理后台公告统计信息
     *
     * @param newsStatisticsReqDTO
     * @return
     */
    @PostMapping("/queryManagerNewsStatistics")
    public ResponseEntity<BaseResultDTO<NewsAnnouncementStatisticsVO>> queryManagerNewsStatistics(@RequestBody @Valid NewsStatisticsReqDTO newsStatisticsReqDTO) {
        return ResponseEntityWrapperUtil.wrapperOk(newsAnnouncementService.queryManagerNewsStatistics(newsStatisticsReqDTO));
    }


    /**
     * 更新公告发布状态
     *
     * @param updateNewsStatusReq 入参
     * @return 返回
     */
    @PostMapping("/updatePublishStatus")
    public ResponseEntity<BaseResultDTO<Boolean>> updatePublishStatus(@RequestBody @Valid UpdateNewsStatusReq updateNewsStatusReq) {
        List<Long> idList = updateNewsStatusReq.getIdList();
        List<NewsAnnouncement> newsAnnouncementList = newsAnnouncementService.getBaseMapper().selectBatchIds(idList);

        // 更新
        newsAnnouncementService.updatePublishStatus(updateNewsStatusReq, newsAnnouncementList);
        trackingUtil.reportUserTrackForUpdatePublishStatus(updateNewsStatusReq);
        return ResponseEntityWrapperUtil.wrapperOk(true);
    }

    /**
     * 更新公告置顶状态
     *
     * @param announcementIdReq 入参
     * @return 返回
     */
    @PostMapping("/updateToppedStatus")
    public ResponseEntity<BaseResultDTO<Boolean>> updateToppedStatus(@RequestBody @Valid NewsUpdateToppedStatusReq announcementIdReq) {
        if (ObjectUtils.isEmpty(announcementIdReq.getToppedFlag())) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "是否置顶标识不能为空");
        }
        NewsAnnouncement newsAnnouncement = newsAnnouncementService.getBaseMapper().selectById(announcementIdReq.getId());
        if (newsAnnouncement == null) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "该公告不存在");
        }
        // 失效公告不能置顶
        if (Constants.TOPPED_FLAG_YES.equals(announcementIdReq.getToppedFlag())) {
            if (Constants.NewsAnnouncementStatusEnum.EXPIRED.getFlag().equals(newsAnnouncement.getNewsStatus())) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "该公告已失效，不能置顶");
            }
            if (new Date().compareTo(DateUtils.StringToDate(newsAnnouncement.getNewsEndTime())) > 0) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "该公告已失效，不能置顶");
            }
        }
        newsAnnouncementService.updateToppedStatus(announcementIdReq.getId(), announcementIdReq.getToppedFlag());
        return ResponseEntityWrapperUtil.wrapperOk(true);
    }

    /**
     * 删除公告
     *
     * @param deleteNewsOrTypeReq 请求入参
     * @return 返回
     */
    @PostMapping("/delNews")
    public ResponseEntity<BaseResultDTO<Boolean>> delNews(@RequestBody DeleteNewsOrTypeReq deleteNewsOrTypeReq) {
        if (CollectionUtils.isEmpty(deleteNewsOrTypeReq.getIdList())) {
            return ResponseEntityWrapperUtil.wrapperOk(true);
        }
        List<NewsAnnouncement> newsAnnouncementList = newsAnnouncementService.getBaseMapper().selectBatchIds(deleteNewsOrTypeReq.getIdList());

        // 删除公告
        newsAnnouncementService.delNews(deleteNewsOrTypeReq.getIdList());

        // 发送事件，异步更新公告类型关联的公告总数
        List<Long> newsTypeIdList = newsAnnouncementList.stream().map(NewsAnnouncement::getNewsTypeId).filter(ObjectUtils::isNotEmpty).distinct().collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(newsTypeIdList)) {
            NewsChangeEventDTO eventDTO = new NewsChangeEventDTO();
            eventDTO.setDataChangeType(DataChangeTypeEnum.DELETE);
            eventDTO.setOldNewsTypeIdList(newsTypeIdList);
            NewsChangeEvent newsChangeEvent = new NewsChangeEvent(this, eventDTO);
            eventPublisher.publishEvent(newsChangeEvent);
        }
        return ResponseEntityWrapperUtil.wrapperOk(true);
    }

    /**
     * 移动端/web端-校验公告的是否存在/是否有效
     *
     * @param announcementIdReq 主键id
     * @return 返回
     */
    @PostMapping("/validateNews")
    public ResponseEntity<BaseResultDTO<ValidateNewsVO>> validateNews(@RequestBody @Valid NewsCommonIdReq announcementIdReq) {
        return ResponseEntityWrapperUtil.wrapperOk(newsAnnouncementService.validateNews(announcementIdReq.getId()));
    }

    /**
     * 查询管理后台公告详情
     *
     * @param announcementIdReq 主键id
     * @return 返回
     */
    @PostMapping("/queryManagerNewsDetail")
    public ResponseEntity<BaseResultDTO<NewsAnnouncementVO>> queryManagerNewsDetail(@RequestBody @Valid NewsCommonIdReq announcementIdReq) {
        NewsAnnouncementVO newsAnnouncementVO = newsAnnouncementService.queryManagerNewsDetail(announcementIdReq.getId());
        if (newsAnnouncementVO == null) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "公告不存在");
        }
        return ResponseEntityWrapperUtil.wrapperOk(newsAnnouncementVO);
    }

    /**
     * 保存公告
     *
     * @param saveAnnouncementReq 入参
     * @return 返回
     */
    @PostMapping("/saveNews")
    public ResponseEntity<BaseResultDTO<Boolean>> saveNews(@RequestBody @Valid SaveAnnouncementReq saveAnnouncementReq) {
        boolean isAdd = ObjectUtils.isEmpty(saveAnnouncementReq.getId());
        // 新增
        if (isAdd) {
            Long newCount = newsAnnouncementService.getBaseMapper().selectCount(Wrappers.emptyWrapper());
            if (newCount >= MAX_NEWS_COUNT) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "最大可新增" + MAX_NEWS_COUNT + "条公告，目前已达上限");
            }
        }
        // 附件数量校验
        if (CollectionUtils.isNotEmpty(saveAnnouncementReq.getFileIdList()) && saveAnnouncementReq.getFileIdList().size() > MAX_NEWS_FILE_COUNT) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "附件数量不能超过" + MAX_NEWS_FILE_COUNT + "个");
        }
        // 生效时间
        if (StringUtils.isNotBlank(saveAnnouncementReq.getNewsEffectiveTime())) {
            if (!DateUtils.validateDate(saveAnnouncementReq.getNewsEffectiveTime(), DateUtils.DATE_TIME_NORMAL_FORMATTER)) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "生效时间格式不正确");
            }
            if (saveAnnouncementReq.getNewsEffectiveTime().compareTo(saveAnnouncementReq.getNewsEndTime()) > 0) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "生效时间必须小于截止时间");
            }
        } else {
            if (Constants.NewsAnnouncementStatusEnum.PUBLISHED.getFlag().equals(saveAnnouncementReq.getNewsStatus())) {
                saveAnnouncementReq.setNewsEffectiveTime(DateUtils.getNowTime(null));
            } else {
                saveAnnouncementReq.setNewsEffectiveTime(null);
            }
        }
        // 截止时间
        if (!DateUtils.validateDate(saveAnnouncementReq.getNewsEndTime(), DateUtils.DATE_TIME_NORMAL_FORMATTER)) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "截止时间格式不正确");
        }
        // 截止日期必须大于当前时间+24h
        String tomorrow = DateUtils.addDate(DateUtils.getNowTime(null), DateUtils.DATE_TIME_NORMAL_FORMATTER, 1, DateUtils.DAY);
        if (saveAnnouncementReq.getNewsEndTime().compareTo(tomorrow) < 0) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "截止日期必须大于当前时间+24h");
        }
        //发布人校验
        if (NewsAnnouncementPublishTypeEnum.LOGIN_NAME.getCode().equals(saveAnnouncementReq.getPublishType()) || NewsAnnouncementPublishTypeEnum.ROLE_NAME.getCode().equals(saveAnnouncementReq.getPublishType())) {
            if (StringUtils.isBlank(saveAnnouncementReq.getPublishId()) || StringUtils.isBlank(saveAnnouncementReq.getPublishName())) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "发布人角色不能为空");
            }
        }
        newsAnnouncementService.saveNews(saveAnnouncementReq);
        // 增加埋点
        trackingUtil.reportUserTrackForSaveNews(saveAnnouncementReq);
        return ResponseEntityWrapperUtil.wrapperOk(true);
    }

    /**
     * web端-前台查询用户公告列表
     *
     * @return 查询结果
     */
    @PostMapping("/queryUserNewsList")
    public ResponseEntity<BaseResultDTO<PageInfoResp<NewsAnnouncementVO>>> queryUserNewsList(@RequestBody @Valid WebQueryNewsListPageReq queryNewsListPageReq) {
        PageInfoResp<NewsAnnouncementVO> pageInfoResp = newsAnnouncementService.queryUserNewsList(queryNewsListPageReq);
        return ResponseEntityWrapperUtil.wrapperOk(pageInfoResp);
    }

    /**
     * web端/移动端-读取公告，记录为已读公告
     *
     * @param newsCommonIdReq 公告id
     * @return 返回
     */
    @PostMapping("/readNews")
    public ResponseEntity<BaseResultDTO<Boolean>> readNews(@RequestBody @Valid NewsCommonIdReq newsCommonIdReq) {
        AnnouncementIdReq announcementIdReq = new AnnouncementIdReq();
        announcementIdReq.setId(newsCommonIdReq.getId());
        NewsAnnouncement newsAnnouncement = newsAnnouncementService.queryUserNewsById(newsCommonIdReq.getId());
        if (Constants.ThirdNewsAnnouncementEnum.OUTSIDER_INFORMATION.getType().equals(newsAnnouncement.getNewsSource())) {
            announcementIdReq.setThirdAppId(newsAnnouncement.getThirdAppId());
            announcementIdReq.setThirdNewsId(newsAnnouncement.getThirdNewsId());
            announcementIdReq.setThirdAppCode(newsAnnouncement.getThirdAppCode());
        }
        // 新增读取记录
        boolean flag = newsAnnouncementReadService.insertNews(announcementIdReq);
        return ResponseEntityWrapperUtil.wrapperOk(flag);
    }

    /**
     * 公告点赞
     *
     * @param newLikeReqDTO
     * @return
     */
    @PostMapping("/likeNews")
    public ResponseEntity<BaseResultDTO<Integer>> likeNews(@RequestBody @Valid NewLikeReqDTO newLikeReqDTO) {
        return ResponseEntityWrapperUtil.wrapperOk(newsAnnouncementLikeService.like(newLikeReqDTO, true));
    }

    /**
     * 公告取消点赞
     */
    @PostMapping("/cancelLikeNews")
    public ResponseEntity<BaseResultDTO<Integer>> cancelLikeNews(@RequestBody @Valid NewLikeReqDTO newLikeReqDTO) {
        return ResponseEntityWrapperUtil.wrapperOk(newsAnnouncementLikeService.like(newLikeReqDTO, false));
    }


    /**
     * 公告收藏
     * 0：表示收藏成功，1：表示重复收藏，2：公告不存在或者无权限，3：收藏失败
     */
    @PostMapping("/favoriteNews")
    public ResponseEntity<BaseResultDTO<Integer>> favoriteNews(@RequestBody @Valid NewsCommonIdReq newsCommonIdReq) {
        return ResponseEntityWrapperUtil.wrapperOk(newsAnnouncementFavoriteService.favoriteNews(newsCommonIdReq));
    }


    /**
     * 公告取消收藏
     * 0：表示取消收藏成功，1：表示没有收藏记录（失败），2：取消收藏失败
     */
    @PostMapping("/cancelFavoriteNews")
    public ResponseEntity<BaseResultDTO<Integer>> cancelFavoriteNews(@RequestBody @Valid NewsCommonIdListReq newsCommonIdListReq) {
        return ResponseEntityWrapperUtil.wrapperOk(newsAnnouncementFavoriteService.cancelFavoriteNews(newsCommonIdListReq));
    }


    /**
     * 我的公告收藏列表
     *
     * @return
     */
    @PostMapping("/myFavoriteNewsList")
    public ResponseEntity<BaseResultDTO<PageInfoResp<MyNewsAnnouncementFavoriteVO>>> myFavoriteNewsList(@RequestBody MyNewsFavoritePageReq myNewsFavoritePageReq) {
        return ResponseEntityWrapperUtil.wrapperOk(newsAnnouncementFavoriteService.myFavoriteNewsList(myNewsFavoritePageReq));
    }


    /**
     * web端/移动端-查询用户端公告详情
     *
     * @param announcementIdReq 主键id
     * @return 返回
     */
    @PostMapping("/queryUserNewsDetail")
    public ResponseEntity<BaseResultDTO<NewsAnnouncementVO>> queryUserNewsDetail(@RequestBody @Valid NewsUserNewsDetailReq announcementIdReq) {
        NewsAnnouncement newsAnnouncement = newsAnnouncementService.queryUserNewsById(announcementIdReq.getId());
        if (ObjectUtils.isEmpty(newsAnnouncement)) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "公告不存在");
        }
        if (Arrays.asList(Constants.NewsAnnouncementStatusEnum.UNPUBLISHED.getFlag(), Constants.NewsAnnouncementStatusEnum.EXPIRED.getFlag()).contains(newsAnnouncement.getNewsStatus())
                || LocalDateTime.now().compareTo(LocalDateTime.parse(newsAnnouncement.getNewsEndTime(), DateTimeFormatter.ofPattern(DateUtils.DATE_TIME_NORMAL_FORMATTER))) > 0) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "公告已失效");
        }
        NewsAnnouncementVO newsAnnouncementVO = newsAnnouncementService.queryUserNewsDetail(newsAnnouncement, announcementIdReq.getCallSource());
        return ResponseEntityWrapperUtil.wrapperOk(newsAnnouncementVO);
    }

    /**
     * web端-查询用户的未读公告数
     *
     * @return 返回
     */
    @PostMapping("/queryNewsUnReadCount")
    public ResponseEntity<BaseResultDTO<Integer>> queryNewsUnReadCount() {
        Integer count = newsAnnouncementService.queryNewsUnReadCount();
        return ResponseEntityWrapperUtil.wrapperOk(count);
    }

    /**
     * 查询当前租户的系统来源
     *
     * @return 返回
     */
    @PostMapping("/queryNewsAppList")
    public ResponseEntity<BaseResultDTO<List<JSONObject>>> queryNewsAppList() {
        return ResponseEntityWrapperUtil.wrapperOk(newsAnnouncementService.queryNewsAppList());
    }

    /**
     * 移动端-前台分页查询用户公告列表
     *
     * @param queryMobileNewsListPageReq 请求参数
     * @return 查询结果
     */
    @PostMapping("/queryMobileNewsListPage")
    public ResponseEntity<BaseResultDTO<PageInfoResp<NewsAnnouncementVO>>> queryMobileNewsListPage(@RequestBody @Valid QueryMobileNewsListPageReq queryMobileNewsListPageReq) {
        // 排序规则字段校验
        if (StringUtils.isNotBlank(queryMobileNewsListPageReq.getOrderByRule()) && !Arrays.asList(Constants.asc, Constants.desc).contains(queryMobileNewsListPageReq.getOrderByRule())) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "排序规则不正确");
        }
        // 时间格式校验
        if ((StringUtils.isNotBlank(queryMobileNewsListPageReq.getEffectiveTimeStart()) && !DateUtils.validateDate(queryMobileNewsListPageReq.getEffectiveTimeStart(), DateUtils.DATE_TIME_NORMAL_FORMATTER))
                || (StringUtils.isNotBlank(queryMobileNewsListPageReq.getEffectiveTimeEnd()) && !DateUtils.validateDate(queryMobileNewsListPageReq.getEffectiveTimeEnd(), DateUtils.DATE_TIME_NORMAL_FORMATTER))) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "时间格式不正确");
        }
        PageInfoResp<NewsAnnouncementVO> pageInfoResp = newsAnnouncementService.queryMobileNewsListPage(queryMobileNewsListPageReq);
        return ResponseEntityWrapperUtil.wrapperOk(pageInfoResp);
    }
}

