package com.digiwin.athena.semc.controller.news;


import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.I18NKey;
import com.digiwin.athena.semc.common.NewsTypeConstants;
import com.digiwin.athena.semc.dto.news.AnnouncementIdReq;
import com.digiwin.athena.semc.dto.news.DeleteNewsOrTypeReq;
import com.digiwin.athena.semc.dto.news.QueryAnnouncementTypeReq;
import com.digiwin.athena.semc.dto.news.SaveNewsTypeReqDTO;
import com.digiwin.athena.semc.dto.news.ValidSaveNewsTypeReqDTO;
import com.digiwin.athena.semc.entity.news.NewsAnnouncement;
import com.digiwin.athena.semc.entity.news.NewsAnnouncementAut;
import com.digiwin.athena.semc.entity.news.NewsAnnouncementRead;
import com.digiwin.athena.semc.entity.news.NewsAnnouncementType;
import com.digiwin.athena.semc.mapper.news.NewsAnnouncementMapper;
import com.digiwin.athena.semc.mapper.news.NewsAnnouncementReadMapper;
import com.digiwin.athena.semc.service.news.NewsAnnouncementAuthService;
import com.digiwin.athena.semc.service.news.NewsAnnouncementService;
import com.digiwin.athena.semc.service.news.NewsAnnouncementTypeAuthService;
import com.digiwin.athena.semc.service.news.NewsAnnouncementTypeService;
import com.digiwin.athena.semc.util.DateUtils;
import com.digiwin.athena.semc.util.InterceptorIgnoreUtil;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;
import com.digiwin.athena.semc.vo.news.NewsAnnouncementTypeVO;
import com.digiwin.athena.semc.vo.news.NewsTypeAuthVO;
import com.digiwin.athena.semc.vo.news.ValidSaveNewsTypeVO;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.time.LocalDateTime;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.Resource;
import javax.validation.Valid;

import io.github.linpeilie.Converter;
import lombok.extern.slf4j.Slf4j;

/**
 * 新闻公告类型表(TNewsAnnouncementType)表控制层
 *
 * @author sungqz
 * @since 2024-05-13
 */
@Slf4j
@RestController
@RequestMapping("/semc/news/type")
public class NewsAnnouncementTypeController {

    /**
     * 服务对象
     */
    @Resource
    private NewsAnnouncementTypeService newsAnnouncementTypeService;

    @Resource
    private MessageUtils messageUtils;

    @Resource
    private Converter converter;

    @Resource
    private NewsAnnouncementAuthService newsAnnouncementAuthService;

    @Resource
    private NewsAnnouncementTypeAuthService newsAnnouncementTypeAuthService;

    @Resource
    private NewsAnnouncementMapper newsAnnouncementMapper;

    @Resource
    private NewsAnnouncementService newsAnnouncementService;

    @Resource
    private NewsAnnouncementReadMapper newsAnnouncementReadMapper;

    /**
     * 查询公告类型列表
     *
     * @param queryAnnouncementTypeReq 请求参数
     * @return 查询结果
     */
    @PostMapping("/queryNewsTypeList")
    public ResponseEntity<BaseResultDTO<List<NewsAnnouncementTypeVO>>> queryNewsTypeList(@RequestBody QueryAnnouncementTypeReq queryAnnouncementTypeReq) {
        try {
            List<NewsAnnouncementTypeVO> newsTypeList = converter.convert(newsAnnouncementTypeService.queryNewsTypeListByParentId(null), NewsAnnouncementTypeVO.class);
            // 预设组件名支持多语言
            String locale = LocaleContextHolder.getLocale().toString();
            if (Constants.ZH_TW_LOCALE.equals(locale)) {
                newsTypeList.forEach(x -> {
                    if (StringUtils.isNotEmpty(x.getNameTw())) {
                        x.setName(x.getNameTw());
                    }
                    if (StringUtils.isNotEmpty(x.getDescriptionTw())) {
                        x.setDescription(x.getDescriptionTw());
                    }
                });
            }
            //赋值有权限的公告
            Map<Long, List<NewsAnnouncement>> newsTypeNewsMap = Maps.newHashMap();
            List<NewsAnnouncementRead> newsAnnouncementReadList = Lists.newArrayList();
            if (Boolean.TRUE.equals(queryAnnouncementTypeReq.getQueryAuthNewsCount())) {
                List<NewsAnnouncementAut> autList = newsAnnouncementAuthService.queryAuthNewsList();
                if (CollectionUtils.isNotEmpty(autList)) {
                    // 公告id去重
                    List<Long> newsIdDistinctList = autList.stream().map(NewsAnnouncementAut::getNewsId).distinct().collect(Collectors.toList());
                    // 查询用户的公告列表
                    List<NewsAnnouncement> newsAnnouncementList = newsAnnouncementMapper.selectNewsAnnouncementIdList(Constants.NewsAnnouncementStatusEnum.PUBLISHED.getFlag(), newsIdDistinctList, null, DateUtils.getNowTime(null));
                    // 添加三方公告
                    newsAnnouncementList.addAll(newsAnnouncementService.queryThirdNewsList(null));
                    newsTypeNewsMap = newsAnnouncementList.stream().collect(Collectors.groupingBy(NewsAnnouncement::getNewsTypeId, Collectors.toList()));
                    //查询已读公告记录
                    if (CollectionUtils.isNotEmpty(newsAnnouncementList)) {
                        LambdaQueryWrapper<NewsAnnouncementRead> readCondition = new LambdaQueryWrapper<>();
                        readCondition.in(NewsAnnouncementRead::getNewsId, newsAnnouncementList.stream().map(NewsAnnouncement::getId).collect(Collectors.toList()));
                        readCondition.eq(NewsAnnouncementRead::getCreateUserId, AppAuthContextHolder.getContext().getAuthoredUser().getUserId());
                        readCondition.select(NewsAnnouncementRead::getNewsId);
                        newsAnnouncementReadList = newsAnnouncementReadMapper.selectList(readCondition);
                    }
                }
            }
            //子级处理
            Map<Integer, List<NewsAnnouncementTypeVO>> newsTypeMap = newsTypeList.stream().collect(Collectors.groupingBy(NewsAnnouncementTypeVO::getLevel));
            List<NewsAnnouncementTypeVO> oneNewTypeList = newsTypeMap.get(1);
            softNewTypeList(oneNewTypeList);
            //其他类型公告沉底
            List<NewsAnnouncementTypeVO> other = oneNewTypeList.stream().filter(x -> NewsTypeConstants.NEWS_TYPE_OTHER_ID.equals(x.getId())).collect(Collectors.toList());
            oneNewTypeList.removeIf(x -> NewsTypeConstants.NEWS_TYPE_OTHER_ID.equals(x.getId()));
            oneNewTypeList.addAll(other);
            //循环公告一级二级进行处理赋值
            List<NewsAnnouncementTypeVO> twoNewTypeList = newsTypeMap.get(2);
            for (NewsAnnouncementTypeVO newsTypeVO : oneNewTypeList) {
                int sumTwoAuthNewsCount = 0;
                int sumTwoAuthUnReadNewsCount = 0;
                if (CollectionUtils.isNotEmpty(twoNewTypeList)) {
                    List<NewsAnnouncementTypeVO> collect = twoNewTypeList.stream().filter(m -> m.getParentId().equals(newsTypeVO.getId())).collect(Collectors.toList());
                    softNewTypeList(collect);
                    //赋值二级公告有权限数
                    for (NewsAnnouncementTypeVO newsTypeVO2 : collect) {
                        List<NewsAnnouncement> twoAuthNewsList = newsTypeNewsMap.getOrDefault(newsTypeVO2.getId(), Lists.newArrayList());
                        int twoAuthNewsCount = twoAuthNewsList.size();
                        newsTypeVO2.setAuthNewsCount(twoAuthNewsCount);
                        sumTwoAuthNewsCount += twoAuthNewsCount;
                        //已读公告
                        long twoNewsReadCount = newsAnnouncementReadList.stream().filter(m -> existNews(twoAuthNewsList, m.getNewsId())).count();
                        newsTypeVO2.setAuthUnReadNewsCount(twoAuthNewsCount - (int) twoNewsReadCount);
                        sumTwoAuthUnReadNewsCount += newsTypeVO2.getAuthUnReadNewsCount();
                        // 赋值父级类型名称
                        newsTypeVO2.setParentName(newsTypeVO.getName());
                    }
                    newsTypeVO.setChildrenList(collect);
                }
                //说明公告都挂在二级分类下
                if (sumTwoAuthNewsCount > 0) {
                    newsTypeVO.setAuthNewsCount(sumTwoAuthNewsCount);
                }
                //说明公告都挂在一级分类下
                else {
                    List<NewsAnnouncement> oneAuthNewsList = newsTypeNewsMap.getOrDefault(newsTypeVO.getId(), Lists.newArrayList());
                    newsTypeVO.setAuthNewsCount(oneAuthNewsList.size());
                }
                if (sumTwoAuthUnReadNewsCount > 0) {
                    newsTypeVO.setAuthUnReadNewsCount(sumTwoAuthUnReadNewsCount);
                } else {
                    List<NewsAnnouncement> oneAuthNewsList = newsTypeNewsMap.getOrDefault(newsTypeVO.getId(), Lists.newArrayList());
                    long oneNewsReadCount = newsAnnouncementReadList.stream().filter(m -> existNews(oneAuthNewsList, m.getNewsId())).count();
                    newsTypeVO.setAuthUnReadNewsCount(oneAuthNewsList.size() - (int) oneNewsReadCount);
                }
            }
            return ResponseEntityWrapperUtil.wrapperOk(oneNewTypeList);
        } catch (Exception e) {
            log.error("query news type list error. param:{}", queryAnnouncementTypeReq, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/news/type/queryNewsTypeList, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }


    private boolean existNews(List<NewsAnnouncement> newsAnnouncementList, Long newsId) {
        return newsAnnouncementList.stream().anyMatch(x -> x.getId().equals(newsId));
    }


    /**
     * 公告排序
     *
     * @param newsTypeList
     * @return
     */
    private void softNewTypeList(List<NewsAnnouncementTypeVO> newsTypeList) {
        if (CollectionUtils.isEmpty(newsTypeList)) {
            return;
        }
        //按优先级降序，相同按创建时间升序，相同按名称升序
        newsTypeList.sort(Comparator.comparing(NewsAnnouncementTypeVO::getSort, Comparator.reverseOrder())
                .thenComparing(NewsAnnouncementTypeVO::getId, Comparator.naturalOrder())
                .thenComparing(NewsAnnouncementTypeVO::getName, Comparator.naturalOrder()));
    }

    /**
     * 查询管理后台公告类型列表
     *
     * @param queryAnnouncementTypeReq 请求参数
     * @return 查询结果
     */
    @PostMapping("/queryManageNewsTypeList")
    public ResponseEntity<BaseResultDTO<List<NewsAnnouncementTypeVO>>> queryManageNewsTypeList(@RequestBody QueryAnnouncementTypeReq queryAnnouncementTypeReq) {
        List<NewsAnnouncementTypeVO> newsTypeRespList = Lists.newArrayList();
        // 只查询一级公告类型
        Long parentId = null;
        if (queryAnnouncementTypeReq.getOnlyFirstFlag()) {
            parentId = NewsTypeConstants.NEWS_TYPE_FIRST_LEVEL_PARENT_ID;
        }

        // 查询有权限的公告类型
        List<NewsAnnouncementType> newsTypeList;
        if (queryAnnouncementTypeReq.getOnlyAuthFlag()) {
            NewsTypeAuthVO newsTypeAuthVO = newsAnnouncementTypeAuthService.queryAuthRelNewsType(null);
            // 如果没有全部类型的权限，则查询有权限的公告类型
            if (Constants.ALL_AUTH_FLAG_NO.equals(newsTypeAuthVO.getAllNewsTypeFlag())) {
                List<Long> authorizedNewsTypeIdList = newsTypeAuthVO.getAuthorizedNewsTypeIdList();
                // 查询当前公告类型的父级公告类型
                newsTypeList = InterceptorIgnoreUtil.handler(() -> newsAnnouncementTypeService.getBaseMapper().selectBatchIds(authorizedNewsTypeIdList));
                List<Long> parentIdList = newsTypeList.stream().map(NewsAnnouncementType::getParentId)
                        .filter(x -> !NewsTypeConstants.NEWS_TYPE_FIRST_LEVEL_PARENT_ID.equals(x)).collect(Collectors.toList());
                if (CollectionUtils.isNotEmpty(parentIdList)) {
                    List<NewsAnnouncementType> newsAnnouncementTypeList = InterceptorIgnoreUtil.handler(() -> newsAnnouncementTypeService.getBaseMapper().selectBatchIds(parentIdList));
                    newsTypeList.addAll(newsAnnouncementTypeList);
                }
            } else {
                newsTypeList = newsAnnouncementTypeService.queryNewsTypeListByParentId(parentId);
            }
        } else {
            newsTypeList = newsAnnouncementTypeService.queryNewsTypeListByParentId(parentId);
        }
        if (CollectionUtils.isEmpty(newsTypeList)) {
            return ResponseEntityWrapperUtil.wrapperOk(newsTypeRespList);
        }

        // 查询默认公告类型关联的公告，用于计算数量
        Map<Long, List<NewsAnnouncement>> defaultNewsTypeIdMap = queryNewsListWithDefaultType(newsTypeList);

        // 构建一二级类型结构
        Map<Long, List<NewsAnnouncementTypeVO>> parentIdMap = newsTypeList.stream().filter(x -> !NewsTypeConstants.NEWS_TYPE_FIRST_LEVEL_PARENT_ID.equals(x.getParentId()))
                .map(x -> converter.convert(x, NewsAnnouncementTypeVO.class)).collect(Collectors.groupingBy(NewsAnnouncementTypeVO::getParentId));
        // 所有一级类型公告
        List<NewsAnnouncementType> parentNewsTypeList = newsTypeList.stream().filter(x -> NewsTypeConstants.NEWS_TYPE_FIRST_LEVEL_PARENT_ID.equals(x.getParentId())).collect(Collectors.toList());
        for (NewsAnnouncementType parentNewsType : parentNewsTypeList) {
            NewsAnnouncementTypeVO newsTypeRespVO = converter.convert(parentNewsType, NewsAnnouncementTypeVO.class);
            newsTypeRespList.add(newsTypeRespVO);

            // 默认公告类型关联的公告数量
            if (NewsTypeConstants.NEWS_TYPE_DEFAULT_FLAG.equals(parentNewsType.getDefaultFlag())) {
                List<NewsAnnouncement> newsAnnouncementList = defaultNewsTypeIdMap.get(parentNewsType.getId());
                if (CollectionUtils.isNotEmpty(newsAnnouncementList)) {
                    Integer totalCount = newsAnnouncementList.size();
                    Long publishedCount = newsAnnouncementList.stream().filter(x -> Constants.NewsAnnouncementStatusEnum.PUBLISHED.getFlag().equals(x.getNewsStatus())).count();
                    newsTypeRespVO.setNewsCount(totalCount);
                    newsTypeRespVO.setPublishedNewsCount(publishedCount.intValue());
                } else {
                    newsTypeRespVO.setNewsCount(0);
                    newsTypeRespVO.setPublishedNewsCount(0);
                }
            }

            // 二级公告类型
            List<NewsAnnouncementTypeVO> newsTypeVOList = parentIdMap.get(parentNewsType.getId());
            if (CollectionUtils.isNotEmpty(newsTypeVOList)) {
                // 赋值父级类型名称
                newsTypeVOList.forEach(x -> x.setParentName(newsTypeRespVO.getName()));
                // 子级排序
                softNewTypeList(newsTypeVOList);
                newsTypeRespVO.setChildrenList(newsTypeVOList);
            }
        }
        // 父级排序
        softNewTypeList(newsTypeRespList);

        // 默认公告类型的排序置为空，不能修改
        newsTypeRespList.stream().filter(x -> NewsTypeConstants.NEWS_TYPE_DEFAULT_FLAG.equals(x.getDefaultFlag())).forEach(x -> x.setSort(null));
        // "其他"类型排在最后
        NewsAnnouncementTypeVO other = newsTypeRespList.stream().filter(x -> NewsTypeConstants.NEWS_TYPE_OTHER_ID.equals(x.getId())).collect(Collectors.toList()).stream().findFirst().orElse(null);
        if (ObjectUtils.isNotEmpty(other)) {
            newsTypeRespList.removeIf(x -> NewsTypeConstants.NEWS_TYPE_OTHER_ID.equals(x.getId()));
            other.setOtherFlag(Boolean.TRUE);
            other.setSort(null);
            newsTypeRespList.add(other);
        }
        return ResponseEntityWrapperUtil.wrapperOk(newsTypeRespList);
    }

    /**
     * 查询默认公告类型关联的公告列表
     *
     * @param newsTypeList 公告类型
     * @return 返回
     */
    private Map<Long, List<NewsAnnouncement>> queryNewsListWithDefaultType(List<NewsAnnouncementType> newsTypeList) {
        Map<Long, List<NewsAnnouncement>> defaultNewsTypeIdMap = Maps.newHashMap();
        List<Long> defaultNewsTypeIdList = newsTypeList.stream().filter(x -> NewsTypeConstants.NEWS_TYPE_DEFAULT_FLAG.equals(x.getDefaultFlag())).map(NewsAnnouncementType::getId).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(defaultNewsTypeIdList)) {
            return defaultNewsTypeIdMap;
        }
        LambdaQueryWrapper<NewsAnnouncement> condition = new LambdaQueryWrapper<>();
        condition.in(NewsAnnouncement::getNewsTypeId, defaultNewsTypeIdList);
        condition.select(NewsAnnouncement::getId, NewsAnnouncement::getNewsStatus, NewsAnnouncement::getNewsTypeId);
        List<NewsAnnouncement> newsAnnouncementList = newsAnnouncementMapper.selectList(condition);
        defaultNewsTypeIdMap = newsAnnouncementList.stream().collect(Collectors.groupingBy(NewsAnnouncement::getNewsTypeId));
        return defaultNewsTypeIdMap;
    }

    /**
     * 保存公告类型-校验公告类型
     *
     * @param validSaveNewsTypeReqDTO 请求参数
     * @return 返回
     */
    @PostMapping("/validSaveNewsType")
    public ResponseEntity<BaseResultDTO<ValidSaveNewsTypeVO>> validSaveNewsType(@RequestBody @Valid ValidSaveNewsTypeReqDTO validSaveNewsTypeReqDTO) {
        ValidSaveNewsTypeVO validSaveNewsTypeVO = new ValidSaveNewsTypeVO();
        // 校验
        String msgCode = validNewsTypeInner(validSaveNewsTypeReqDTO);
        validSaveNewsTypeVO.setMsgCode(msgCode);
        return ResponseEntityWrapperUtil.wrapperOk(validSaveNewsTypeVO);
    }

    /**
     * 保存公告类型-校验公告类型内部方法
     *
     * @param validSaveNewsTypeReqDTO 请求入参
     */
    private String validNewsTypeInner(ValidSaveNewsTypeReqDTO validSaveNewsTypeReqDTO) {
        // 查询当前父级公告类型的所有子级（parentId为0表示查询所有一级公告类型）
        List<NewsAnnouncementType> newsTypeList = newsAnnouncementTypeService.queryNewsTypeListByParentId(validSaveNewsTypeReqDTO.getParentId());
        // 新增公告类型
        if (ObjectUtils.isEmpty(validSaveNewsTypeReqDTO.getNewsTypeId())) {
            // 作为一级类型，告类型不能超过30个
            if (NewsTypeConstants.NEWS_TYPE_FIRST_LEVEL_PARENT_ID.equals(validSaveNewsTypeReqDTO.getParentId())
                    && newsTypeList.size() >= NewsTypeConstants.MAX_NEWS_TYPE_COUNT) {
                return NewsTypeConstants.FIRST_MAX_COUNT;
            }
            // 作为二级类型
            if (!NewsTypeConstants.NEWS_TYPE_FIRST_LEVEL_PARENT_ID.equals(validSaveNewsTypeReqDTO.getParentId())) {
                long count = newsTypeList.stream().filter(x -> !NewsTypeConstants.NEWS_TYPE_DEFAULT_FLAG.equals(x.getDefaultFlag())).count();
                String respCode = validBelongingParentNewsType(count >= NewsTypeConstants.MAX_NEWS_TYPE_COUNT, validSaveNewsTypeReqDTO);
                if (StringUtils.isNotBlank(respCode)) {
                    return respCode;
                }
            }
        }
        // 编辑公告类型
        else {
            // 作为一级类型
            if (NewsTypeConstants.NEWS_TYPE_FIRST_LEVEL_PARENT_ID.equals(validSaveNewsTypeReqDTO.getParentId())) {
                // 一级公告类型不能超过30个，排除掉当前的公告类型id
                long count = newsTypeList.stream().filter(x -> !validSaveNewsTypeReqDTO.getNewsTypeId().equals(x.getId())).count();
                if (count >= NewsTypeConstants.MAX_NEWS_TYPE_COUNT) {
                    return NewsTypeConstants.FIRST_MAX_COUNT;
                }
            } else { // 作为二级类型
                // 所属父级下的二级公告类型不能超过30个，排除掉当前的公告类型id
                long count = newsTypeList.stream().filter(x -> !NewsTypeConstants.NEWS_TYPE_DEFAULT_FLAG.equals(x.getDefaultFlag())
                        && !validSaveNewsTypeReqDTO.getNewsTypeId().equals(x.getId())).count();
                String respCode = validBelongingParentNewsType(count >= NewsTypeConstants.MAX_NEWS_TYPE_COUNT, validSaveNewsTypeReqDTO);
                if (StringUtils.isNotBlank(respCode)) {
                    return respCode;
                }

                // 当前公告类型已经包含子级，仅可作为一级类型
                LambdaQueryWrapper<NewsAnnouncementType> condition = new LambdaQueryWrapper<>();
                condition.eq(NewsAnnouncementType::getParentId, validSaveNewsTypeReqDTO.getNewsTypeId());
                Long childCount = newsAnnouncementTypeService.getBaseMapper().selectCount(condition);
                if (childCount > 0) {
                    return NewsTypeConstants.CONTAINS_CHILD;
                }
            }
        }
        return NewsTypeConstants.NEWS_TYPE_VALID_SUCCESS;
    }

    /**
     * 校验所属父级的公告类型
     *
     * @param flag                    数量限制标识
     * @param validSaveNewsTypeReqDTO 请求入参
     * @return 返回
     */
    private String validBelongingParentNewsType(boolean flag, ValidSaveNewsTypeReqDTO validSaveNewsTypeReqDTO) {
        // 所属父级下的二级公告类型不能超过30个
        if (flag) {
            return NewsTypeConstants.SECOND_MAX_COUNT;
        }
        NewsAnnouncementType parentNewsType = newsAnnouncementTypeService.getBaseMapper().selectById(validSaveNewsTypeReqDTO.getParentId());
        // 所属父级类型不存在
        if (ObjectUtils.isEmpty(parentNewsType)) {
            return NewsTypeConstants.PARENT_NOT_EXIST;
        }
        // 所属父级类型不可包含公告
        if (ObjectUtils.isNotEmpty(parentNewsType.getNewsCount()) && parentNewsType.getNewsCount() > 0) {
            return NewsTypeConstants.PARENT_CONTAINS_NEWS;
        }
        return null;
    }

    /**
     * 保存公告类型
     *
     * @param saveNewsTypeReqDTO 请求参数
     * @return 编辑结果
     */
    @PostMapping("/saveNewsType")
    public ResponseEntity<BaseResultDTO<Boolean>> saveNewsType(@RequestBody @Valid SaveNewsTypeReqDTO saveNewsTypeReqDTO) {
        // 校验
        ValidSaveNewsTypeReqDTO validSaveNewsTypeReqDTO = new ValidSaveNewsTypeReqDTO();
        validSaveNewsTypeReqDTO.setNewsTypeId(saveNewsTypeReqDTO.getId());
        validSaveNewsTypeReqDTO.setParentId(saveNewsTypeReqDTO.getParentId());
        String msgCode = validNewsTypeInner(validSaveNewsTypeReqDTO);
        if (!NewsTypeConstants.NEWS_TYPE_VALID_SUCCESS.equals(msgCode)) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "参数校验不通过，请检查");
        }
        // 保存
        return ResponseEntityWrapperUtil.wrapperOk(newsAnnouncementTypeService.saveNewsType(saveNewsTypeReqDTO));
    }

    /**
     * 校验公告类型是否存在
     *
     * @param queryAnnouncementTypeReq 公告类型id
     * @return 返回
     */
    @PostMapping("/validNewsType")
    public ResponseEntity<BaseResultDTO<Boolean>> validNewsType(@RequestBody @Valid QueryAnnouncementTypeReq queryAnnouncementTypeReq) {
        if (ObjectUtils.isEmpty(queryAnnouncementTypeReq.getNewsTypeId())) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "公告类型id不能为空");
        }
        return ResponseEntityWrapperUtil.wrapperOk(newsAnnouncementTypeService.existByTypeId(queryAnnouncementTypeReq.getNewsTypeId()));
    }

    /**
     * 公告类型列表添加子级-校验公告类型
     *
     * @param announcementIdReq 请求参数
     * @return 返回
     */
    @PostMapping("/validChildLevel")
    public ResponseEntity<BaseResultDTO<ValidSaveNewsTypeVO>> validChildLevel(@RequestBody @Valid AnnouncementIdReq announcementIdReq) {
        ValidSaveNewsTypeVO validSaveNewsTypeVO = new ValidSaveNewsTypeVO();
        // 当前公告类型不存在
        NewsAnnouncementType newsAnnouncementType = newsAnnouncementTypeService.getBaseMapper().selectById(announcementIdReq.getId());
        if (ObjectUtils.isEmpty(newsAnnouncementType)) {
            validSaveNewsTypeVO.setMsgCode(NewsTypeConstants.CURRENT_NOT_EXIST);
            return ResponseEntityWrapperUtil.wrapperOk(validSaveNewsTypeVO);
        }
        // 当前公告类型包含数据，不可添加子级
        if (ObjectUtils.isNotEmpty(newsAnnouncementType.getNewsCount()) && newsAnnouncementType.getNewsCount() != 0) {
            validSaveNewsTypeVO.setMsgCode(NewsTypeConstants.CURRENT_CONTAINS_NEWS);
            return ResponseEntityWrapperUtil.wrapperOk(validSaveNewsTypeVO);
        }

        // 当前公告类型的子级已达上限
        List<NewsAnnouncementType> childNewsTypeList = queryChildNewsTypeList(announcementIdReq.getId());
        if (childNewsTypeList.size() >= NewsTypeConstants.MAX_NEWS_TYPE_COUNT) {
            validSaveNewsTypeVO.setMsgCode(NewsTypeConstants.CURRENT_CHILD_MAX_COUNT);
            return ResponseEntityWrapperUtil.wrapperOk(validSaveNewsTypeVO);
        }
        validSaveNewsTypeVO.setMsgCode(NewsTypeConstants.NEWS_TYPE_VALID_SUCCESS);
        return ResponseEntityWrapperUtil.wrapperOk(validSaveNewsTypeVO);
    }

    /**
     * 删除公告类型
     *
     * @param announcementIdReq 主键id
     * @return 返回
     */
    @PostMapping("/delNewsType")
    public ResponseEntity<BaseResultDTO<Boolean>> delNewsType(@RequestBody @Valid AnnouncementIdReq announcementIdReq) {
        NewsAnnouncementType newsAnnouncementType = newsAnnouncementTypeService.getBaseMapper().selectById(announcementIdReq.getId());
        if (ObjectUtils.isEmpty(newsAnnouncementType)) {
            return ResponseEntityWrapperUtil.wrapperOk(true);
        }
        newsAnnouncementTypeService.delNewsType(newsAnnouncementType);
        return ResponseEntityWrapperUtil.wrapperOk(true);
    }

    /**
     * 校验用户是否有当前公告类型的权限
     * 单条公告-编辑/删除/发布/取消发布/置顶/取消置顶
     * 多条公告-批量删除/发布/取消发布
     *
     * @param deleteNewsOrTypeReq 请求参数
     * @return 返回
     */
    @PostMapping("/validOperateNews")
    public ResponseEntity<BaseResultDTO<ValidSaveNewsTypeVO>> validOperateNews(@RequestBody @Valid DeleteNewsOrTypeReq deleteNewsOrTypeReq) {
        ValidSaveNewsTypeVO validSaveNewsTypeVO = new ValidSaveNewsTypeVO();
        validSaveNewsTypeVO.setMsgCode(NewsTypeConstants.NEWS_TYPE_VALID_SUCCESS);
        if (CollectionUtils.isEmpty(deleteNewsOrTypeReq.getIdList())) {
            return ResponseEntityWrapperUtil.wrapperOk(validSaveNewsTypeVO);
        }

        // 查询当前登录用户有权限的公告类型
        NewsTypeAuthVO newsTypeAuthVO = newsAnnouncementTypeAuthService.queryAuthRelNewsType(null);
        // 有全部公告类型的权限
        if (Constants.ALL_AUTH_FLAG_YES.equals(newsTypeAuthVO.getAllNewsTypeFlag())) {
            return ResponseEntityWrapperUtil.wrapperOk(validSaveNewsTypeVO);
        }

        // 有权限的公告类型id
        List<Long> newsTypeIdList = newsTypeAuthVO.getAuthorizedNewsTypeIdList();
        // 查询公告详情
        List<NewsAnnouncement> newsAnnouncementList = newsAnnouncementService.getBaseMapper().selectBatchIds(deleteNewsOrTypeReq.getIdList());
        // 存在没权限的公告类型
        boolean flag = newsAnnouncementList.stream().anyMatch(x -> !newsTypeIdList.contains(x.getNewsTypeId()));
        if (flag) {
            if (deleteNewsOrTypeReq.getIdList().size() == 1) {
                validSaveNewsTypeVO.setMsgCode(NewsTypeConstants.CURRENT_SINGLE_WITHOUT_PERMISSION);
            } else {
                validSaveNewsTypeVO.setMsgCode(NewsTypeConstants.CURRENT_BATCH_WITHOUT_PERMISSION);
            }
        }
        return ResponseEntityWrapperUtil.wrapperOk(validSaveNewsTypeVO);
    }

    /**
     * 保存公告-校验公告类型
     *
     * @param validSaveNewsTypeReqDTO 请求参数
     * @return 返回
     */
    @PostMapping("/validSaveNews")
    public ResponseEntity<BaseResultDTO<ValidSaveNewsTypeVO>> validSaveNews(@RequestBody @Valid ValidSaveNewsTypeReqDTO validSaveNewsTypeReqDTO) {
        ValidSaveNewsTypeVO validSaveNewsTypeVO = new ValidSaveNewsTypeVO();
        // 判断公告类型是否存在
        NewsAnnouncementType newsAnnouncementType = InterceptorIgnoreUtil.handler(() -> newsAnnouncementTypeService.getBaseMapper().selectById(validSaveNewsTypeReqDTO.getNewsTypeId()));
        if (ObjectUtils.isEmpty(newsAnnouncementType)) {
            validSaveNewsTypeVO.setMsgCode(NewsTypeConstants.CURRENT_NOT_EXIST);
            return ResponseEntityWrapperUtil.wrapperOk(validSaveNewsTypeVO);
        }

        // 选择的是非一级公告类型，变成了一级公告类型
        if (!NewsTypeConstants.NEWS_TYPE_FIRST_LEVEL_PARENT_ID.equals(validSaveNewsTypeReqDTO.getParentId())
                && NewsTypeConstants.NEWS_TYPE_FIRST_LEVEL_PARENT_ID.equals(newsAnnouncementType.getParentId())) {
            validSaveNewsTypeVO.setMsgCode(NewsTypeConstants.CHANGE_TO_FIRST);
            return ResponseEntityWrapperUtil.wrapperOk(validSaveNewsTypeVO);
        }

        // 选择的一级公告类型，包含了二级类型公告
        if (NewsTypeConstants.NEWS_TYPE_FIRST_LEVEL_PARENT_ID.equals(validSaveNewsTypeReqDTO.getParentId())) {
            List<NewsAnnouncementType> childNewsTypeList = queryChildNewsTypeList(validSaveNewsTypeReqDTO.getNewsTypeId());
            if (CollectionUtils.isNotEmpty(childNewsTypeList)) {
                validSaveNewsTypeVO.setMsgCode(NewsTypeConstants.FIRST_CONTAINS_SECOND);
                return ResponseEntityWrapperUtil.wrapperOk(validSaveNewsTypeVO);
            }
        }

        // 判断权限
        NewsTypeAuthVO newsTypeAuthVO = newsAnnouncementTypeAuthService.queryAuthRelNewsType(null);
        if (Constants.ALL_AUTH_FLAG_YES.equals(newsTypeAuthVO.getAllNewsTypeFlag())
                || newsTypeAuthVO.getAuthorizedNewsTypeIdList().contains(validSaveNewsTypeReqDTO.getNewsTypeId())) {
            validSaveNewsTypeVO.setMsgCode(NewsTypeConstants.NEWS_TYPE_VALID_SUCCESS);
        } else {
            validSaveNewsTypeVO.setMsgCode(NewsTypeConstants.CURRENT_WITHOUT_PERMISSION);
        }
        return ResponseEntityWrapperUtil.wrapperOk(validSaveNewsTypeVO);
    }

    /**
     * 查询当前公告类型的子级类型列表
     *
     * @param newsTypeId 公告类型id
     * @return 返回
     */
    private List<NewsAnnouncementType> queryChildNewsTypeList(Long newsTypeId) {
        LambdaQueryWrapper<NewsAnnouncementType> condition = new LambdaQueryWrapper<>();
        condition.eq(NewsAnnouncementType::getParentId, newsTypeId);
        return newsAnnouncementTypeService.getBaseMapper().selectList(condition);
    }
}

