package com.digiwin.athena.semc.controller.open;

import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.I18NKey;
import com.digiwin.athena.semc.dto.open.AppJobAccountVO;
import com.digiwin.athena.semc.dto.open.AppJobUrlReq;
import com.digiwin.athena.semc.dto.open.AppJobVO;
import com.digiwin.athena.semc.entity.applink.AppLinkDTO;
import com.digiwin.athena.semc.service.open.OpenAppJobService;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;
import com.google.common.collect.Lists;
import io.swagger.v3.oas.annotations.Operation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.validation.constraints.NotEmpty;
import java.time.LocalDateTime;
import java.util.List;
import java.util.Objects;

/**
 * @author dengkaiwen
 * @Description  作业同步
 * Create 2023-8-28
 */
@Slf4j
@RestController
@RequestMapping("/semc/openapi")
public class JobController {

    private static final Logger logger = LoggerFactory.getLogger(JobController.class);

    @Resource
    private MessageUtils messageUtils;

    @Resource
    private OpenAppJobService openAppJobService;


    /**
     * 获取指定应用下的作业列表
     *
     * @param applicationType 应用类型，用于区分不同的应用类别，不能为空
     * @param applicationCode 应用代号，用于唯一标识一个应用，不能为空
     * @return 返回一个包含作业列表的ResponseEntity对象
     */
    @Operation(summary = "应用下作业")
    @GetMapping("/job/{applicationType}/{applicationCode}")
    public ResponseEntity<BaseResultDTO<List<AppJobVO>>> getAppJobList(@PathVariable("applicationType") @NotEmpty(message = "应用类型不能为空") Integer applicationType, @PathVariable("applicationCode") @NotEmpty(message = "应用代号不能为空") String applicationCode) {
        try {
            //调用服务层方法获取作业列表
            return ResponseEntityWrapperUtil.wrapperOk(openAppJobService.queryAppJobList(applicationCode,applicationType));
        } catch (Exception e) {
            //异常处理
            logger.error("queryAppJobList exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/openapi/job/list");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }




    /**
     * 应用下作业账套查询接口
     * 该接口用于获取指定应用下的作业账套信息
     */
    @Operation(summary = "应用下作业账套")
    @PostMapping("/job/accounts")
    public ResponseEntity<BaseResultDTO<List<AppJobAccountVO>>> queryAppJobAccounts(@RequestBody List<AppLinkDTO> appCodeList) {
        try {
            //校验应用编码
            if (CollectionUtils.isEmpty(appCodeList)) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "app code cannot be empty");
            }
            //调用服务层方法查询作业账套信息
            return ResponseEntityWrapperUtil.wrapperOk(openAppJobService.queryAppJobAccount(appCodeList));
        } catch (Exception e) {
            //记录异常日志
            logger.error("queryAppJobList exception", e);
            //格式化系统错误信息
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/openapi/job/account");
            //返回系统错误响应
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }


    /**
     * 查询应用下作业账套
     *
     * 该接口用于查询特定应用下的作业账套信息通过POST请求接收应用信息，并返回该应用下的作业账套详情
     * 如果找到相应的作业账套，则返回第一个账套的信息；如果没有找到，则返回空值
     *
     * @param app 包含应用信息的AppLinkDTO对象，用于查询作业账套
     * @return 返回包含应用作业账套信息的响应实体
     */
    @Operation(summary = "应用下作业账套")
    @PostMapping("/job/account")
    public ResponseEntity<BaseResultDTO<AppJobAccountVO>> queryAppJobAccount(@RequestBody AppLinkDTO app) {
        try {
            // 调用服务层方法查询应用作业账户信息
            List<AppJobAccountVO> appJobAccounts = openAppJobService.queryAppJobAccount(Lists.newArrayList(app));
            // 检查查询结果是否为空
            if(CollectionUtils.isNotEmpty(appJobAccounts)){
                // 如果结果不为空，返回第一个作业账户信息
                return ResponseEntityWrapperUtil.wrapperOk(appJobAccounts.get(0));
            }
            // 如果结果为空，返回空值
            return ResponseEntityWrapperUtil.wrapperOk(null);
        } catch (Exception e) {
            // 记录异常信息
            logger.error("queryAppJobList exception", e);
            // 格式化错误信息
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/openapi/job/account");
            // 返回系统错误响应
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }



    /**
     * 查询应用或作业链接
     *
     * @param jobUrlReq 作业参数
     * @return 查询结果
     */
    @PostMapping("/app/link")
    public ResponseEntity<BaseResultDTO<String>> queryCustomAppLink(@RequestBody AppJobUrlReq jobUrlReq) {
        try {
            // 调用服务层方法查询应用或作业链接
            AppLinkDTO appLinkDTO = openAppJobService.queryAppOrJobLink(jobUrlReq);
            // 返回查询结果，如果结果为空，则返回空字符串
            return ResponseEntityWrapperUtil.wrapperOk(Objects.nonNull(appLinkDTO)?appLinkDTO.getCloudwebsite():"");
        } catch (Exception e) {
            // 记录错误日志
            log.error("queryCustomAppLink is error：{}", jobUrlReq, e);
            // 格式化错误信息
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/openapi/app/link");
            // 返回系统错误响应
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

}
