package com.digiwin.athena.semc.controller.open;

import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.semc.common.ResponseBody;
import com.digiwin.athena.semc.common.*;
import com.digiwin.athena.semc.controller.BasicController;
import com.digiwin.athena.semc.dto.PageInfo;
import com.digiwin.athena.semc.dto.open.SystemToDoReq;
import com.digiwin.athena.semc.dto.portal.*;
import com.digiwin.athena.semc.entity.portal.LabelSystemData;
import com.digiwin.athena.semc.entity.portal.TodoList;
import com.digiwin.athena.semc.service.open.OpenThirdToDoService;
import com.digiwin.athena.semc.service.portal.TodoListService;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;
import io.swagger.v3.oas.annotations.Operation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.validation.Valid;
import javax.validation.constraints.NotEmpty;
import java.time.LocalDateTime;
import java.util.List;

/**
 * 待办列表(TodoList)表控制层
 *
 * @author sunyfa
 * @since 2022-12-06 13:48:51
 */
@Slf4j
@RestController
@RequestMapping("/semc/openapi/todo")
public class ThirdTodoController extends BasicController<TodoListService, TodoList> {
    @Resource
    private TodoListService todoListService;

    @Resource
    private OpenThirdToDoService openThirdToDoService;

    @Resource
    private MessageUtils messageUtils;

    /**
     * 查询组件下数据源的应用系统
     *
     * @return
     */
    @Operation(summary = "查询组件下数据源的应用系统")
    @PostMapping("/application/list")
    public ResponseEntity<BaseResultDTO<List<LabelSystemData>>> queryTodoAppList() {
        try {
            return ResponseEntityWrapperUtil.wrapperOk(openThirdToDoService.queryTodoAppList());
        } catch (Exception e) {
            log.error("queryTodoAppList is exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/openapi/todo/application/list");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 根据SSO配置中的AppId查询应用下，鼎捷云账号对应地端用户的三方待办信息
     *
     * @param todoListReq 查询待办列表请求参数
     * @return 待办数据
     */
    @PostMapping("list")
    public ResultPageBean selectTodoList(@Valid @RequestBody SystemToDoReq todoListReq) {
        TodoListResp todoListResp = new TodoListResp();
        try {
            todoListResp = openThirdToDoService.queryTodoList(todoListReq);
        } catch (Exception e) {
            log.error("selectTodoList query exception", e);
            todoListResp.setResultCode("3001");
        }
        return ResultPageBean.success(ResponseBody.getInstance(todoListResp, PageInfo.getPageInfo(todoListReq.getPageNum(), todoListReq.getPageSize(), todoListResp.getTotalResults())));
    }

    /**
     * 根据SSO配置中的AppId查询应用下，鼎捷云账号对应地端用户的三方待办信息
     *
     * @param todoListReq 查询待办列表请求参数
     * @return 待办数据
     */
    @PostMapping("list/{applicationType}/{applicationCode}")
    public ResultPageBean selectAppTodoList(@PathVariable("applicationType") @NotEmpty(message = "应用类型不能为空") Integer applicationType,@PathVariable("applicationCode") @NotEmpty(message = "应用代号不能为空") String applicationCode, @Valid @RequestBody SystemToDoReq todoListReq) {
        TodoListResp todoListResp = new TodoListResp();
        try {
            todoListResp = openThirdToDoService.queryTodoListByCode(applicationCode,applicationType,todoListReq);
        } catch (Exception e) {
            log.error("selectTodoList query exception", e);
            todoListResp.setResultCode("3001");
        }
        return ResultPageBean.success(ResponseBody.getInstance(todoListResp, PageInfo.getPageInfo(todoListReq.getPageNum(), todoListReq.getPageSize(), todoListResp.getTotalResults())));
    }

    /**
     * 获取跳转URL接口
     * 本接口用于获取待办列表中的特定任务详情页面的跳转URL
     */
    @Operation(method = "getTodoUrl", description = "获取跳转url")
    @GetMapping("/jumpUrl")
    public ResponseEntity<?> getTodoUrl(@RequestParam(value = "taskId") Long taskId) {
        // 创建报销请求对象，并设置任务ID
        ReimburseReq reimburseReq = new ReimburseReq();
        reimburseReq.setId(taskId);
        // 调用服务层方法获取跳转URL
        return todoListService.getTodoListUrl(reimburseReq);
    }

}
