package com.digiwin.athena.semc.controller.portal;
import com.alibaba.fastjson.JSON;

import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.GlobalConstant;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.I18NKey;
import com.digiwin.athena.semc.common.ResultBean;
import com.digiwin.athena.semc.common.enums.BizObjectTypeEnum;
import com.digiwin.athena.semc.dto.erpsso.GetSsoUrlReq;
import com.digiwin.athena.semc.dto.erpsso.GetSsoUrlResp;
import com.digiwin.athena.semc.dto.erpsso.QueryDisplayReq;
import com.digiwin.athena.semc.entity.applink.AppLinkDTO;
import com.digiwin.athena.semc.entity.common.BizObjCommonUse;
import com.digiwin.athena.semc.service.applink.AppLinkService;
import com.digiwin.athena.semc.service.common.IBizObjAuthRelService;
import com.digiwin.athena.semc.service.common.IBizObjCommonUseService;
import com.digiwin.athena.semc.service.menu.AuthService;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;
import com.digiwin.athena.semc.util.SecurityLogUtil;
import com.digiwin.athena.semc.vo.common.BizObjAuthRelBatchForLinkReq;
import com.digiwin.athena.semc.vo.common.BizObjCommonUseVo;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.validation.Valid;
import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;


/**
 * 友情链接
 *
 * @date 2022/12/22
 */
@Slf4j
@RestController
@RequestMapping(value = "/tenant/semc/applink")
public class AppLinkController {

    @Autowired
    private AppLinkService appLinkService;

    @Resource
    private IBizObjAuthRelService bizObjAuthRelService;

    @Resource
    private IBizObjCommonUseService bizObjCommonUseService;

    @Resource
    private MessageUtils messageUtils;

    @Resource
    private AuthService authService;

    /**
     * 记录一次访问友情链接的操作(目前不用了)
     *
     * @param appJson 应用id
     * @return
     */
    @PostMapping("/accesses")
    public ResultBean accessLink(@RequestBody Map<String, String> appJson) {
        ResultBean resultBean = new ResultBean();
        JSONObject jsonObject = JSONObject.parseObject(JSONObject.toJSONString(appJson));
        appLinkService.accessLink(jsonObject.getString("id"));
        resultBean.setResponse(Boolean.TRUE);
        return resultBean;
    }

    /**
     * 首页，查询全部应用列表，即当前登录用户权限内可查看的应用列表
     *
     * @param user 用户信息
     * @param key  搜索关键字
     * @return
     */
    @GetMapping("/queryDisplayList")
    public ResponseEntity<BaseResultDTO<List<AppLinkDTO>>> queryDisplayList(@RequestAttribute(value = GlobalConstant.AUTH_USER) AuthoredUser user,
                                                          @RequestParam(required = false, value = "key") String key) {
        try {
            List<AppLinkDTO> appLinkListDTO = appLinkService.queryDisplayList(user);
            if (StringUtils.isNotBlank(key)) {
                List<AppLinkDTO> matchedLinkList = appLinkListDTO.stream().filter(x -> x.getName().contains(key)).collect(Collectors.toList());
                return ResponseEntityWrapperUtil.wrapperOk(matchedLinkList);
            }
            return ResponseEntityWrapperUtil.wrapperOk(appLinkListDTO);
        } catch (Exception e) {
            log.error("AppLinkController query manage list exception", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * TBB报表支持串查ERP报表
     *
     * @param user 用户信息
     * @param req  搜索关键字
     * @return
     */
    @PostMapping("/queryDisplay")
    public ResponseEntity<BaseResultDTO<AppLinkDTO>> queryDisplay(@RequestAttribute(value = GlobalConstant.AUTH_USER) AuthoredUser user,@RequestBody QueryDisplayReq req) {
        try {
            log.info("queryDisplay req:{}", req);
            AppLinkDTO appLinkDTO = appLinkService.queryDisplay(user, req);
            if(null == appLinkDTO){
                return ResponseEntityWrapperUtil.wrapperFail(HttpStatus.INTERNAL_SERVER_ERROR.value(), messageUtils.getMessage(I18NKey.QUERY_NOT_FOUND));
            }
            return ResponseEntityWrapperUtil.wrapperOk(appLinkDTO);
        } catch (Exception e) {
            log.error("AppLinkController queryDisplay exception", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, e.getMessage());
        }
    }

    /**
     * 首页，查询常用应用列表
     *
     * @param user 用户信息
     * @return
     */
    @GetMapping("/queryCommonUseList")
    public ResponseEntity<BaseResultDTO<List<AppLinkDTO>>> queryCommonUseList(@RequestAttribute(value = GlobalConstant.AUTH_USER) AuthoredUser user) {
        try {
            List<BizObjCommonUse> commonUseList = bizObjCommonUseService.queryCommonUseList(BizObjectTypeEnum.APPlICATION_MANAGEMENT.getValue());
            List<AppLinkDTO> appLinkDTOList = appLinkService.queryCommonList(commonUseList, user);
            return ResponseEntityWrapperUtil.wrapperOk(appLinkDTOList);
        } catch (Exception e) {
            log.error("AppLinkController query common list exception", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }


    /**
     * 获取标准化应用跳转SSO地址
     *
     * @return
     */
    @PostMapping("/getLinkSsoUrl")
    public ResponseEntity<BaseResultDTO<AppLinkDTO>> getLinkSsoUrl(@RequestBody AppLinkDTO req) {
        try {
             if(StringUtils.isEmpty(req.getPrimaryId())){
                 return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR,"primaryId 不可为空");
             }
            req=appLinkService.getLinkSsoUrl(req);
            return ResponseEntityWrapperUtil.wrapperOk(req);
        } catch (Exception e) {
            log.error("getLinkSsoUrl  exception", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 首页，设置或取消常用应用
     *
     * @param bizObjCommonUseVo 设置请求参数
     * @return
     */
    @PostMapping("/saveCommonUse")
    public ResponseEntity<BaseResultDTO<Boolean>> saveCommonUse(@RequestBody @Valid BizObjCommonUseVo bizObjCommonUseVo) {
        try {
            bizObjCommonUseService.saveCommonUse(bizObjCommonUseVo, BizObjectTypeEnum.APPlICATION_MANAGEMENT.getValue());
            return ResponseEntityWrapperUtil.wrapperOk(true);
        } catch (Exception e) {
            log.error("AppLinkController query common list exception", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 管理后台，查询所有b/s应用、c/s应用、预设应用
     *
     * @param user 用户信息
     * @return
     */
    @GetMapping("/queryManageList")
    public ResponseEntity<BaseResultDTO<List<AppLinkDTO>>> queryManageList(@RequestAttribute(value = GlobalConstant.AUTH_USER) AuthoredUser user) {
        try {
            List<AppLinkDTO> appLinkListDTO = appLinkService.queryManageList(user);
            return ResponseEntityWrapperUtil.wrapperOk(appLinkListDTO);
        } catch (Exception e) {
            log.error("AppLinkController query manage list exceptiona", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 管理后台，根据权限id查询有权限的应用列表
     *
     * @param authId 用户信息
     * @return
     */
    @GetMapping("/queryAuthorizedList")
    public ResponseEntity<BaseResultDTO<List<AppLinkDTO>>> queryAuthorizedList(@RequestParam(value = "authId") String authId) {
        try {
            List<AppLinkDTO> appLinkListDTO = appLinkService.queryAuthorizedList(authId);
            return ResponseEntityWrapperUtil.wrapperOk(appLinkListDTO);
        } catch (Exception e) {
            log.error("AppLinkController query manage list exception", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 管理后台，保存权限对象对应的应用
     *
     * @param bizObjAuthRelBatchForLinkReq 入参
     * @return
     */
    @PostMapping("/saveAuth")
    public ResponseEntity<BaseResultDTO<Boolean>> saveAuth(@RequestBody @Valid BizObjAuthRelBatchForLinkReq bizObjAuthRelBatchForLinkReq, HttpServletRequest request) {
        try {
            //用户及权限配置/工作台权限配置/应用权限配置（权限判断）
            if (authService.hasAuth("permission-application")) {
                bizObjAuthRelService.saveBizObjectAuthRelForLinkApp(bizObjAuthRelBatchForLinkReq);
                return ResponseEntityWrapperUtil.wrapperOk(true);
            }
            else {
                //写入安全日志
                SecurityLogUtil.writeSecurityLog(request, JSON.toJSONString(bizObjAuthRelBatchForLinkReq));
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "无权限");
            }
        } catch (Exception e) {
            log.error("AppLinkController save auth exception, param:{}", bizObjAuthRelBatchForLinkReq, e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 管理后台，查询所有b/s应用、c/s应用、预设应用
     *
     * @return
     */
    @PostMapping("/querySsoAppList")
    public ResponseEntity<BaseResultDTO<List<AppLinkDTO>>> querySsoAppList() {
        try {
            AuthoredUser user = AppAuthContextHolder.getContext().getAuthoredUser();
            List<AppLinkDTO> appLinkListDTO = appLinkService.querySsoList(user);
            return ResponseEntityWrapperUtil.wrapperOk(appLinkListDTO);
        } catch (Exception e) {
            log.error("querySsoAppList exceptiona", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/tenant/semc/applink/querySsoAppList");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 获取SSO url
     *
     * @return
     */
    @PostMapping("/getSsoUrl")
    public ResponseEntity<?> getSsoUrl(@RequestBody GetSsoUrlReq req) {
        try {
            GetSsoUrlResp resp = appLinkService.getSsoUrl(req);
            return ResponseEntityWrapperUtil.wrapperOk(resp);
        } catch (Exception e) {
            log.error("getSsoUrl exceptiona", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/tenant/semc/applink/getSsoUrl");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }
}
