package com.digiwin.athena.semc.controller.portal;

import com.alibaba.fastjson.JSON;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.appcore.util.ResponseEntityWrapper;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.I18NKey;
import com.digiwin.athena.semc.common.ResultBean;
import com.digiwin.athena.semc.dto.homepage.DataMonitorReq;
import com.digiwin.athena.semc.dto.homepage.DataMonitorToDoReq;
import com.digiwin.athena.semc.dto.mq.MessageDO;
import com.digiwin.athena.semc.dto.portal.JobSyncReq;
import com.digiwin.athena.semc.dto.portal.TodoReceiveDto;
import com.digiwin.athena.semc.entity.portal.ThirdTodoConfig;
import com.digiwin.athena.semc.service.cache.ICacheService;
import com.digiwin.athena.semc.service.mq.MessageSendService;
import com.digiwin.athena.semc.service.portal.ThirdTodoConfigService;
import com.digiwin.athena.semc.service.portal.TodoListService;
import com.digiwin.athena.semc.util.AESUtils;
import com.digiwin.athena.semc.vo.maycur.MaycurTodoNoticeReq;
import com.digiwin.athena.semc.vo.maycur.MaycurTodoNoticeResp;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;

import javax.annotation.Resource;

/**
 * @description 待办数据接收
 * @author: dengkaiwen
 * @create: 2023-8-28
 */
@RestController
@RequestMapping("/tenant/semc")
public class DataMonitorController {

    private static final Logger logger = LoggerFactory.getLogger(DataMonitorController.class);

    @Resource
    private MessageSendService messageSendService;

    @Autowired
    ICacheService cacheService;

    @Resource
    private MessageUtils messageUtils;

    @Resource
    private TodoListService todoListService;

    @Resource
    private ThirdTodoConfigService thirdTodoConfigService;

    /**
     * 数据侦测接收
     *
     * @param dataMonitorReq 数据侦测接收
     * @return
     * @throws Exception
     */
    @PostMapping("/data/notice")
    public ResultBean dataNotice(@RequestBody DataMonitorReq dataMonitorReq) {
        logger.info("dataNotice param:{}", JSON.toJSONString(dataMonitorReq));
        ResultBean resultBean = new ResultBean();
        try {
            List<String> verifyUserldList = dataMonitorReq.getVerifyUserldList();
            List<DataMonitorToDoReq> changeDataList = dataMonitorReq.getChangeData();
            if (CollectionUtils.isNotEmpty(changeDataList)) {
                verifyUserldList=new ArrayList<>();
                //获取地端用户ID
                String userId = changeDataList.get(0).getFkEmpId();
                verifyUserldList.add(userId);
            }
            for (String verifyUserld : verifyUserldList) {
                String key = Constants.SEMC_TODO_USER + ":" + dataMonitorReq.getAppName();
                if (StringUtils.isNotEmpty(verifyUserld)) {
                    key = key + ":" + verifyUserld;
                }
                Object isUserObj = cacheService.get(key);
                //同一个用户，同一个业务，5秒内只刷新一次
                if (null != isUserObj) {
                    return resultBean;
                }
                //上锁，5秒钟
                cacheService.cache(key, key, Duration.ofSeconds(1));
                MessageDO payload = new MessageDO();
                payload.setAppName(dataMonitorReq.getAppName());
                payload.setAppId(dataMonitorReq.getAppId());
                payload.setUserId(verifyUserld);
                payload.setType(Constants.VALID_STATUS_UNUSABLE);
                messageSendService.sendToClient(verifyUserld, payload, Constants.SEMC_TODO_USER);
            }
            return resultBean;
        } catch (Exception e) {
            logger.error("dataNotice  error req:{},e:{}", JSON.toJSONString(dataMonitorReq), e);
            return resultBean;
        }
    }



    /**
     * 三方待办数据接收
     *
     * @return
     * @throws Exception
     */
    @PostMapping("/todo/receive")
    public ResponseEntity<?> toDoReceive(@RequestBody JobSyncReq req) {
        logger.info("toDoReceive param:{}", JSON.toJSONString(req));
        String bodyJsonStr = req.getBodyJsonStr();
        if (StringUtils.isEmpty(bodyJsonStr)) {
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "missing bodyJsonStr parameter");
        }
        TodoReceiveDto receiveDto;
        try {
            String aesDecryptResult = AESUtils.aesDecrypt(bodyJsonStr, Constants.SYNC_SECRET);
              receiveDto = JSON.parseObject(aesDecryptResult, TodoReceiveDto.class);
        } catch (Exception e) {
            logger.error("toDoReceive parseObject error", e);
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "入参解析异常，请检查入参是否正确");
        }
        try {
            if (receiveDto==null) {
                return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "入参解析异常，请检查入参是否正确");
            }
            //判断appCode是否为空
            if (StringUtils.isEmpty(receiveDto.getAppCode())) {
                return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "missing app_code parameter");
            }
            //判断to_do_list是否为空
            if (CollectionUtils.isEmpty(receiveDto.getToDoList())) {
                return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "missing to_do_list parameter");
            }
            //判断display_field_list是否为空
            if (CollectionUtils.isEmpty(receiveDto.getDisplayFieldList())) {
                return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "missing display_field_list parameter");
            }
            //查询待办集成配置，校验是否配置且启用状态
            ThirdTodoConfig thirdTodoConfig= thirdTodoConfigService.getToDoConfigBy("",receiveDto.getAppCode());
            if (null == thirdTodoConfig) {
                return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "当前应用不在配置中，无权推送待办");
            }
            if (Constants.VALID_STATUS_UNUSABLE.equals(thirdTodoConfig.getValidStatus())) {
                return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "当前应用未启用，无权推送待办");
            }
            todoListService.toDoReceive(receiveDto);
            return ResponseEntityWrapper.wrapperOk();
        } catch (Exception e) {
            logger.error("toDoReceive exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "待办同步异常");
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }



    /**
     * 每刻报销待办通知
     *
     * @return
     * @throws Exception
     */
    @PostMapping("/maycur/approvals/notice")
    public MaycurTodoNoticeResp maycurReceive(@RequestBody MaycurTodoNoticeReq req) {
        logger.info("maycurReceive approvals param:{}", JSON.toJSONString(req));
        MaycurTodoNoticeResp resp=new MaycurTodoNoticeResp();
        try {
            //  String  decryptStr= DesUtil.decrypt(req.getEncryptParams(),DesUtil.MAYCUR_APPROVALS_KEY);
            //MaycurTodoNoticeReq todoNoticeReq=JSON.parseObject(decryptStr, MaycurTodoNoticeReq.class);
            //判断外调参数是否为空
            resp.setCode(ErrorCodeConstant.PARAM_MISSING_ERROR+"");
            if (null ==req.getCalloutParams()) {
                resp.setMessage("外调参数为空");
                return resp;
            }
            //判断单据号是否为空
            if (StringUtils.isEmpty(req.getCalloutParams().getFormCode())) {
                resp.setMessage("单据号为空");
                return resp;
            }
            //判断任务类型是否为空
            if (StringUtils.isEmpty(req.getCalloutParams().getTaskActionType())) {
                resp.setMessage("任务类型为空");
                return resp;
            }
            //判断任务处理人是否为空
            if (CollectionUtils.isEmpty(req.getCalloutParams().getTaskAssignee())) {
                resp.setMessage("任务处理人为空");
                return resp;
            }
            todoListService.approvalsNotice(req);
            resp.setCode("ACK");
            return resp;
        } catch (Exception e) {
            logger.error("maycurReceive approvals exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "每刻报销回调异常");
            resp.setCode(ErrorCodeConstant.PARAM_MISSING_ERROR+"");
            resp.setMessage(error);
            return resp;
        }
    }
}
