package com.digiwin.athena.semc.controller.portal;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.appcore.util.ResponseEntityWrapper;
import com.digiwin.athena.semc.common.BizException;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.I18NKey;
import com.digiwin.athena.semc.common.ResponseBody;
import com.digiwin.athena.semc.common.ResultPageBean;
import com.digiwin.athena.semc.common.ServiceException;
import com.digiwin.athena.semc.common.enums.BizObjectTypeEnum;
import com.digiwin.athena.semc.common.enums.BusinessSceneEnum;
import com.digiwin.athena.semc.common.enums.FolderLevelEnum;
import com.digiwin.athena.semc.common.enums.FolderTypeEnum;
import com.digiwin.athena.semc.controller.BasicController;
import com.digiwin.athena.semc.dto.PageInfo;
import com.digiwin.athena.semc.dto.portal.FineReportQueryReq;
import com.digiwin.athena.semc.dto.portal.QueryReportReq;
import com.digiwin.athena.semc.dto.portal.SaveReportUrlParamReq;
import com.digiwin.athena.semc.entity.common.BizObjAuthRel;
import com.digiwin.athena.semc.entity.common.Folder;
import com.digiwin.athena.semc.entity.portal.FineReport;
import com.digiwin.athena.semc.service.cache.ICacheService;
import com.digiwin.athena.semc.service.cache.impl.CacheServiceImpl;
import com.digiwin.athena.semc.service.common.IBizObjAuthRelService;
import com.digiwin.athena.semc.service.common.IFolderService;
import com.digiwin.athena.semc.service.portal.IFineReportService;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;
import com.digiwin.athena.semc.vo.common.BizObjAuthRelBatchVo;
import com.digiwin.athena.semc.vo.common.DeleteFolderVo;
import com.digiwin.athena.semc.vo.common.FolderViewVo;
import com.digiwin.athena.semc.vo.common.FolderVo;
import com.digiwin.athena.semc.vo.common.MoveFolderChildVo;
import com.digiwin.athena.semc.vo.common.MoveFolderListVo;
import com.digiwin.athena.semc.vo.common.MoveFolderVo;
import com.digiwin.athena.semc.vo.portal.FineReportVo;
import com.digiwin.athena.semc.vo.portal.MoveFineReportVo;

import io.github.linpeilie.Converter;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.http.ResponseEntity;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import javax.annotation.Resource;
import javax.validation.Valid;
import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;

import lombok.extern.slf4j.Slf4j;

/**
 * @description: 帆软报表Controller
 * @createDate: 2023/5/26 9:11
 * @author: sunyfa
 */
@Slf4j
@RestController
@RequestMapping("/semc/fineReport")
public class FineReporterController extends BasicController<IFineReportService, FineReport> {
    @Resource
    private IFolderService folderService;

    @Resource
    private IFineReportService fineReportService;

    @Resource
    private IBizObjAuthRelService bizObjAuthRelService;

    @Resource
    private ICacheService cacheService;

    @Resource
    private MessageUtils messageUtils;

    @Resource
    private Converter converter;
    /**
     * 查询管理后台目录树
     *
     * @return ResponseEntity
     */
    @GetMapping("/qryFolder")
    public ResponseEntity<BaseResultDTO<List<FolderViewVo>>> qryFolder() {
        try {
            List<FolderViewVo> ResponseEntityWrapperUtilUtil = fineReportService.qryFineReportFolder(BusinessSceneEnum.BACK_MANAGEMENT.getValue());
            log.info("FineReporterController qryFolder result:{}", ResponseEntityWrapperUtilUtil);
            return ResponseEntityWrapperUtil.wrapperOk(ResponseEntityWrapperUtilUtil);
        } catch (Exception e) {
            log.error("FineReporterController qryFolder exception", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 查询管理后台当前目录下的所有帆软报表，同时查询权限对象
     *
     * @param folderId 报表id
     * @return
     */
    @GetMapping("/qryFineReport/{folderId}")
    public ResponseEntity<BaseResultDTO<List<FineReportVo>>> qryFineReport(@PathVariable("folderId") @NotNull(message = "{NotNull" +
            ".DeleteFolderVo.id}") @Valid Long folderId) {
        try {
            // 判断目录是否存在
            if (!isExistFolder(folderId)) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.FOLDER_NON_EXISTS));
            }
            List<FineReportVo> resultFineReportList = fineReportService.qryFineReport(folderId);
            return ResponseEntityWrapperUtil.wrapperOk(resultFineReportList);
        } catch (Exception e) {
            log.error("FineReporterController qryFineReport exception", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 查询portal页面目录树
     *
     * @return ResponseEntity
     */
    @GetMapping("/qryCurrentUserFolder")
    public ResponseEntity<BaseResultDTO<List<FolderViewVo>>> qryCurrentUserFolder() {
        try {
            List<FolderViewVo> ResponseEntityWrapperUtilUtil = fineReportService.qryFineReportFolder(BusinessSceneEnum.PORTAL.getValue());
            log.info("FineReporterController qryCurrentUserFolder end, result:{}", JSONObject.toJSONString(ResponseEntityWrapperUtilUtil));
            return ResponseEntityWrapperUtil.wrapperOk(ResponseEntityWrapperUtilUtil);
        } catch (Exception e) {
            log.error("FineReporterController qryCurrentUserFolder exception", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 查询portal当前目录下当前用户所属权限内的帆软报表
     *
     * @param folderId 报表id
     * @param empId    员工id
     * @return
     */
    @GetMapping("/qryCurrentUserFineReport/{folderId}")
    public ResponseEntity<BaseResultDTO<List<FineReportVo>>> qryCurrentUserFineReport(@PathVariable("folderId") @NotNull(message =
            "{NotNull.DeleteFolderVo.id}") @Valid Long folderId,
                                                      @RequestParam @NotBlank(message = "{NotBlank.DeleteFolderVo.empId}") @Valid String empId) {
        try {
            // 判断目录是否存在
            if (!isExistFolder(folderId)) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.FOLDER_NON_EXISTS));
            }
            // 查询当前用户权限内的帆软报表
            List<FineReportVo> resultFineReportList = fineReportService.qryCurrentUserFineReport(folderId, empId);
            return ResponseEntityWrapperUtil.wrapperOk(resultFineReportList);
        } catch (Exception e) {
            log.error("FineReporterController qryCurrentUserFineReport exception", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 新增/修改目录
     *
     * @return ResponseEntity
     */
    @PostMapping("/addOrModFolder")
    public ResponseEntity<?> addOrModFolder(@RequestBody @Valid FolderVo folderVo) {
        try {
            log.info("FineReporterController addOrModFolder begin folderVo:{}", folderVo);
            if (Objects.nonNull(folderVo.getParentFolderId()) && StringUtils.isEmpty(folderVo.getParentFolderId() + "")) {
                folderVo.setParentFolderId(null);
            }

            boolean isNameExist;
            // 新增目录
            if (Objects.isNull(folderVo.getId())) {
                // 层级不能为空
                if (Objects.isNull(folderVo.getLevel())) {
                    return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.FOLDER_LEVEL_NOT_EXISTS, messageUtils.getMessage(I18NKey.FOLDER_LEVEL_NOT_EXISTS));
                }
                // 如果存在父目录，则判断父目录的层级是否已经是三级
                if (Objects.nonNull(folderVo.getParentFolderId())) {
                    Folder parentFolder = folderService.getBaseMapper().selectOne(new QueryWrapper<Folder>().eq("obj_type", FolderTypeEnum.FINE_REPORT.getValue()).eq("id", folderVo.getParentFolderId()));

                    // 判断父目录是否存在
                    if (Objects.isNull(parentFolder)) {
                        return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.FOLDER_PARENT_NOT_EXISTS, messageUtils.getMessage(I18NKey.FOLDER_ROOT_NON_EXISTS));
                    }
                    // 判断目录层级，目录不能超过三级
                    if (FolderLevelEnum.THIRD.getValue().equals(parentFolder.getLevel())) {
                        return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.FOLDER_LEVEL_ERROR, messageUtils.getMessage(I18NKey.FOLDER_CHILD_NODE_ADD_LEVEL));
                    }
                }
                isNameExist = folderService.exists(
                        new QueryWrapper<Folder>().eq("obj_type", FolderTypeEnum.FINE_REPORT.getValue()).eq("name", folderVo.getName()));
            } else {  // 修改目录
                isNameExist = folderService.exists(
                        new QueryWrapper<Folder>().eq("obj_type", FolderTypeEnum.FINE_REPORT.getValue()).eq("name", folderVo.getName()).ne("id", folderVo.getId()));
            }
            // 判断目录名称是否存在
            if (isNameExist) {
                return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.FOLDER_CHILD_CREATE_NAME_ERROR, messageUtils.getMessage(I18NKey.FOLDER_NAME_EXISTS));
            }
            Folder folder = converter.convert(folderVo,Folder.class);
            // 保存目录
            folderService.addOrModFolder(folder);
            return ResponseEntityWrapper.wrapperOk();
        } catch (Exception e) {
            log.error("FineReporterController addOrModFolder exception", e);
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 删除目录，同时删除当前目录下所有子目录及所有报表数据和权限数据
     *
     * @return ResponseEntity
     */
    @DeleteMapping("/delFolder")
    public ResponseEntity<?> delFolder(@RequestBody @Valid DeleteFolderVo deleteFolderVo) {
        if (Objects.nonNull(deleteFolderVo.getParentFolderId()) && StringUtils.isEmpty(deleteFolderVo.getParentFolderId() + "")) {
            deleteFolderVo.setParentFolderId(null);
        }
        try {
            log.info("FineReporterController delFolder deleteFolderVo:{}", JSONObject.toJSONString(deleteFolderVo));
            // 如果删除的是一级目录，则判断最少要有一个一级目录，不允许将一级目录全部删除
            if (Objects.isNull(deleteFolderVo.getParentFolderId())) {
                boolean isExistsRoot = folderService.exists(
                        new QueryWrapper<Folder>().isNull("parent_folder_id").ne("id", deleteFolderVo.getId()).eq("obj_type", FolderTypeEnum.FINE_REPORT.getValue()));
                if (!isExistsRoot) {
                    return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.FOLDER_ROOT_DELETE));
                }
            }
            folderService.delFolder(deleteFolderVo.getId(), FolderTypeEnum.FINE_REPORT.getValue());
            return ResponseEntityWrapper.wrapperOk();
        } catch (Exception e) {
            log.error("FineReporterController delFolder exception", e);
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 移动目录
     *
     * @return ResponseEntity
     */
    @PostMapping("/moveFolder")
    public ResponseEntity<BaseResultDTO<List<FolderViewVo>>> moveFolder(@RequestBody @Validated MoveFolderListVo moveFolderVoListVo) {
        try {
            log.info("FineReporterController moveFolder moveFolderVoListVo:{}", JSONObject.toJSONString(moveFolderVoListVo));
            List<MoveFolderVo> moveFolderVoList = moveFolderVoListVo.getMoveFolderVoList();

            // 校验目录层级是否合法
            Map<Long, MoveFolderChildVo> childrenFolderMap = Maps.newHashMap();
            moveFolderVoList.forEach(moveFolderVo -> {
                // 构建非一级目录信息
                buildChildFolderMap(childrenFolderMap, moveFolderVo.getChildren());
            });
            if (MapUtils.isNotEmpty(childrenFolderMap)) {
                for (MoveFolderChildVo moveFolderChildVo : childrenFolderMap.values()) {
                    if (moveFolderChildVo.getLevel() > FolderLevelEnum.THIRD.getValue()) {
                        return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.FOLDER_LEVEL_ERROR, messageUtils.getMessage(I18NKey.FOLDER_LEVEL_RANGE_ERROR));
                    }
                }
            }

            // 更新目录树
            folderService.moveFolder(moveFolderVoList, childrenFolderMap);
            // 移动后返回最新的目录树
            List<FolderViewVo> ResponseEntityWrapperUtilUtil = fineReportService.qryFineReportFolder(BusinessSceneEnum.BACK_MANAGEMENT.getValue());
            return ResponseEntityWrapperUtil.wrapperOk(ResponseEntityWrapperUtilUtil);
        } catch (Exception e) {
            log.error("FineReporterController moveFolder exception", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 构建非一级目录信息
     *
     * @param childrenFolderMap  非一级目录
     * @param childrenFolderList 非一级目录列表
     */
    private void buildChildFolderMap(Map<Long, MoveFolderChildVo> childrenFolderMap, List<MoveFolderChildVo> childrenFolderList) {
        if (CollectionUtils.isNotEmpty(childrenFolderList)) {
            childrenFolderList.forEach(childFolder -> {
                childrenFolderMap.put(childFolder.getId(), childFolder);
                buildChildFolderMap(childrenFolderMap, childFolder.getChildren());
            });
        }
    }

    /**
     * 保存帆软报表，同时需要保存权限对象
     *
     * @param fineReportVo 保存报表的入参
     * @return
     */
    @PostMapping("/addOrModFineReport")
    public ResponseEntity<?> addOrModFineReport(@RequestBody @Valid FineReportVo fineReportVo) {
        try {
            log.info("FineReporterController addOrModFineReport begin fineReportVo:{}", JSONObject.toJSONString(fineReportVo));
            // 判断目录是否存在
            if (!isExistFolder(fineReportVo.getFolderId())) {
                return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.FOLDER_NON_EXISTS));
            }
            // 判断报表名称是否唯一
            boolean existsName = fineReportService.isReportExist(1, fineReportVo);
            if (existsName) {
                return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.FINE_REPORT_NAME_EXISTS));
            }
            // 判断报表链接是否唯一
            boolean existsUrl = fineReportService.isReportExist(2, fineReportVo);
            if (existsUrl) {
                return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.FINE_REPORT_LINK_EXISTS));
            }

            // 保存/更新报表数据, 保存/更新权限
            FineReport fineReport = converter.convert(fineReportVo, FineReport.class);
            fineReportService.addOrModFineReport(fineReport, fineReportVo.getOrg(), fineReportVo.getRole(), fineReportVo.getUser());
            return ResponseEntityWrapper.wrapperOk();
        } catch (Exception e) {
            log.error("FineReporterController addOrModFineReport exception", e);
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR,
                    messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 判断目录是否存在
     *
     * @return
     */
    private boolean isExistFolder(Long folderId) {
        return folderService.exists(
                new QueryWrapper<Folder>().eq("obj_type", FolderTypeEnum.FINE_REPORT.getValue()).eq("id", folderId));
    }

    /**
     * 移动报表
     *
     * @param moveFineReportVo 入参
     * @return
     */
    @PostMapping("/moveFineReport")
    public ResponseEntity<?> moveFineReport(@RequestBody @Valid MoveFineReportVo moveFineReportVo) {
        try {
            log.info("FineReporterController moveFineReport begin moveFineReportVo:{}", moveFineReportVo);
            // 判断目录是否存在
            boolean isFolderExist = folderService.exists(
                    new QueryWrapper<Folder>().eq("obj_type", FolderTypeEnum.FINE_REPORT.getValue())
                            .in("id", Arrays.asList(moveFineReportVo.getSourceFolderId(), moveFineReportVo.getTargetFolderId())));
            if (!isFolderExist) {
                return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.FOLDER_NON_EXISTS));
            }
            // 源目录和目标目录不能相同
            if (Objects.equals(moveFineReportVo.getSourceFolderId(), moveFineReportVo.getTargetFolderId())) {
                return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.FINE_REPORT_FOLDER));
            }
            // 查询目标目录下报表的最大顺序
            Integer maxOrder = fineReportService.queryMaxOrder(moveFineReportVo.getTargetFolderId(),
                    AppAuthContextHolder.getContext().getAuthoredUser().getTenantId());
            log.debug("FineReporterController target folder id:{}, report max order:{}", moveFineReportVo.getTargetFolderId(), maxOrder);

            // 查询当前报表,重新指定目录和顺序
            FineReport fineReport = fineReportService.getBaseMapper().selectById(moveFineReportVo.getId());
            fineReport.setFolderId(moveFineReportVo.getTargetFolderId());
            fineReport.setReportOrder((maxOrder == null ? 0 : maxOrder) + 1);
            fineReportService.updateById(fineReport);
            return ResponseEntityWrapper.wrapperOk();
        } catch (Exception e) {
            log.error("FineReporterController moveFineReport exception, param:{}", JSONObject.toJSONString(moveFineReportVo), e);
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 删除帆软报，同时需要删除权限对象
     *
     * @param id 报表id
     * @return
     */
    @DeleteMapping("/delFineReport/{id}")
    public ResponseEntity<?> delFineReport(@PathVariable("id") @NotNull(message = "{NotNull.Folder.id}") @Valid Long id) {
        try {
            fineReportService.delFineReport(id);
            return ResponseEntityWrapper.wrapperOk();
        } catch (Exception e) {
            log.error("FineReporterController delFineReport exception", e);
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 新增/修改报表的查看权限对象
     *
     * @param bizObjAuthRelBatchVo 入参
     * @return
     */
    @PostMapping("/modFineReportAuth")
    public ResponseEntity<?> modFineReportAuth(@RequestBody @Valid BizObjAuthRelBatchVo bizObjAuthRelBatchVo) {
        try {
            log.info("FineReporterController modFineReportAuth begin bizObjAuthRelBatchVo:{}", bizObjAuthRelBatchVo);
            // 新增或者更新业务权限关系
            bizObjAuthRelService.addBizObjectAuthRel(bizObjAuthRelBatchVo, BizObjectTypeEnum.FINE_REPORT.getValue());
            return ResponseEntityWrapper.wrapperOk();
        } catch (Exception e) {
            log.error("FineReporterController modFineReportAuth exception", e);
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 保存帆软报表链接后缀
     *
     * @param saveReportUrlParamReq 入参
     * @return
     */
    @PostMapping("/saveReportUrlParam")
    public ResponseEntity<BaseResultDTO<Boolean>> saveReportUrlParam(@RequestBody @Valid SaveReportUrlParamReq saveReportUrlParamReq) {
        try {
            log.info("FineReporterController saveReportUrlParam saveReportUrlParamReq:{}", JSONObject.toJSONString(saveReportUrlParamReq));
            fineReportService.saveReportUrlParam(saveReportUrlParamReq);
            return ResponseEntityWrapperUtil.wrapperOk(Boolean.TRUE);
        } catch (Exception e) {
            log.error("FineReporterController save report url param exception", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 将启用停用帆软报表模块存在redis中
     *
     * @param availability 启用停用标识
     * @return
     */
    @PostMapping("/changeFineReportAvailability/{availability}")
    public ResponseEntity<?> changeFineReportAvailability(@PathVariable("availability") boolean availability) {
        try {
            // 将启用停用帆软报表模块存在redis中
            cacheService.cache("FineReport-Availability:" + AppAuthContextHolder.getContext().getAuthoredUser().getTenantId(), String.valueOf(availability), CacheServiceImpl.DEFAULT_DURATION);
            return ResponseEntityWrapper.wrapperOk();
        } catch (Exception e) {
            log.error("FineReporterController changeFineReportAvailability exception", e);
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 查询帆软报表模块是否启用
     *
     * @return
     */
    @GetMapping("/qryFineReportAvailability")
    public ResponseEntity<BaseResultDTO<Boolean>> qryFineReportAvailability() {
        try {
            // 查询redis中设置的是否可用
            Object result = cacheService.get("FineReport-Availability:" + AppAuthContextHolder.getContext().getAuthoredUser().getTenantId());
            return ResponseEntityWrapperUtil.wrapperOk(Objects.isNull(result) ? true : Boolean.parseBoolean(result.toString()));
        } catch (Exception e) {
            log.error("FineReporterController qryFineReportAvailability exception", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 根据条件查询帆软报表
     *
     * @param queryReportReq 查询条件
     * @return
     */
    @PostMapping("/queryReportByKey")
    public ResponseEntity<BaseResultDTO<List<FineReportVo>>> queryReportByCondition(@RequestBody @Valid QueryReportReq queryReportReq) {
        try {
            // 搜索报表名称
            List<String> keyList = queryReportReq.getKeyList().stream().distinct().collect(Collectors.toList());
            QueryWrapper<FineReport> reportParam = new QueryWrapper<>();
            reportParam.or(x -> keyList.forEach(key -> x.or().like("name", key)));

            // 搜索目录
            QueryWrapper<Folder> folderParam = new QueryWrapper<>();
            folderParam.eq("obj_type", FolderTypeEnum.FINE_REPORT.getValue());
            folderParam.and(x -> keyList.forEach(key -> x.or().like("name", key)));
            List<Folder> folderList = folderService.getBaseMapper().selectList(folderParam);
            if (CollectionUtils.isNotEmpty(folderList)) {
                List<Long> folderIdList = folderList.stream().map(Folder::getId).collect(Collectors.toList());
                reportParam.or().in("folder_id", folderIdList);
            }

            // 搜索权限范围数据 员工名称、角色名称、组织名称
            QueryWrapper<BizObjAuthRel> authRelParam = new QueryWrapper<>();
            authRelParam.eq("biz_obj_type", FolderTypeEnum.FINE_REPORT.getValue());
            authRelParam.and(x -> keyList.forEach(key -> x.or().like("auth_name", key)));
            List<BizObjAuthRel> authRelList = bizObjAuthRelService.getBaseMapper().selectList(authRelParam);
            if (CollectionUtils.isNotEmpty(authRelList)) {
                List<String> objIdList = authRelList.stream().map(BizObjAuthRel::getBizObjId).collect(Collectors.toList());
                reportParam.or().in("id", objIdList);
            }

            // 查询报表
            List<FineReport> reportList = fineReportService.getBaseMapper().selectList(reportParam);
            if (CollectionUtils.isEmpty(reportList)) {
                return ResponseEntityWrapperUtil.wrapperOk(Lists.newArrayList());
            }
            List<Long> folderIdList = reportList.stream().map(FineReport::getFolderId).distinct().collect(Collectors.toList());
            List<Folder> reportFolderList = folderService.getBaseMapper().selectList(new QueryWrapper<Folder>()
                    .eq("obj_type", FolderTypeEnum.FINE_REPORT.getValue()).in("id", folderIdList));
            List<FineReportVo> fineReportVoList = fineReportService.buildFineReportVoList(reportFolderList, reportList);
            return ResponseEntityWrapperUtil.wrapperOk(fineReportVoList);
        } catch (Exception e) {
            log.error("FineReporterController query report by key occur exception", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }


    /**
     * 分页查询帆软报表
     *
     * @param reportPageQueryReq 查询报表请求参数
     * @return
     */
    @PostMapping("/pageQuery")
    public ResultPageBean pageQueryReport(@Valid @RequestBody FineReportQueryReq reportPageQueryReq) {
        log.info("/semc/fineReport/pageQuery param: reportPageQueryReq:{}", JSON.toJSONString(reportPageQueryReq));
        ResultPageBean resultPageBean;
        String tenantId = AppAuthContextHolder.getContext().getAuthoredUser().getTenantId();
        List<FineReportVo> fineReportVoList = new ArrayList<>();
        if (StringUtils.isEmpty(reportPageQueryReq.getQryCondition())) {
            return ResultPageBean.success(ResponseBody.getInstance(fineReportVoList, PageInfo.getPageInfo(reportPageQueryReq.getPageNum(), reportPageQueryReq.getPageSize(), 0)));
        }
        try {
            return fineReportService.fineReportSearch(tenantId, reportPageQueryReq);
        } catch (BizException bizException) {
            return ResultPageBean.bizException(BizException.getDefaultBizException(ErrorCodeConstant.ADAPT_SYSTEM_ERROR,
                    messageUtils.getMessage(I18NKey.REPORT_OPEN_ERROR)));
        } catch (Exception e) {
            log.error("/semc/fineReport/pageQuery exception", e);
            resultPageBean = ResultPageBean.sysException(new ServiceException(ErrorCodeConstant.SYSTEM_ERROR,
                    messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR)));

        }
        return resultPageBean;
    }

    /**
     * 根据id查询帆软报表
     *
     * @param reportId 报表id
     * @param scene    场景
     * @return
     */
    @GetMapping("/queryReportById")
    public ResponseEntity<BaseResultDTO<FineReport>> queryFineReportById(@RequestParam Long reportId,
                                                              @RequestParam Integer scene) {
        try {
            FineReport fineReport = fineReportService.queryFineReportById(reportId, scene);
            return ResponseEntityWrapperUtil.wrapperOk(fineReport);
        } catch (Exception e) {
            log.error("FineReporterController query fine report by id exception", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }
}