package com.digiwin.athena.semc.controller.portal;

import com.alibaba.excel.EasyExcel;
import com.baomidou.mybatisplus.core.toolkit.ObjectUtils;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.I18NKey;
import com.digiwin.athena.semc.common.PageInfoResp;
import com.digiwin.athena.semc.common.enums.BizAuthTypeEnum;
import com.digiwin.athena.semc.common.enums.ImportTypeEnum;
import com.digiwin.athena.semc.common.enums.VirtualApplicationEnum;
import com.digiwin.athena.semc.controller.BasicController;
import com.digiwin.athena.semc.dto.portal.QueryImportRecordReq;
import com.digiwin.athena.semc.dto.portal.UserLabelAuthReq;
import com.digiwin.athena.semc.entity.common.ImportRecord;
import com.digiwin.athena.semc.entity.portal.LabelSystemAuth;
import com.digiwin.athena.semc.entity.portal.LabelSystemCustom;
import com.digiwin.athena.semc.entity.portal.LabelSystemData;
import com.digiwin.athena.semc.entity.portal.LabelSystemPre;
import com.digiwin.athena.semc.env.EnvProperties;
import com.digiwin.athena.semc.proxy.iam.service.IamService;
import com.digiwin.athena.semc.proxy.iam.service.model.RoleDTO;
import com.digiwin.athena.semc.service.portal.IImportRecordService;
import com.digiwin.athena.semc.service.portal.LabelSystemAuthService;
import com.digiwin.athena.semc.service.portal.LabelSystemCustomService;
import com.digiwin.athena.semc.service.portal.LabelSystemPreService;
import com.digiwin.athena.semc.util.DateUtils;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;
import com.digiwin.athena.semc.vo.common.ImportRecordResp;
import com.digiwin.athena.semc.vo.portal.ImportExcelResp;
import com.digiwin.athena.semc.vo.portal.ImportLabelExcelVo;
import com.digiwin.dap.middleware.dmc.model.ShareInfo;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import javax.validation.Valid;
import java.io.File;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 组件权限导入
 *
 * @author sungqz
 * @since 2024-06-03
 */
@Slf4j
@RestController
@RequestMapping("/semc/import")
public class ImportLabelSystemController extends BasicController<IImportRecordService, ImportRecord> {

    @Resource
    private IImportRecordService importRecordService;

    @Resource
    private LabelSystemCustomService labelSystemCustomService;

    @Resource
    private LabelSystemPreService labelSystemPreService;

    @Resource
    private LabelSystemAuthService labelSystemAuthService;

    @Resource
    private IamService iamService;

    @Resource
    private MessageUtils messageUtils;

    @Resource
    private EnvProperties envProperties;

    /**
     * 允许导入的最大条数
     */
    private static final Integer MAX_LABEL_IMPORT_COUNT = 1000;


    /**
     * 获取导入模板的文件id
     *
     * @return 返回
     */
    @PostMapping("/queryFileId")
    public ResponseEntity<BaseResultDTO<String>> queryFileId() {
        try {
            String labelFileId = envProperties.getLabelFileId();
            log.info("envProperties | get label file id:{}", labelFileId);
            return ResponseEntityWrapperUtil.wrapperOk(labelFileId);
        } catch (Exception e) {
            log.error("query file id error.", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/import/queryFileId, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 导入用户组件关系
     *
     * @param userLabelAuthReq 入参
     * @return 返回
     */
    @PostMapping("/importLabel")
    public ResponseEntity<BaseResultDTO<ImportExcelResp>> importLabel(@RequestBody @Valid UserLabelAuthReq userLabelAuthReq) {
        ImportExcelResp excelResp = new ImportExcelResp();
        try {
            // 读取报表内容
            List<ImportLabelExcelVo> excelVoList = importRecordService.readLabelExcel(userLabelAuthReq.getFileId());
            if (CollectionUtils.isEmpty(excelVoList)) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.USER_LABEL_IMPORT_NOT_EMPTY));
            }
            // 校验数量
            if (excelVoList.size() > MAX_LABEL_IMPORT_COUNT) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, "数据量不能超过" + MAX_LABEL_IMPORT_COUNT + "条");
            }
            List<String> excelStr = Lists.newArrayList();

            // 获取所当前租户下所有用户
            List<RoleDTO> userList = iamService.queryAllUserList();
            Map<String, RoleDTO> userIdMap = userList.stream().collect(Collectors.toMap(RoleDTO::getId, y -> y, (a, b) -> a));

            // 获取当前租户下所有角色列表
            List<RoleDTO> roleList = iamService.queryAllRoleList();
            Map<String, RoleDTO> roleIdMap = roleList.stream().collect(Collectors.toMap(RoleDTO::getId, y -> y, (a, b) -> a));

            // 获取当前租户下所有自定义组件
            List<LabelSystemCustom> customList = labelSystemCustomService.getBaseMapper().selectList(Wrappers.emptyWrapper());
            Map<String, LabelSystemCustom> customNameMap = customList.stream().collect(Collectors.toMap(LabelSystemCustom::getNameZh, y -> y, (a, b) -> a));

            // 获取当前租户下所有预设组件
            List<LabelSystemPre> preList = labelSystemPreService.initPreData(Constants.ClientTypeEnum.PC.getValue());
            Map<String, LabelSystemPre> preNameMap = preList.stream().collect(Collectors.toMap(LabelSystemPre::getName, y -> y, (a, b) -> a));

            // 获取当前租户下所有设置过的用户组件权限
            List<LabelSystemAuth> labelAuthList = labelSystemAuthService.queryAuthListByCondition(Arrays.asList(BizAuthTypeEnum.USER.getValue(), BizAuthTypeEnum.ROLE.getValue()));
            Map<Long, Map<Integer, List<LabelSystemAuth>>> authMap = labelAuthList.stream()
                    .collect(Collectors.groupingBy(LabelSystemAuth::getAuthId, Collectors.groupingBy(LabelSystemAuth::getLabelType)));

            // 校验通过的记录列表
            List<LabelSystemAuth> userAuthRelList = Lists.newArrayList();
            int successNum = 0;
            // 校验失败的记录列表
            List<ImportLabelExcelVo> failList = Lists.newArrayList();
            int failNum = 0;
            String tenantId = AppAuthContextHolder.getContext().getAuthoredUser().getTenantId();
            for (ImportLabelExcelVo excelVo : excelVoList) {
                // 校验
                StringBuilder errorSb = validate(excelVo, userIdMap, roleIdMap, customNameMap, preNameMap, excelStr);
                if (StringUtils.isNotBlank(errorSb.toString())) {
                    failNum++;
                    excelVo.setFailDesc(errorSb.substring(0, errorSb.length() - 1));
                    failList.add(excelVo);
                    continue;
                }

                RoleDTO roleDTO = roleIdMap.get(excelVo.getRoleEmpId());
                RoleDTO userDTO = ObjectUtils.isEmpty(userIdMap.get(excelVo.getRoleEmpId())) ? userIdMap.get(tenantId + "$" + excelVo.getRoleEmpId()) : userIdMap.get(excelVo.getRoleEmpId());
               /* if (ObjectUtils.isEmpty(roleDTO)) {
                    failNum++;
                    excelVo.setFailDesc("用户信息不存在");
                    failList.add(excelVo);
                    continue;
                }*/
                // 校验成功，构建组件权限对象
                successNum++;
                LabelSystemAuth labelSystemAuth = new LabelSystemAuth();
                // 优先取角色
                if (ObjectUtils.isNotEmpty(roleDTO)) {
                    labelSystemAuth.setAuthId(roleDTO.getSid());
                    labelSystemAuth.setAuthName(roleDTO.getName());
                    labelSystemAuth.setAuthType(BizAuthTypeEnum.ROLE.getValue());
                } else {
                    labelSystemAuth.setAuthId(userDTO.getSid());
                    labelSystemAuth.setAuthName(userDTO.getName());
                    labelSystemAuth.setAuthType(BizAuthTypeEnum.USER.getValue());
                }
                if (Constants.LabelTypeEnum.SYSTEM_CUSTOM.getName().equals(excelVo.getLabelSource())) {
                    LabelSystemCustom labelSystemCustom = customNameMap.get(excelVo.getLabelName());
                    labelSystemAuth.setLabelId(labelSystemCustom.getId());
                    labelSystemAuth.setLabelType(Constants.LabelTypeEnum.SYSTEM_CUSTOM.getVal());
                }
                if (Constants.LabelTypeEnum.SYSTEM_PRE.getName().equals(excelVo.getLabelSource())) {
                    LabelSystemPre labelSystemPre = preNameMap.get(excelVo.getLabelName());
                    labelSystemAuth.setLabelId(labelSystemPre.getId());
                    labelSystemAuth.setLabelType(Constants.LabelTypeEnum.SYSTEM_PRE.getVal());
                }

                // 过滤表中已存在的记录，如果没有，则新增
                if (MapUtils.isEmpty(authMap.get(labelSystemAuth.getAuthId()))) {
                    userAuthRelList.add(labelSystemAuth);
                    continue;
                }
                List<LabelSystemAuth> authList = authMap.get(labelSystemAuth.getAuthId()).get(labelSystemAuth.getLabelType());
                if (CollectionUtils.isEmpty(authList)) {
                    userAuthRelList.add(labelSystemAuth);
                    continue;
                }
                List<Long> labelIdList = authList.stream().map(LabelSystemAuth::getLabelId).collect(Collectors.toList());
                if (!labelIdList.contains(labelSystemAuth.getLabelId())) {
                    userAuthRelList.add(labelSystemAuth);
                }
            }

            // 新增校验通过的组件权限关系
            if (CollectionUtils.isNotEmpty(userAuthRelList)) {
                labelSystemAuthService.saveBatch(userAuthRelList);
            }
            // 存在失败记录，则上传失败文件
            ImportRecord importRecord = ImportRecord.builder().fileId(userLabelAuthReq.getFileId()).fileName(userLabelAuthReq.getFileName()).successCount(successNum).failCount(failNum)
                    .objType(ImportTypeEnum.USER_LABEL.getValue()).tenantId(AppAuthContextHolder.getContext().getAuthoredUser().getTenantId()).createUserName(AppAuthContextHolder.getContext().getAuthoredUser().getUserName()).build();
            if (failNum > 0) {
                // 生成失败的excel文件
                ShareInfo shareInfo = importRecordService.uploadExcel(write(failList));
                excelResp.setFailFileId(shareInfo.getFileId());
                excelResp.setFailFileName(shareInfo.getFileName());
                excelResp.setFailFileUrl(shareInfo.getUrl());
                importRecord.setFailFileId(shareInfo.getFileId());
                importRecord.setFailFileName(shareInfo.getFileName());
                importRecord.setFailFileUrl(shareInfo.getUrl());
            }

            // 记录导入历史
            importRecordService.insertImportRecord(importRecord);
            // 构建返回信息
            String resultCode;
            if (successNum > 0 && failNum == 0) {
                resultCode = "2001";
            } else if (successNum > 0 && failNum > 0) {
                resultCode = "2002";
            } else if (successNum == 0 && failNum > 0) {
                resultCode = "2003";
            } else {
                resultCode = "2003";
            }
            excelResp.setResultCode(resultCode);
            excelResp.setFailCount(failNum);
            excelResp.setSuccessCount(successNum);
            return ResponseEntityWrapperUtil.wrapperOk(excelResp);
        } catch (Exception e) {
            log.error("import label excel error. param:{}", userLabelAuthReq, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/import/importLabel, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 校验excel数据
     *
     * @param excelVo excel数据
     * @return 返回
     */
    private StringBuilder validate(ImportLabelExcelVo excelVo, Map<String, RoleDTO> userIdMap, Map<String, RoleDTO> roleIdMap,
                                   Map<String, LabelSystemCustom> customNameMap, Map<String, LabelSystemPre> preNameMap, List<String> excelStr) {
        StringBuilder tenantId = new StringBuilder(AppAuthContextHolder.getContext().getAuthoredUser().getTenantId());
        StringBuilder errorSb = new StringBuilder();
        if (StringUtils.isBlank(excelVo.getRoleEmpId())) {
            errorSb.append("角色ID或用户ID不能为空").append(",");
        }
        if (StringUtils.isBlank(excelVo.getLabelName())) {
            errorSb.append("组件名称不能为空").append(",");
        }
        if (StringUtils.isBlank(excelVo.getLabelSource())) {
            errorSb.append("组件来源不能为空").append(",");
        }
        if (StringUtils.isNotBlank(excelVo.getRoleEmpId()) && ObjectUtils.isEmpty(roleIdMap.get(excelVo.getRoleEmpId()))
                && ObjectUtils.isEmpty(userIdMap.get(excelVo.getRoleEmpId()))
                && ObjectUtils.isEmpty(userIdMap.get(tenantId.append("$").append(excelVo.getRoleEmpId()).toString()))) {
            errorSb.append("角色ID或用户ID不存在").append(",");
        }
        if (StringUtils.isNotBlank(excelVo.getRoleEmpId()) && excelVo.getRoleEmpId().length() > 100) {
            errorSb.append("角色ID或用户ID的长度不能超过100个字").append(",");
        }
        if (StringUtils.isNotBlank(excelVo.getLabelName()) && excelVo.getLabelName().length() > 200) {
            errorSb.append("组件名称的长度不能超过200个字").append(",");
        }
        if (!Arrays.asList(Constants.LabelTypeEnum.SYSTEM_CUSTOM.getName(), Constants.LabelTypeEnum.SYSTEM_PRE.getName()).contains(excelVo.getLabelSource())) {
            errorSb.append("组件来源不正确").append(",");
        }
        if (Constants.LabelTypeEnum.SYSTEM_CUSTOM.getName().equals(excelVo.getLabelSource())) {
            if (StringUtils.isNotBlank(excelVo.getLabelName()) && ObjectUtils.isEmpty(customNameMap.get(excelVo.getLabelName()))) {
                errorSb.append("组件名称不存在").append(",");
            }
            if (StringUtils.isNotBlank(excelVo.getLabelName()) && ObjectUtils.isNotEmpty(customNameMap.get(excelVo.getLabelName()))) {
                LabelSystemCustom systemCustom=customNameMap.get(excelVo.getLabelName());
                if(Constants.DataCategoryEnum.CATEGORY_COUNT.getVal().equals(systemCustom.getDataCategory())){
                    errorSb.append("不支持统计组件导入").append(",");
                }
                if(Constants.DataCategoryEnum.BUSINESS_TODO.getVal().equals(systemCustom.getDataCategory())){
                    errorSb.append("不支持业务待办组件导入").append(",");
                }
                //校验不支持鼎捷雅典娜APP组件导入
                List<Long> systemIds=new ArrayList<>();
                systemIds.add(systemCustom.getId());
                List<LabelSystemData> permSystemDataList= labelSystemCustomService.queryAthenaAppList(systemIds, VirtualApplicationEnum.ATHENA_APP.getPrimaryId());
                if(CollectionUtils.isNotEmpty(permSystemDataList)){
                    errorSb.append("不支持鼎捷雅典娜APP组件导入").append(",");
                }
            }


        }
        if (Constants.LabelTypeEnum.SYSTEM_PRE.getName().equals(excelVo.getLabelSource())) {
            if (StringUtils.isNotBlank(excelVo.getLabelName()) && ObjectUtils.isEmpty(preNameMap.get(excelVo.getLabelName()))) {
                errorSb.append("组件名称不存在").append(",");
            }
            if (StringUtils.isNotBlank(excelVo.getLabelName())
                    && ObjectUtils.isNotEmpty(preNameMap.get(excelVo.getLabelName()))
                    && Constants.DataTypeEnum.TYPE_THIRD_TODO.getVal().equals(preNameMap.get(excelVo.getLabelName()).getDataType())) {
                errorSb.append("不支持三方待办组件导入").append(",");
            }
            if (StringUtils.isNotBlank(excelVo.getLabelName())
                    && ObjectUtils.isNotEmpty(preNameMap.get(excelVo.getLabelName()))
                    && Constants.DataTypeEnum.TYPE_TODO.getVal().equals(preNameMap.get(excelVo.getLabelName()).getDataType())) {
                errorSb.append("不支持待办组件导入").append(",");
            }
            //不支持轮播图组件导入
            if (StringUtils.isNotBlank(excelVo.getLabelName())
                    && ObjectUtils.isNotEmpty(preNameMap.get(excelVo.getLabelName()))
                    && Constants.DataTypeEnum.TYPE_CAROUSEL.getVal().equals(preNameMap.get(excelVo.getLabelName()).getDataType())) {
                errorSb.append("不支持轮播图组件导入").append(",");
            }
        }
        // 校验是否存在重复数据
        if (StringUtils.isNotBlank(excelVo.getRoleEmpId()) && StringUtils.isNotBlank(excelVo.getLabelName()) && StringUtils.isNotBlank(excelVo.getLabelSource())) {
            StringBuilder str = new StringBuilder(excelVo.getRoleEmpId()).append(excelVo.getLabelName()).append(excelVo.getLabelSource());
            if (excelStr.contains(str.toString())) {
                errorSb.append("存在重复记录，用户id:").append(excelVo.getRoleEmpId()).append(",组件名称：").append(excelVo.getLabelName())
                        .append(",组件来源：").append(excelVo.getLabelSource()).append(",");
            }
            excelStr.add(str.toString());
        }
        return errorSb;
    }

    /**
     * 写入excel文件
     *
     * @param list 报表数据
     * @return 返回
     */
    public File write(List<ImportLabelExcelVo> list) {
        File file = new File("权限导入模板_错误日志.xlsx");
        EasyExcel.write(file, ImportLabelExcelVo.class).sheet("权限导入").doWrite(list);
        return file;
    }

    /**
     * 查询导入记录
     *
     * @param queryImportRecordReq 入参
     * @return 返回
     */
    @PostMapping("/queryImportLabelRecord")
    public ResponseEntity<BaseResultDTO<PageInfoResp<ImportRecordResp>>> queryImportRecord(@RequestBody @Valid QueryImportRecordReq queryImportRecordReq) {
        try {
            // 校验时间格式
            if ((StringUtils.isNotBlank(queryImportRecordReq.getStartTime()) && !DateUtils.validateDate(queryImportRecordReq.getStartTime(), DateUtils.DATE_TIME_NORMAL_FORMATTER)) || (StringUtils.isNotBlank(queryImportRecordReq.getEndTime()) && !DateUtils.validateDate(queryImportRecordReq.getEndTime(), DateUtils.DATE_TIME_NORMAL_FORMATTER))) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.COMMON_PARAM_ILLEGAL));
            }
            PageInfoResp<ImportRecordResp> pageInfoResp = importRecordService.queryImportRecord(queryImportRecordReq, ImportTypeEnum.USER_LABEL.getValue());
            return ResponseEntityWrapperUtil.wrapperOk(pageInfoResp);
        } catch (Exception e) {
            log.error("import label excel error. param:{}", queryImportRecordReq, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/import/queryImportLabelRecord, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }
}