package com.digiwin.athena.semc.controller.portal;

import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.write.style.column.LongestMatchColumnWidthStyleStrategy;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.appcore.util.ResponseEntityWrapper;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.I18NKey;
import com.digiwin.athena.semc.common.PageInfoResp;
import com.digiwin.athena.semc.common.enums.FolderLevelEnum;
import com.digiwin.athena.semc.common.enums.FolderTypeEnum;
import com.digiwin.athena.semc.controller.BasicController;
import com.digiwin.athena.semc.dto.portal.QueryImportRecordReq;
import com.digiwin.athena.semc.entity.common.Folder;
import com.digiwin.athena.semc.entity.common.ImportRecord;
import com.digiwin.athena.semc.entity.portal.FineReport;
import com.digiwin.athena.semc.service.common.IFolderService;
import com.digiwin.athena.semc.service.portal.IFineReportService;
import com.digiwin.athena.semc.service.portal.IImportRecordService;
import com.digiwin.athena.semc.util.DateUtils;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;
import com.digiwin.athena.semc.vo.common.*;
import com.digiwin.athena.semc.vo.portal.*;
import com.digiwin.dap.middleware.dmc.model.ShareInfo;
import com.google.common.base.Joiner;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import io.github.linpeilie.Converter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;
import javax.validation.Valid;
import java.io.File;
import java.nio.charset.StandardCharsets;
import java.time.LocalDate;
import java.util.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.BinaryOperator;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @description: 帆软报表导入Controller
 * @createDate: 2023/8/16
 * @author: sungqz
 */
@Slf4j
@RestController
@RequestMapping("/semc/importReport")
public class ImportReportController extends BasicController<IImportRecordService, ImportRecord> {

    @Resource
    private IImportRecordService importRecordService;

    @Resource
    private IFolderService folderService;

    @Resource
    private IFineReportService fineReportService;

    @Resource
    private MessageUtils messageUtils;

    @Resource
    private Converter converter;

    /**
     * 导入帆软报表
     *
     * @param fileId 文件id
     * @return
     */
    @GetMapping("/importExcel")
    public ResponseEntity<?> importExcel(@RequestParam String fileId, @RequestParam String fileName) {
        ImportExcelResp excelResp = new ImportExcelResp();
        try {
            // 读取报表内容
            List<ImportExcelVo> excelVoList = importRecordService.readExcel(fileId);
            if (CollectionUtils.isEmpty(excelVoList)) {
                return ResponseEntityWrapper.wrapperOk(excelVoList);
            }
            if (excelVoList.size() > 1000) {
                excelResp.setResultCode("0");
                return ResponseEntityWrapper.wrapperOk(excelResp);
            }

            int successNum = 0;
            int failNum = 0;
            List<Folder> allFolderList = folderService.list(new QueryWrapper<Folder>().eq("obj_type", FolderTypeEnum.FINE_REPORT.getValue()).orderByAsc("folder_order"));

            // 一级目录最大值
            Folder maxOrderFolder = allFolderList.stream().filter(x -> FolderLevelEnum.FIRST.getValue().equals(x.getLevel())).max(Comparator.comparing(Folder::getFolderOrder)).get();
            AtomicInteger firstOrder = new AtomicInteger(maxOrderFolder.getFolderOrder());

            // 子目录中排序最大的目录
            Map<Long, Integer> childOrderMap = Maps.newHashMap();
            for (Folder folder : allFolderList) {
                // 三级目录
                if (FolderLevelEnum.THIRD.getValue().equals(folder.getLevel())) {
                    continue;
                }
                Folder tempFolder = allFolderList.stream().filter(x -> !FolderLevelEnum.FIRST.getValue().equals(x.getLevel()) && x.getParentFolderId().equals(folder.getId()))
                        .max(Comparator.comparing(Folder::getFolderOrder)).orElse(null);
                childOrderMap.put(folder.getId(), tempFolder == null ? 0 : tempFolder.getFolderOrder());
            }

            List<String> nameList = Lists.newArrayList();
            List<String> urlList = Lists.newArrayList();
            List<FineReport> fineReportList = fineReportService.list();
            if (CollectionUtils.isNotEmpty(fineReportList)) {
                nameList.addAll(fineReportList.stream().map(FineReport::getName).collect(Collectors.toList()));
                urlList.addAll(fineReportList.stream().map(FineReport::getUrl).collect(Collectors.toList()));
            }
            // 新增目录或报表
            List<ImportExcelVo> failList = Lists.newArrayList();

            // 目录下现有报表的最大排序值
            Map<Long, FineReport> orderMap = fineReportList.stream().peek(x -> {
                if (x.getReportOrder() == null) {
                    x.setReportOrder(0);
                }
            }).collect(Collectors.toMap(FineReport::getFolderId, fineReport -> fineReport, BinaryOperator.maxBy(Comparator.comparing(FineReport::getReportOrder))));
            Map<String, Integer> reportOrderMap = Maps.newHashMap();
            allFolderList.forEach(x -> reportOrderMap.put(x.getName(), orderMap.get(x.getId()) != null ? orderMap.get(x.getId()).getReportOrder() : 0));

            for (ImportExcelVo excelVo : excelVoList) {
                List<String> allNameList = allFolderList.stream().map(Folder::getName).collect(Collectors.toList());
                // 基础校验
                StringBuilder errorSb = validate(excelVo, nameList, urlList);
                if (StringUtils.isNotBlank(errorSb.toString())) {
                    failNum++;
                    excelVo.setFailDesc(errorSb.substring(0, errorSb.length() - 1));
                    failList.add(excelVo);
                    continue;
                }
                // 如果一级目录是新的
                if (!allNameList.contains(excelVo.getFirstFolderName())) {
                    if (allNameList.contains(excelVo.getSecondFolderName())) {
                        errorSb.append("2级目录已存在:").append(excelVo.getSecondFolderName()).append(",");
                    }
                    if (allNameList.contains(excelVo.getThirdFolderName())) {
                        errorSb.append("3级目录已存在:").append(excelVo.getThirdFolderName()).append(",");
                    }
                    if (StringUtils.isNotBlank(errorSb.toString())) {
                        failNum++;
                        excelVo.setFailDesc(errorSb.substring(0, errorSb.length() - 1));
                        failList.add(excelVo);
                        continue;
                    }
                    // 校验通过，新建目录和报表
                    importNotExistFolderReport(allFolderList, firstOrder, excelVo, reportOrderMap);
                } else { // 如果一级目录是现有的
                    Map<String, Folder> allNameMap = allFolderList.stream().collect(Collectors.toMap(Folder::getName, Function.identity(), (a, b) -> a));
                    Folder folder = allNameMap.get(excelVo.getFirstFolderName());
                    if (!FolderLevelEnum.FIRST.getValue().equals(folder.getLevel())) {
                        errorSb.append("1级目录已存在:").append(excelVo.getFirstFolderName()).append(",");
                    }
                    if (!allNameList.contains(excelVo.getSecondFolderName()) && allNameList.contains(excelVo.getThirdFolderName())) {
                        errorSb.append("3级目录已存在:").append(excelVo.getThirdFolderName()).append(",");
                    }
                    if (StringUtils.isNotBlank(errorSb.toString())) {
                        failNum++;
                        excelVo.setFailDesc(errorSb.substring(0, errorSb.length() - 1));
                        failList.add(excelVo);
                        continue;
                    }
                    // 如果二级目录不为空
                    if (StringUtils.isNotBlank(excelVo.getSecondFolderName())) {
                        // 如果二级目录是新的，则创建
                        if (!allNameList.contains(excelVo.getSecondFolderName())) {
                            Folder secondFolder = Folder.builder().name(excelVo.getSecondFolderName()).parentFolderId(Long.valueOf(String.valueOf(folder.getId())))
                                    .level(FolderLevelEnum.SECOND.getValue()).folderOrder(childOrderMap.get(folder.getId()) + 1).objType(FolderTypeEnum.FINE_REPORT.getValue()).build();
                            folderService.getBaseMapper().insert(secondFolder);
                            allFolderList.add(secondFolder);
                            childOrderMap.put(folder.getId(), secondFolder.getFolderOrder());

                            // 如果三级是新的，则创建
                            Folder thirdFolder = new Folder();
                            if (StringUtils.isNotBlank(excelVo.getThirdFolderName())) {
                                thirdFolder = Folder.builder().name(excelVo.getThirdFolderName()).parentFolderId(Long.valueOf(String.valueOf(secondFolder.getId())))
                                        .level(FolderLevelEnum.THIRD.getValue()).folderOrder(childOrderMap.get(secondFolder.getId()) + 1).objType(FolderTypeEnum.FINE_REPORT.getValue()).build();
                                folderService.getBaseMapper().insert(thirdFolder);
                                allFolderList.add(thirdFolder);
                                childOrderMap.put(secondFolder.getId(), thirdFolder.getFolderOrder());
                            }
                            importReport(excelVo, null, secondFolder.getId(), thirdFolder.getId(), reportOrderMap);
                        } else { // 二级目录是现有的
                            Folder oldSecondFolder = allNameMap.get(excelVo.getSecondFolderName());
                            if (!folder.getId().equals(oldSecondFolder.getParentFolderId())) {
                                errorSb.append("2级目录已存在:").append(excelVo.getSecondFolderName()).append(",");
                            }
                            if (StringUtils.isNotBlank(errorSb.toString())) {
                                failNum++;
                                excelVo.setFailDesc(errorSb.substring(0, errorSb.length() - 1));
                                failList.add(excelVo);
                                continue;
                            }
                            Long thirdId = null;
                            if (StringUtils.isNotBlank(excelVo.getThirdFolderName())) {
                                Folder lastFolder = allNameMap.get(excelVo.getThirdFolderName());
                                if (lastFolder != null && !lastFolder.getParentFolderId().equals(oldSecondFolder.getId())) {
                                    errorSb.append("3级目录已存在:").append(excelVo.getSecondFolderName()).append(",");
                                }
                                if (StringUtils.isNotBlank(errorSb.toString())) {
                                    failNum++;
                                    excelVo.setFailDesc(errorSb.substring(0, errorSb.length() - 1));
                                    failList.add(excelVo);
                                    continue;
                                }
                                if (lastFolder == null) {
                                    Folder thirdFolder = Folder.builder().name(excelVo.getThirdFolderName()).parentFolderId(Long.valueOf(String.valueOf(oldSecondFolder.getId())))
                                            .level(FolderLevelEnum.THIRD.getValue()).folderOrder(childOrderMap.get(oldSecondFolder.getId()) + 1).objType(FolderTypeEnum.FINE_REPORT.getValue()).build();
                                    folderService.getBaseMapper().insert(thirdFolder);
                                    thirdId = thirdFolder.getId();
                                    allFolderList.add(thirdFolder);
                                    childOrderMap.put(oldSecondFolder.getId(), thirdFolder.getFolderOrder());
                                } else {
                                    thirdId = lastFolder.getId();
                                }
                            }
                            importReport(excelVo, null, oldSecondFolder.getId(), thirdId, reportOrderMap);
                        }
                    } else {
                        Folder oldFirstFolder = allNameMap.get(excelVo.getFirstFolderName());
                        importReport(excelVo, oldFirstFolder.getId(), null, null, reportOrderMap);
                    }
                }
                nameList.add(excelVo.getName());
                urlList.add(excelVo.getUrl());
                successNum++;
            }

            // 存在失败记录，则上传失败文件
            ImportRecord importRecord = ImportRecord.builder().fileId(fileId).fileName(fileName).successCount(successNum).failCount(failNum)
                    .objType(FolderTypeEnum.FINE_REPORT.getValue()).tenantId(AppAuthContextHolder.getContext().getAuthoredUser().getTenantId()).createUserName(AppAuthContextHolder.getContext().getAuthoredUser().getUserName()).build();
            if (failNum > 0) {
                // 生成失败的excel文件
                ShareInfo shareInfo = importRecordService.uploadExcel(write(failList));
                excelResp.setFailFileId(shareInfo.getFileId());
                excelResp.setFailFileName(shareInfo.getFileName());
                excelResp.setFailFileUrl(shareInfo.getUrl());
                importRecord.setFailFileId(shareInfo.getFileId());
                importRecord.setFailFileName(shareInfo.getFileName());
                importRecord.setFailFileUrl(shareInfo.getUrl());
            }

            // 记录导入历史
            importRecordService.insertImportRecord(importRecord);
            // 构建返回信息
            String resultCode;
            if (successNum > 0 && failNum == 0) {
                resultCode = "2001";
            } else if (successNum > 0 && failNum > 0) {
                resultCode = "2002";
            } else if (successNum == 0 && failNum > 0) {
                resultCode = "2003";
            } else {
                resultCode = "2003";
            }
            excelResp.setResultCode(resultCode);
            excelResp.setFailCount(failNum);
            excelResp.setSuccessCount(successNum);
            return ResponseEntityWrapper.wrapperOk(excelResp);
        } catch (Exception e) {
            log.error("FineReporterController import excel exception", e);
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 导入报表
     *
     * @param allFolderList 所有目录
     * @param firstOrder    一级目录
     * @param excelVo       报表行数据
     */
    private void importNotExistFolderReport(List<Folder> allFolderList, AtomicInteger firstOrder, ImportExcelVo excelVo, Map<String, Integer> reportOrderMap) {
        // 新建一级目录
        Folder firstFolder = Folder.builder().name(excelVo.getFirstFolderName()).level(FolderLevelEnum.FIRST.getValue()).folderOrder(firstOrder.incrementAndGet()).objType(FolderTypeEnum.FINE_REPORT.getValue()).build();
        folderService.getBaseMapper().insert(firstFolder);
        allFolderList.add(firstFolder);
        // 二级目录
        Folder secondFolder = new Folder();
        if (StringUtils.isNotBlank(excelVo.getSecondFolderName())) {
            secondFolder = Folder.builder().name(excelVo.getSecondFolderName()).parentFolderId(Long.valueOf(String.valueOf(firstFolder.getId()))).level(FolderLevelEnum.SECOND.getValue()).folderOrder(firstFolder.getFolderOrder() * 10 + 1).objType(FolderTypeEnum.FINE_REPORT.getValue()).build();
            folderService.getBaseMapper().insert(secondFolder);
            allFolderList.add(secondFolder);
        }
        // 三级目录
        Folder thirdFolder = new Folder();
        if (StringUtils.isNotBlank(excelVo.getThirdFolderName())) {
            thirdFolder = Folder.builder().name(excelVo.getThirdFolderName()).parentFolderId(Long.valueOf(String.valueOf(secondFolder.getId()))).level(FolderLevelEnum.THIRD.getValue()).folderOrder(secondFolder.getFolderOrder() * 10 + 1).objType(FolderTypeEnum.FINE_REPORT.getValue()).build();
            folderService.getBaseMapper().insert(thirdFolder);
            allFolderList.add(thirdFolder);
        }
        importReport(excelVo, firstFolder.getId(), secondFolder.getId(), thirdFolder.getId(), reportOrderMap);
    }

    /**
     * 新增报表
     *
     * @param excelVo        报表行数据
     * @param firstId        一级目录id
     * @param secondId       二级目录id
     * @param thirdId        三级目录id
     * @param reportOrderMap 目录下报表数量最大值
     */
    private void importReport(ImportExcelVo excelVo, Long firstId, Long secondId, Long thirdId, Map<String, Integer> reportOrderMap) {
        // 报表id
        Long reportFolderId = thirdId != null ? thirdId : secondId != null ? secondId : firstId;

        // 获取报表的顺序
        String folderName = StringUtils.isNotBlank(excelVo.getThirdFolderName()) ? excelVo.getThirdFolderName()
                : StringUtils.isNotBlank(excelVo.getSecondFolderName()) ? excelVo.getSecondFolderName() : excelVo.getFirstFolderName();
        Integer reportOrder = reportOrderMap.get(folderName);

        FineReport fineReport = FineReport.builder().name(excelVo.getName()).url(excelVo.getUrl()).comments(excelVo.getComments())
                .folderId(reportFolderId).reportOrder(reportOrder != null ? reportOrder + 1 : 1).build();
        fineReportService.addOrModFineReport(fineReport, null, null, null);
        reportOrderMap.put(folderName, fineReport.getReportOrder());
    }

    /**
     * 校验excel数据
     *
     * @param importExcelVo excel数据
     * @param nameList      报表名称列表
     * @param urlList       报表url列表
     * @return
     */
    private StringBuilder validate(ImportExcelVo importExcelVo, List<String> nameList, List<String> urlList) {
        StringBuilder errorSb = new StringBuilder();
        if (nameList.contains(importExcelVo.getName())) {
            errorSb.append("报表名称已存在").append(",");
        }
        if (urlList.contains(importExcelVo.getUrl())) {
            errorSb.append("报表链接已存在").append(",");
        }
        if (StringUtils.isBlank(importExcelVo.getName())) {
            errorSb.append("报表名称不能为空").append(",");
        }
        if (StringUtils.isNotBlank(importExcelVo.getName()) && importExcelVo.getName().length() > 50) {
            errorSb.append("报表名称长度不能超过50").append(",");
        }
        if (StringUtils.isNotBlank(importExcelVo.getComments()) && importExcelVo.getComments().length() > 500) {
            errorSb.append("报表备注长度不能超过500").append(",");
        }
        if (StringUtils.isBlank(importExcelVo.getUrl())) {
            errorSb.append("报表链接不能为空").append(",");
        }
        if (StringUtils.isNotBlank(importExcelVo.getUrl()) && importExcelVo.getUrl().length() > 500) {
            errorSb.append("报表链接长度不能超过500").append(",");
        }
        if (StringUtils.isBlank(importExcelVo.getFirstFolderName())) {
            errorSb.append("报表1级目录不能为空").append(",");
        }
        List<String> folderNameList = Arrays.asList(importExcelVo.getFirstFolderName(), importExcelVo.getSecondFolderName(), importExcelVo.getThirdFolderName());
        long count = folderNameList.stream().filter(StringUtils::isNotBlank).count();
        long disCount = folderNameList.stream().filter(StringUtils::isNotBlank).distinct().count();
        if (count != disCount) {
            errorSb.append("目录不能重复").append(",");
        }
        if (StringUtils.isNotBlank(importExcelVo.getFirstFolderName()) && importExcelVo.getFirstFolderName().length() > 30) {
            errorSb.append("1级目录长度不能超过30").append(",");
        }
        if (StringUtils.isNotBlank(importExcelVo.getSecondFolderName()) && importExcelVo.getSecondFolderName().length() > 30) {
            errorSb.append("2级目录长度不能超过30").append(",");
        }
        if (StringUtils.isNotBlank(importExcelVo.getThirdFolderName()) && importExcelVo.getThirdFolderName().length() > 30) {
            errorSb.append("3级目录长度不能超过30").append(",");
        }
        if (StringUtils.isNotBlank(importExcelVo.getFirstFolderName())
                && StringUtils.isNotBlank(importExcelVo.getThirdFolderName())
                && StringUtils.isBlank(importExcelVo.getSecondFolderName())) {
            errorSb.append("存在1级目录和3级目录时，2级目录不能为空").append(",");
        }
        if (StringUtils.isNotBlank(importExcelVo.getReportOrder()) && !importExcelVo.getReportOrder().matches("[1-9]*")) {
            errorSb.append("排序必须为正整数").append(",");
        }
        if (StringUtils.isNotBlank(importExcelVo.getReportOrder()) && importExcelVo.getReportOrder().length() > 4) {
            errorSb.append("排序不能超过4位数").append(",");
        }
        return errorSb;
    }

    /**
     * 写入excel文件
     *
     * @param list 报表数据
     * @return
     */
    public File write(List<ImportExcelVo> list) {
        File file = new File("导入报表模板_失败日志.xlsx");
        EasyExcel.write(file, ImportExcelVo.class).sheet("帆软报表").doWrite(list);
        return file;
    }

    /**
     * 查询导入记录操作人列表
     *
     * @return
     */
    @GetMapping("/queryOperatorList")
    public ResponseEntity<BaseResultDTO<List<JSONObject>>> queryOperatorList() {
        try {
            List<ImportRecord> userIdList = importRecordService.queryOperatorList();
            if (CollectionUtils.isEmpty(userIdList)) {
                return ResponseEntityWrapperUtil.wrapperOk(Lists.newArrayList());
            }
            Map<String, ImportRecord> idMap = userIdList.stream().collect(Collectors.toMap(ImportRecord::getCreateUserId, Function.identity(), (a, b) -> a));
            List<JSONObject> userList = idMap.values().stream().map(x -> {
                JSONObject jsonObject = new JSONObject();
                jsonObject.put("userId", x.getCreateUserId());
                jsonObject.put("userName", x.getCreateUserName());
                return jsonObject;
            }).collect(Collectors.toList());
            return ResponseEntityWrapperUtil.wrapperOk(userList);
        } catch (Exception e) {
            log.error("FineReporterController save report url param exception", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 查询导入记录
     *
     * @param queryImportRecordReq 入参
     * @return
     */
    @PostMapping("/queryImportRecord")
    public ResponseEntity<BaseResultDTO<PageInfoResp<ImportRecordResp>>> queryImportRecord(@RequestBody @Valid QueryImportRecordReq queryImportRecordReq) {
        try {
            // 校验时间格式
            if ((StringUtils.isNotBlank(queryImportRecordReq.getStartTime()) && !DateUtils.validateDate(queryImportRecordReq.getStartTime(), DateUtils.DATE_TIME_NORMAL_FORMATTER)) || (StringUtils.isNotBlank(queryImportRecordReq.getEndTime()) && !DateUtils.validateDate(queryImportRecordReq.getEndTime(), DateUtils.DATE_TIME_NORMAL_FORMATTER))) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.COMMON_PARAM_ILLEGAL));
            }
            PageInfoResp<ImportRecordResp> pageInfoResp = importRecordService.queryImportRecord(queryImportRecordReq, FolderTypeEnum.FINE_REPORT.getValue());
            return ResponseEntityWrapperUtil.wrapperOk(pageInfoResp);
        } catch (Exception e) {
            log.error("FineReporterController save report url param exception", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 导出报表
     *
     * @return
     */
    @GetMapping("/exportReport")
    public void exportReport(HttpServletResponse response) {
        try {
            List<ExportExcelVo> excelVoList = Lists.newArrayList();
            // 整合数据
            List<Folder> folderList = folderService.getBaseMapper().selectList(new QueryWrapper<Folder>().eq("obj_type", FolderTypeEnum.FINE_REPORT.getValue()).orderByAsc("folder_order"));
            List<FineReport> reportList = fineReportService.getBaseMapper().selectList(Wrappers.emptyWrapper());
            // 报表数据，包含角色权限
            List<FineReportVo> fineReportVoList = fineReportService.buildFineReportVoList(folderList, reportList);
            if (CollectionUtils.isNotEmpty(fineReportVoList)) {
                Map<Long, String> folderIdMap = folderList.stream().collect(Collectors.toMap(Folder::getId, Folder::getName));
                fineReportVoList.forEach(x -> {
                    ExportExcelVo exportExcelVo = converter.convert(x, ExportExcelVo.class);;
                    // 权限对象
                    StringBuilder authRel = new StringBuilder();
                    if (CollectionUtils.isNotEmpty(x.getOrg())) {
                        List<String> orgNameList = x.getOrg().stream().map(BizObjAuthRelVo::getAuthName).collect(Collectors.toList());
                        authRel.append("组织:").append(Joiner.on(",").join(orgNameList)).append("\n");
                    }
                    if (CollectionUtils.isNotEmpty(x.getRole())) {
                        List<String> roleNameList = x.getRole().stream().map(BizObjAuthRelVo::getAuthName).collect(Collectors.toList());
                        authRel.append("角色:").append(Joiner.on(",").join(roleNameList)).append("\n");
                    }
                    if (CollectionUtils.isNotEmpty(x.getUser())) {
                        List<String> userNameList = x.getUser().stream().map(BizObjAuthRelVo::getAuthName).collect(Collectors.toList());
                        authRel.append("用户:").append(Joiner.on(",").join(userNameList)).append("\n");
                    }
                    exportExcelVo.setAuthRel(authRel.toString());

                    // 处理目录名称
                    // 查询当前报表和所有父报表的层级
                    List<FolderLevelVo> folderLevelList = folderService.qryCurrentAndParentFolderList(FolderTypeEnum.FINE_REPORT.getValue(), x.getFolderId());
                    Map<Integer, Long> levelMap = folderLevelList.stream().filter(y -> y.getFolderId() != null).collect(Collectors.toMap(FolderLevelVo::getLevel, FolderLevelVo::getFolderId));
                    // 三级目录
                    exportExcelVo.setThirdFolderName(folderIdMap.get(levelMap.get(FolderLevelEnum.THIRD.getValue())));
                    // 二级目录
                    exportExcelVo.setSecondFolderName(folderIdMap.get(levelMap.get(FolderLevelEnum.SECOND.getValue())));
                    // 一级目录
                    exportExcelVo.setFirstFolderName(folderIdMap.get(levelMap.get(FolderLevelEnum.FIRST.getValue())));
                    excelVoList.add(exportExcelVo);
                });
            }

            //设置文件名
            String filename = "帆软导出报表_" + LocalDate.now().toString().replace("-", "") + ".xlsx";
            //设置下载信息
            response.setContentType("application/vnd.ms-excel");
            response.setCharacterEncoding("utf-8");
            response.setHeader("Content-disposition", "attachment;filename=" + new String(filename.getBytes(StandardCharsets.UTF_8), StandardCharsets.ISO_8859_1));
            EasyExcel.write(response.getOutputStream(), ExportExcelVo.class).sheet("导出结果").registerWriteHandler(new LongestMatchColumnWidthStyleStrategy()).doWrite(excelVoList);
        } catch (Exception e) {
            log.error("FineReporterController export report excel exception", e);
            throw new RuntimeException(e);
        }
    }
}