package com.digiwin.athena.semc.controller.portal;

import com.alibaba.fastjson.JSON;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.appcore.util.ResponseEntityWrapper;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.I18NKey;
import com.digiwin.athena.semc.dto.portal.JobSyncReq;
import com.digiwin.athena.semc.entity.bench.SyncJobInfo;
import com.digiwin.athena.semc.entity.bench.ThirdJobConfig;
import com.digiwin.athena.semc.service.bench.SyncJobInfoService;
import com.digiwin.athena.semc.service.bench.ThirdJobConfigService;
import com.digiwin.athena.semc.util.AESUtils;
import io.swagger.v3.oas.annotations.Operation;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.compress.utils.Lists;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import java.time.LocalDateTime;
import java.util.List;

/**
 * @description 作业同步
 * @author: dengkaiwen
 * @create: 2023-8-28
 */
@RestController
@RequestMapping("/semc/dataSync/")
public class JobSyncController {

    private static final Logger logger = LoggerFactory.getLogger(JobSyncController.class);

    @Resource
    private SyncJobInfoService syncJobInfoService;

    @Resource
    private MessageUtils messageUtils;

    @Autowired
    ThirdJobConfigService thirdJobConfigService;


    /**
     * 作业同步接收
     *
     * @param req 请求实体
     * @return
     * @throws Exception
     */
    @Operation(summary = "作业同步接收")
    @PostMapping("/jobSync")
    public ResponseEntity<?> jobSync(@RequestBody JobSyncReq req) {
        logger.info("jobSync param:{}", JSON.toJSONString(req));
        String bodyJsonStr = req.getBodyJsonStr();
        if (StringUtils.isEmpty(bodyJsonStr)) {
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "missing bodyJsonStr parameter");
        }
        try {
            String aesDecryptResult = AESUtils.aesDecrypt(bodyJsonStr, Constants.SYNC_SECRET);
            req = JSON.parseObject(aesDecryptResult, JobSyncReq.class);
        } catch (Exception e) {
            logger.error("jobSync parseObject error", e);
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "入参解析异常，请检查入参是否正确");
        }
        try {

            if (null == req) {
                return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "入参解析异常，请检查入参是否正确");
            }
            //判断appid是否为空
            if (StringUtils.isEmpty(req.getAppId())) {
                return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "missing appid parameter");
            }
            //判断作业是否为空
            if (CollectionUtils.isEmpty(req.getJobList())) {
                return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "作业为空");
            }

            //查询作业集成配置，校验是否配置且启用状态
            ThirdJobConfig thirdJobConfig= thirdJobConfigService.getJobConfigBy("",req.getAppId());
            if (null == thirdJobConfig) {
                return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "当前应用不在配置中，无权推送作业");
            }
            if (Constants.VALID_STATUS_UNUSABLE.equals(thirdJobConfig.getValidStatus())) {
                return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "当前应用未启用，无权推送作业");
            }
            // 推送作业的维度是账套维度
            boolean accountFlag = req.getJobList().stream().anyMatch(x -> StringUtils.isNotBlank(x.getAccountId()));
            if (accountFlag) {
                boolean flag = req.getJobList().stream().anyMatch(x -> StringUtils.isBlank(x.getAccountId()));
                if (flag) {
                    return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "missing accountId parameter");
                }
                List<String> jobCodeList = Lists.newArrayList();
                for (SyncJobInfo syncJobInfo : req.getJobList()) {
                    String jobCode = syncJobInfo.getAccountId() + "-" + syncJobInfo.getJobCode();
                    if (jobCodeList.contains(jobCode)) {
                        return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "exist repeated accountId and jobCode parameter");
                    }
                    syncJobInfo.setJobCode(jobCode);
                    jobCodeList.add(jobCode);
                }
            }
            syncJobInfoService.jobSyncSave(req);
            return ResponseEntityWrapper.wrapperOk();
        } catch (Exception e) {
            //保存同步历史
            // syncJobInfoService.jobSyncHistorySave(req, 1, e.getMessage());
            logger.error("jobSync exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "作业同步异常");
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }


}
