package com.digiwin.athena.semc.controller.portal;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.util.ResponseEntityWrapper;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.ResultBean;
import com.digiwin.athena.semc.common.ResultPageBean;
import com.digiwin.athena.semc.common.enums.ApplicationTypeEnum;
import com.digiwin.athena.semc.dto.portal.KnowledgeApplicationEnableReq;
import com.digiwin.athena.semc.dto.portal.KnowledgeApplicationListReq;
import com.digiwin.athena.semc.dto.portal.KnowledgeApplicationNewDataReq;
import com.digiwin.athena.semc.dto.portal.KnowledgeApplicationNewDataResp;
import com.digiwin.athena.semc.entity.applink.AppLinkDTO;
import com.digiwin.athena.semc.entity.portal.KnowledgeApplicationRef;
import com.digiwin.athena.semc.proxy.ania.service.AniaService;
import com.digiwin.athena.semc.proxy.ania.service.model.KnowledgeAssistantDTO;
import com.digiwin.athena.semc.service.applink.AppLinkService;
import com.digiwin.athena.semc.service.portal.IknowledgeApplicationService;
import com.digiwin.athena.semc.util.DateUtils;
import io.swagger.v3.oas.annotations.Operation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.time.LocalDateTime;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;


/**
 * @description 列表查询
 */
@Slf4j
@RestController
@RequestMapping("/semc/api/knowledge/application")
public class KnowledgeApplicationController {
    private static final Logger logger = LoggerFactory.getLogger(KnowledgeApplicationController.class);

    @Autowired
    private IknowledgeApplicationService knowledgeApplicationService;

    @Autowired
    private AppLinkService appLinkService;

    @Autowired
    private AniaService aniaService;

    @Operation(summary = "查询知识库应用关系列表")
    @PostMapping("/list")
    public ResultPageBean list(@RequestBody @Valid KnowledgeApplicationListReq knowledgeApplicationListReq) throws Exception {
        try {
            String tenantId = AppAuthContextHolder.getContext().getAuthoredUser().getTenantId();
            knowledgeApplicationListReq.setTenantId(tenantId);
            return ResultPageBean.success(knowledgeApplicationService.getKnowledgeApplicationList(knowledgeApplicationListReq));
        } catch (Exception e) {
            logger.error("query list error. knowledgeApplicationListReq:{}", knowledgeApplicationListReq, e);
            throw new Exception("查询知识库应用关系列表");
        }
    }

    @Operation(summary = "新增知识库应用关系")
    @PostMapping("/batchInsert")
    public ResultBean batchInsert(@RequestBody List<KnowledgeApplicationNewDataReq> reqList) throws Exception {
        try {
            String tenantId = AppAuthContextHolder.getContext().getAuthoredUser().getTenantId();
            if (CollectionUtils.isNotEmpty(reqList) && reqList.size() > 20) {
                throw new Exception("批量数据大于20");
            }
            KnowledgeApplicationNewDataResp resp = new KnowledgeApplicationNewDataResp();
            List<String> repeatApplicationIdList = new ArrayList<>();
            List<String> repeatKnowledgeIdList = new ArrayList<>();
            resp.setRepeatKnowledgeIdList(repeatKnowledgeIdList);
            resp.setRepeatApplicationIdList(repeatApplicationIdList);

            QueryWrapper<KnowledgeApplicationRef> param = new QueryWrapper<>();
            param.eq("tenant_id", AppAuthContextHolder.getContext().getAuthoredUser().getTenantId());
            param.eq("del_flag", 0);
            List<KnowledgeApplicationRef> knowledgeApplicationRefs = knowledgeApplicationService.getBaseMapper().selectList(param);

            if (CollectionUtils.isNotEmpty(knowledgeApplicationRefs)) {
                List<String> applicationList = knowledgeApplicationRefs.stream().map(
                        d -> "" + d.getApplicationType() + "@@@" + d.getApplicationId()

                ).collect(Collectors.toList());
                List<String> knowledgeList = knowledgeApplicationRefs.stream().map(KnowledgeApplicationRef::getKnowledgeId).collect(Collectors.toList());

                for (KnowledgeApplicationNewDataReq knowledgeApplicationNewDataReq : reqList) {
                    String appIdType = "" + knowledgeApplicationNewDataReq.getApplicationType() + "@@@" + knowledgeApplicationNewDataReq.getApplicationId();
                    if (applicationList.contains(appIdType)) {
                        repeatApplicationIdList.add(appIdType);
                    }
                    if (knowledgeList.contains(knowledgeApplicationNewDataReq.getKnowledgeId())) {
                        repeatKnowledgeIdList.add(knowledgeApplicationNewDataReq.getKnowledgeId());
                    }
                }
            }

            if (CollectionUtils.isNotEmpty(repeatApplicationIdList) || CollectionUtils.isNotEmpty(repeatKnowledgeIdList)) {
                return ResultBean.success(resp);
            }
            if (CollectionUtils.isNotEmpty(reqList)) {
                for (KnowledgeApplicationNewDataReq knowledgeApplicationNewDataReq : reqList) {
                    knowledgeApplicationNewDataReq.setTenantId(tenantId);
                    knowledgeApplicationNewDataReq.setModifyUserName(AppAuthContextHolder.getContext().getAuthoredUser().getUserName());
                    knowledgeApplicationNewDataReq.setCreateUserName(AppAuthContextHolder.getContext().getAuthoredUser().getUserName());
                }
            }
            knowledgeApplicationService.batchInsertRecord(reqList);
            return ResultBean.success(resp);
        } catch (Exception e) {
            logger.error("batchInsert error. reqList:{}", reqList, e);
            throw new Exception("查询知识库应用关系列表");
        }
    }

    @Operation(summary = "更新知识库应用关系")
    @PostMapping("/updateRecord")
    public ResultBean updateRecord(@RequestBody KnowledgeApplicationNewDataReq req) throws Exception {
        try {
            KnowledgeApplicationNewDataResp resp = new KnowledgeApplicationNewDataResp();
            List<String> repeatApplicationIdList = new ArrayList<>();
            List<String> repeatKnowledgeIdList = new ArrayList<>();
            resp.setRepeatKnowledgeIdList(repeatKnowledgeIdList);
            resp.setRepeatApplicationIdList(repeatApplicationIdList);

            QueryWrapper<KnowledgeApplicationRef> param = new QueryWrapper<>();
            param.eq("tenant_id", AppAuthContextHolder.getContext().getAuthoredUser().getTenantId());
            param.eq("del_flag", 0);
            List<KnowledgeApplicationRef> knowledgeApplicationRefs = knowledgeApplicationService.getBaseMapper().selectList(param);

            KnowledgeApplicationRef nowData = knowledgeApplicationService.getBaseMapper().selectById(req.getId());

            if (CollectionUtils.isNotEmpty(knowledgeApplicationRefs)) {
                List<String> applicationList = knowledgeApplicationRefs.stream().map(
                        d -> "" + d.getApplicationType() + "@@@" + d.getApplicationId()

                ).collect(Collectors.toList());
                List<String> knowledgeList = knowledgeApplicationRefs.stream().map(KnowledgeApplicationRef::getKnowledgeId).collect(Collectors.toList());

                String appIdType = "" + req.getApplicationType() + "@@@" + req.getApplicationId();
                String nowAppIdType = "" + nowData.getApplicationType() + "@@@" + nowData.getApplicationId();

                if (!Objects.equals(nowAppIdType, appIdType) && applicationList.contains(appIdType)) {
                    repeatApplicationIdList.add(appIdType);
                }

                if (!Objects.equals(nowData.getKnowledgeId(), req.getKnowledgeId()) && knowledgeList.contains(req.getKnowledgeId())) {
                    repeatKnowledgeIdList.add(req.getKnowledgeId());
                }
            }

            if (CollectionUtils.isNotEmpty(repeatApplicationIdList) || CollectionUtils.isNotEmpty(repeatKnowledgeIdList)) {
                return ResultBean.success(resp);
            }

            KnowledgeApplicationRef ref = new KnowledgeApplicationRef();
            ref.setApplicationId(req.getApplicationId());
            ref.setAppPrimaryId(req.getAppPrimaryId());
            ref.setKnowledgeId(req.getKnowledgeId());
            ref.setApplicationSource(req.getApplicationSource());
            ref.setApplicationType(req.getApplicationType());
            ref.setIsEnable(req.getIsEnable());
//            ref.setTenantId(req.getTenantId());
            ref.setId(req.getId());
            ref.setModifyUserName(AppAuthContextHolder.getContext().getAuthoredUser().getUserName());
            ref.setModifyUserId(AppAuthContextHolder.getContext().getAuthoredUser().getUserId());
            ref.setModifyTime(DateUtils.getNowTime(""));
            knowledgeApplicationService.updateById(ref);

            // 如果状态有变更
            if (!nowData.getIsEnable().equals(req.getIsEnable())) {
                // 变为启用则授权新应用
                if (Constants.VALID_STATUS_ENABLE.equals(req.getIsEnable())) {
                    log.info("updateRecord | update record authorize user app, param:{}, actionType:{}", JSONObject.toJSONString(Collections.singletonList(ref)), Constants.IamAuthorizeTypeEnum.INSERT.getVal());
                    knowledgeApplicationService.authorizeUserApp(Collections.singletonList(ref), Constants.IamAuthorizeTypeEnum.INSERT.getVal());
                }
                // 变为停用则取消授权老应用
                if (Constants.VALID_STATUS_UNUSABLE.equals(req.getIsEnable())) {
                    log.info("updateRecord | update record authorize user app, param:{}, actionType:{}", JSONObject.toJSONString(Collections.singletonList(nowData)), Constants.IamAuthorizeTypeEnum.DELETE.getVal());
                    knowledgeApplicationService.authorizeUserApp(Collections.singletonList(nowData), Constants.IamAuthorizeTypeEnum.DELETE.getVal());
                }
            }
            // 如果状态没有变更且为启用状态，则取消授权老应用，同时授权新应用
            if (nowData.getIsEnable().equals(req.getIsEnable()) && Constants.VALID_STATUS_ENABLE.equals(req.getIsEnable())) {
                if (!(nowData.getApplicationType() + "-" + nowData.getAppPrimaryId()).equals(ref.getApplicationType() + "-" + ref.getAppPrimaryId())
                        || !nowData.getKnowledgeId().equals(ref.getKnowledgeId())) {
                    log.info("updateRecord | update record authorize user app, param:{}, actionType:{}", JSONObject.toJSONString(Collections.singletonList(nowData)), Constants.IamAuthorizeTypeEnum.INSERT.getVal());
                    knowledgeApplicationService.authorizeUserApp(Collections.singletonList(nowData), Constants.IamAuthorizeTypeEnum.DELETE.getVal());
                    log.info("updateRecord | update record authorize user app, param:{}, actionType:{}", JSONObject.toJSONString(Collections.singletonList(ref)), Constants.IamAuthorizeTypeEnum.DELETE.getVal());
                    knowledgeApplicationService.authorizeUserApp(Collections.singletonList(ref), Constants.IamAuthorizeTypeEnum.INSERT.getVal());
                }
            }
            return ResultBean.success(resp);
        } catch (Exception e) {
            logger.error("updateRecord error. req:{}", req, e);
            throw new Exception("更新知识库应用关系");
        }
    }

    @Operation(summary = "修改状态")
    @PostMapping("/batchEnable")
    public ResultBean batchEnable(@RequestBody KnowledgeApplicationEnableReq req) throws Exception {
        try {
            if (CollectionUtils.isEmpty(req.getIdList()) || Objects.isNull(req.getStatus())) {
                return ResultBean.success(null);
            }
            List<KnowledgeApplicationRef> result = new ArrayList<>();
            for (Long aLong : req.getIdList()) {
                KnowledgeApplicationRef ref = new KnowledgeApplicationRef();
                ref.setIsEnable(req.getStatus());
                ref.setId(aLong);
                ref.setModifyUserName(AppAuthContextHolder.getContext().getAuthoredUser().getUserName());
                ref.setModifyUserId(AppAuthContextHolder.getContext().getAuthoredUser().getUserId());
                ref.setModifyTime(DateUtils.getNowTime(""));
                result.add(ref);
            }
            knowledgeApplicationService.updateBatchById(result);

            // 查询修改的应用详情
            QueryWrapper<KnowledgeApplicationRef> param = new QueryWrapper<>();
            param.eq("tenant_id", AppAuthContextHolder.getContext().getAuthoredUser().getTenantId());
            param.in("id", req.getIdList());
            List<KnowledgeApplicationRef> knowledgeApplicationRefs = knowledgeApplicationService.getBaseMapper().selectList(param);

            // 给用户授权(或取消授权)应用
            Integer actionType = Constants.VALID_STATUS_ENABLE.equals(req.getStatus()) ? Constants.IamAuthorizeTypeEnum.INSERT.getVal() : Constants.IamAuthorizeTypeEnum.DELETE.getVal();
            log.info("batchEnable | batch enable authorize user app, param:{}, actionType:{}", JSONObject.toJSONString(knowledgeApplicationRefs), actionType);
            knowledgeApplicationService.authorizeUserApp(knowledgeApplicationRefs, actionType);
            return ResultBean.success(null);
        } catch (Exception e) {
            throw new Exception("更新知识库应用关系");
        }
    }


    @Operation(summary = "删除知识库应用关系")
    @PostMapping("/delete")
    public ResultBean delete(@RequestBody KnowledgeApplicationEnableReq req) throws Exception {
        try {
            if (CollectionUtils.isEmpty(req.getIdList())) {
                return ResultBean.success(null);
            }
            List<KnowledgeApplicationRef> list = new ArrayList<>();
            for (Long aLong : req.getIdList()) {
                KnowledgeApplicationRef ref = new KnowledgeApplicationRef();
                ref.setId(aLong);
                ref.setDelFlag(1);
                ref.setDelTime(LocalDateTime.now());
                ref.setDelUserId(AppAuthContextHolder.getContext().getAuthoredUser().getUserId());
                list.add(ref);
            }
            knowledgeApplicationService.updateBatchById(list);

            // 查询当前已删除的应用详情
            QueryWrapper<KnowledgeApplicationRef> param = new QueryWrapper<>();
            param.eq("tenant_id", AppAuthContextHolder.getContext().getAuthoredUser().getTenantId());
            param.in("id", req.getIdList());
            List<KnowledgeApplicationRef> knowledgeApplicationRefs = knowledgeApplicationService.getBaseMapper().selectList(param);

            // 删除的知识库应用取消授权用户
            List<KnowledgeApplicationRef> enableList = knowledgeApplicationRefs.stream().filter(x -> Constants.VALID_STATUS_ENABLE.equals(x.getIsEnable())).collect(Collectors.toList());
            if (CollectionUtils.isNotEmpty(enableList)) {
                log.info("delete | delete record authorize user app, param:{}, actionType:{}", JSONObject.toJSONString(enableList), Constants.IamAuthorizeTypeEnum.DELETE.getVal());
                knowledgeApplicationService.authorizeUserApp(enableList, Constants.IamAuthorizeTypeEnum.DELETE.getVal());
            }
            return ResultBean.success(null);
        } catch (Exception e) {
            logger.error("updateRecord error. req:{}", req, e);
            throw new Exception("更新知识库应用关系");
        }
    }


    @Operation(summary = "应用下拉列表")
    @PostMapping("/applicationList")
    public ResultBean applicationList() throws Exception {
        try {
            List<JSONObject> result = new ArrayList<>();
            AuthoredUser user = AppAuthContextHolder.getContext().getAuthoredUser();
            //以预设表里的为准，删除重复的
            List<AppLinkDTO> appLinkListDTO = appLinkService.queryManageListSync(user);
            if (CollectionUtils.isEmpty(appLinkListDTO)) {
                ResultBean resultBean = new ResultBean();
                resultBean.setResponse(result);
                return resultBean;
            }
            //去重
            QueryWrapper<KnowledgeApplicationRef> param = new QueryWrapper<>();
            param.eq("tenant_id", AppAuthContextHolder.getContext().getAuthoredUser().getTenantId());
            param.eq("del_flag", 0);
            List<KnowledgeApplicationRef> knowledgeApplicationRefs = knowledgeApplicationService.getBaseMapper().selectList(param);
            List<String> kList = knowledgeApplicationRefs.stream().map(
                    d -> "" + d.getApplicationType() + "@@@" + d.getApplicationId()
            ).collect(Collectors.toList());
            appLinkListDTO = filterRepeatApp(appLinkListDTO, kList);
            if (CollectionUtils.isNotEmpty(appLinkListDTO)) {
                for (AppLinkDTO appLinkDTO : appLinkListDTO) {
                    JSONObject jsonObject = new JSONObject();
                    jsonObject.put("applicationId", appLinkDTO.getCode());
                    jsonObject.put("appPrimaryId", appLinkDTO.getPrimaryId());
                    jsonObject.put("applicationName", appLinkDTO.getName());
                    jsonObject.put("applicationSource", appLinkDTO.getDataSource());
                    jsonObject.put("applicationType", appLinkDTO.getType());
                    jsonObject.put("applicationDesc", appLinkDTO.getCode() + " " + appLinkDTO.getName());
                    result.add(jsonObject);
                }
            }
            ResultBean resultBean = new ResultBean();
            resultBean.setResponse(result);
            return resultBean;
        } catch (Exception e) {
            throw new Exception("应用下拉列表");
        }
    }

    @Operation(summary = "知识库下拉列表")
    @PostMapping("/knowledgeList")
    public ResultBean knowledgeList() throws Exception {
        try {
            List<JSONObject> result = new ArrayList<>();
            List<KnowledgeAssistantDTO> ks = aniaService.getKnowledgeAssistants(AppAuthContextHolder.getContext().getAuthoredUser().getTenantId());
            if (CollectionUtils.isEmpty(ks)) {
                ResultBean resultBean = new ResultBean();
                resultBean.setResponse(result);
                return resultBean;
            }
            //去重
            QueryWrapper<KnowledgeApplicationRef> param = new QueryWrapper<>();
            param.eq("tenant_id", AppAuthContextHolder.getContext().getAuthoredUser().getTenantId());
            param.eq("del_flag", 0);
            List<KnowledgeApplicationRef> knowledgeApplicationRefs = knowledgeApplicationService.getBaseMapper().selectList(param);
            List<String> kList = knowledgeApplicationRefs.stream().map(KnowledgeApplicationRef::getKnowledgeId).collect(Collectors.toList());
            if (CollectionUtils.isNotEmpty(ks)) {
                for (KnowledgeAssistantDTO knowledgeAssistantDTO : ks) {
                    if (StringUtils.isNoneBlank(knowledgeAssistantDTO.getAssistantCode()) && !kList.contains(knowledgeAssistantDTO.getAssistantCode())) {
                        JSONObject jsonObject = new JSONObject();
                        jsonObject.put("knowledgeId", knowledgeAssistantDTO.getAssistantCode());
                        jsonObject.put("knowledgeName", knowledgeAssistantDTO.getName());
                        jsonObject.put("knowledgeDesc", knowledgeAssistantDTO.getAssistantCode() + " " + knowledgeAssistantDTO.getName());
                        result.add(jsonObject);
                    }
                }
            }
            ResultBean resultBean = new ResultBean();
            resultBean.setResponse(result);
            return resultBean;
        } catch (Exception e) {
            throw new Exception("知识库下拉列表");
        }
    }

    private List<AppLinkDTO> filterRepeatApp(List<AppLinkDTO> appLinkListDT, List<String> kList) {
        List<AppLinkDTO> appLinkDTOList = new ArrayList<>();
        //预设表里的数据
        List<AppLinkDTO> matchedLinkList = appLinkListDT.stream().filter(x -> x.getDataSource().equals(ApplicationTypeEnum.PRESET_APPLICATION.getType())).collect(Collectors.toList());
        Map<String, AppLinkDTO> preListMap = matchedLinkList.stream().collect(Collectors.toMap(AppLinkDTO::getAppId, Function.identity(), (a, b) -> a));
        //IAM 数据
        List<AppLinkDTO> iamLinkList = appLinkListDT.stream().filter(x -> x.getDataSource().equals(ApplicationTypeEnum.IAM_APPLICATION.getType())).collect(Collectors.toList());
        Map<String, AppLinkDTO> iamListMap = iamLinkList.stream().collect(Collectors.toMap(AppLinkDTO::getPrimaryId, Function.identity(), (a, b) -> a));
        for (AppLinkDTO appLink : appLinkListDT) {
            //预设表数据，且预设表的appid 和 cs、bs、iam返回的应用code 一致，则需要过滤改应用
            AppLinkDTO iamData = iamListMap.get(appLink.getAppId());
            if (appLink.getDataSource().equals(ApplicationTypeEnum.PRESET_APPLICATION.getType()) && null != iamData) {
                continue;
            }
            AppLinkDTO preData = preListMap.get(appLink.getPrimaryId());
            //非预设表数据，且预设表的appid 和 cs、bs、iam返回的应用code 一致，则应用名称取预设表的名称
            if (!appLink.getDataSource().equals(ApplicationTypeEnum.PRESET_APPLICATION.getType()) && null != preData) {
                appLink.setName(preData.getName());
            }

            if (StringUtils.isNoneBlank(appLink.getCode()) && !kList.contains(
                    "" + appLink.getType() + "@@@" +
                            appLink.getCode())) {
                appLinkDTOList.add(appLink);
            }
        }
        return appLinkDTOList;
    }

}
