package com.digiwin.athena.semc.controller.portal;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.I18NKey;
import com.digiwin.athena.semc.common.enums.IfNotIntegerEnum;
import com.digiwin.athena.semc.entity.portal.News;
import com.digiwin.athena.semc.entity.portal.Notice;
import com.digiwin.athena.semc.entity.portal.TodoList;
import com.digiwin.athena.semc.service.portal.NewsService;
import com.digiwin.athena.semc.service.portal.NoticeService;
import com.digiwin.athena.semc.service.portal.TodoListService;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.validation.constraints.NotBlank;
import java.util.Objects;

/**
 * 待办列表(TodoList)表控制层
 *
 * @author sunyfa
 * @since 2022-12-20 11:48:51
 */
@Slf4j
@RestController
@RequestMapping("/semc/label/common")
public class LabelCommonController {
    private static final String LABEL = "label";

    private static final String ID = "id";

    private static final String IS_READ = "is_read";

    private static final String NOTICE = "notice";

    private static final String TODO = "todo";

    private static final String NEWS = "news";

    @Resource
    private TodoListService todoListService;

    @Resource
    private NoticeService noticeService;

    @Resource
    private NewsService newsService;

    @Resource
    private MessageUtils messageUtils;

    /**
     * 更新消息的状态为已读(通知公告、待办)
     * 
     * @param request
     * @param body 需要更新的消息
     * @return
     */
    @PostMapping("/read")
    public ResponseEntity<BaseResultDTO<Boolean>> read(HttpServletRequest request,
                                              @NotBlank(message = "消息体不能为空") @RequestBody JSONObject body) {

        if (Objects.isNull(body.get(LABEL)) || Objects.isNull(body.get(ID))) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR,
                messageUtils.getMessage(I18NKey.COMMON_PARAM_MISSING));
        }
        else if (!Lists.newArrayList(NOTICE, TODO, NEWS).contains(body.get(LABEL))
            || !(body.get(ID) instanceof Integer)) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR,
                messageUtils.getMessage(I18NKey.COMMON_PARAM_ILLEGAL));
        }

        try {
            boolean success = false;
            if (NOTICE.equals(body.get(LABEL))) {
                Notice notice = new Notice();
                notice.setId(Long.parseLong(body.get(ID).toString()));
                notice.setIsRead(IfNotIntegerEnum.YES.getValue());
                success = noticeService.updateById(notice);
            }
            else if (TODO.equals(body.get(LABEL))) {
                TodoList todoList = new TodoList();
                todoList.setId(Long.parseLong(body.get(ID).toString()));
                todoList.setIsRead(IfNotIntegerEnum.YES.getValue());
                success = todoListService.updateById(todoList);
            }
            else if (NEWS.equals(body.get(LABEL))) {
                News news = new News();
                news.setId(Long.parseLong(body.get(ID).toString()));
                news.setIsRead(IfNotIntegerEnum.YES.getValue());
                success = newsService.updateById(news);
            }

            if (!success) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.UPDATE_FAILED,
                    messageUtils.getMessage(I18NKey.COMMON_UPDATE_ERROR));
            }
            else {
                return ResponseEntityWrapperUtil.wrapperOk(true);
            }
        }
        catch (Exception e) {
            log.error("read exception", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR,
                messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 获取到未读消息的数量
     * 
     * @param request
     * @param label 组件名称 通知公告传notice 待办传todo
     * @return
     */
    @GetMapping("/getUnread")
    public ResponseEntity<BaseResultDTO<Long>> getUnread(HttpServletRequest request,
        @NotBlank(message = "label不能为空") @RequestParam String label) {

        if (!Lists.newArrayList(NOTICE, TODO).contains(label)) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR,
                messageUtils.getMessage(I18NKey.COMMON_PARAM_ILLEGAL));

        }

        try {
            Long unreadCount = 0L;

            if (TODO.equals(label)) {
                QueryWrapper<TodoList> wrapper = new QueryWrapper<>();
                wrapper.eq("to_do_user", AppAuthContextHolder.getContext().getAuthoredUser().getUserId());
                wrapper.eq(IS_READ, IfNotIntegerEnum.NO.getValue());

                unreadCount = todoListService.count(wrapper);
            }
            else if (NOTICE.equals(label) || NEWS.equals(label)) {
                QueryWrapper<Notice> noticeQueryWrapper = new QueryWrapper<>();
                noticeQueryWrapper.eq("notice_user", AppAuthContextHolder.getContext().getAuthoredUser().getUserId());
                noticeQueryWrapper.eq(IS_READ, IfNotIntegerEnum.NO.getValue());

                QueryWrapper<News> newsWrapper = new QueryWrapper<>();
                newsWrapper.eq("news_user", AppAuthContextHolder.getContext().getAuthoredUser().getUserId());
                newsWrapper.eq(IS_READ, IfNotIntegerEnum.NO.getValue());

                unreadCount = noticeService.count(noticeQueryWrapper) + newsService.count(newsWrapper);
            }

            return ResponseEntityWrapperUtil.wrapperOk(unreadCount);
        }
        catch (Exception e) {
            log.error("updateBrowseTime exception", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR,
                messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }
}