package com.digiwin.athena.semc.controller.portal;

import com.alibaba.fastjson.JSON;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.I18NKey;
import com.digiwin.athena.semc.common.ResultPageBean;
import com.digiwin.athena.semc.common.ServiceException;
import com.digiwin.athena.semc.service.menu.AuthService;
import com.digiwin.athena.semc.service.portal.LabelSystemAuthService;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;
import com.digiwin.athena.semc.util.SecurityLogUtil;
import com.digiwin.athena.semc.vo.common.AuthSystemLabelReq;
import com.digiwin.athena.semc.vo.common.BizObjAuthRelBatchSystemReq;
import com.digiwin.athena.semc.vo.common.UserSystemLabelVo;
import io.swagger.v3.oas.annotations.Operation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.validation.Valid;
import java.time.LocalDateTime;
import java.util.List;

/**
 * @description 组件权限控制
 * @author: dengkaiwen
 */
@RestController
@RequestMapping("/tenant/semc/system/")
public class LabelSystemAuthController {

    private static final Logger logger = LoggerFactory.getLogger(LabelSystemAuthController.class);

    @Autowired
    LabelSystemAuthService labelSystemAuthService;

    @Resource
    private MessageUtils messageUtils;

    @Resource
    private AuthService authService;


    /**
     *  分页查询组件权限列表
     * @return
     */
    @PostMapping("/pageListAuthSystem")
    public ResultPageBean pageListAuthSystem(@RequestBody AuthSystemLabelReq req){
        ResultPageBean resultPageBean;
        try {
            resultPageBean=labelSystemAuthService.pageListAuthSystem(req);
        } catch (Exception e) {
            logger.error("pageListAuthSystem Error",e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/tenant/semc/system/pageListAuthSystem");
            resultPageBean = ResultPageBean.sysException(new ServiceException(ErrorCodeConstant.LABEL_SYSTEM_CUSTOM,error));
        }
        return resultPageBean;
    }

    /**
     * 单个组件或者 多个组件的 解除关联/批量关联
     *
     * @param req 请求实体
     * @return
     * @throws Exception
     */
    @Operation(summary = "保存组件权限")
    @PostMapping("/saveSystemAuth")
    public ResponseEntity<BaseResultDTO<Integer>> saveSystemAuth(@RequestBody @Valid BizObjAuthRelBatchSystemReq req, HttpServletRequest request) {
        try {
            //用户及权限配置/工作台权限配置/组件权限配置
            //暂时不阻断，只是写入安全日志
            if (!authService.hasAuth("application-configuration")) {
                //写入安全日志
                SecurityLogUtil.writeSecurityLog(request, JSON.toJSONString(req));
            }
            return ResponseEntityWrapperUtil.wrapperOk(labelSystemAuthService.saveSystemAuth(req));
        } catch (Exception e) {
            logger.error("saveSystemAuth exception", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 全部关联/解除组件权限
     *
     * @param req 请求实体
     * @return
     * @throws Exception
     */
    @Operation(summary = "全部关联/解除组件权限")
    @PostMapping("/saveSystemAuthAll")
    public ResponseEntity<BaseResultDTO<Integer>> saveSystemAuthAll(@RequestBody AuthSystemLabelReq req, HttpServletRequest request) {
        try {
            //用户及权限配置/工作台权限配置/组件权限配置
            //暂时不阻断，只是写入安全日志
            if (!authService.hasAuth("application-configuration")) {
                //写入安全日志
                SecurityLogUtil.writeSecurityLog(request, JSON.toJSONString(req));
            }
            if(null == req.getAuthId()){
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "authId is not null");
            }
            if(null ==req.getIsRelated()){
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "isRelated is not null");
            }
            return ResponseEntityWrapperUtil.wrapperOk(labelSystemAuthService.saveSystemAuthAll(req));
        } catch (Exception e) {
            logger.error("saveSystemAuthAll exception", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }



    /**
     * 查询首页用户组件
     *
     * @return
     * @throws Exception
     */
    @Operation(summary = "查询首页用户组件")
    @PostMapping("/queryUserLayout")
    public ResponseEntity<BaseResultDTO<List<UserSystemLabelVo>>> queryUserLayout() {
        try {
            return ResponseEntityWrapperUtil.wrapperOk(labelSystemAuthService.queryUserLayout());
        } catch (Exception e) {
            logger.error("queryUserLayout exception", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }



}
