package com.digiwin.athena.semc.controller.portal;

import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.I18NKey;
import com.digiwin.athena.semc.common.enums.ApplicationTypeEnum;
import com.digiwin.athena.semc.dto.portal.QueryGroupReq;
import com.digiwin.athena.semc.entity.portal.LabelSystemData;
import com.digiwin.athena.semc.service.portal.LabelSystemBusinessTodoService;
import com.digiwin.athena.semc.service.portal.LabelSystemDataService;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;
import com.digiwin.athena.semc.util.Utils;
import com.digiwin.athena.semc.vo.portal.GroupRuleResp;
import com.digiwin.athena.semc.vo.portal.GroupRuleValueResp;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.time.LocalDateTime;
import java.util.Collections;
import java.util.List;

import javax.annotation.Resource;
import javax.validation.Valid;

import cn.hutool.core.util.ObjectUtil;
import lombok.extern.slf4j.Slf4j;

/**
 * 业务待办
 *
 * @author sungqz
 * @since 2024-07-31
 */
@Slf4j
@RestController
@RequestMapping("/semc/business/todo")
public class LabelSystemBusinessToDoController {

    @Resource
    private LabelSystemBusinessTodoService labelSystemBusinessTodoService;

    @Resource
    private MessageUtils messageUtils;

    @Autowired
    private LabelSystemDataService labelSystemDataService;


    /**
     * 查询雅典娜智驱平台的分组条件或分组条件值
     *
     * @return 返回
     */
    @PostMapping("/queryAgileGroupList")
    public ResponseEntity<BaseResultDTO<List<GroupRuleResp>>> queryAgileGroupList(@RequestBody QueryGroupReq queryGroupReq) {
        try {
            List<GroupRuleResp> groupRuleRespList = labelSystemBusinessTodoService.queryAgileGroupList(queryGroupReq);
            return ResponseEntityWrapperUtil.wrapperOk(groupRuleRespList);
        } catch (Exception e) {
            log.error("query agile group list error. param:{}", queryGroupReq, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/business/todo/queryAgileGroupList, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 查询三方应用的业务待办分组列表
     *
     * @param queryGroupReq 入参
     * @return 返回
     */
    @PostMapping("/queryThirdGroupList")
    public ResponseEntity<BaseResultDTO<List<GroupRuleResp>>> queryThirdGroupList(@RequestBody @Valid QueryGroupReq queryGroupReq) {
        try {
            // 参数校验
            if (ObjectUtil.isNull(queryGroupReq.getDataModel())) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "dataModel is missing");
            }
            if (Constants.DataModelEnum.MODEL_HYBRID_CLOUD.getVal().equals(queryGroupReq.getDataModel())) {
                if (StringUtils.isBlank(queryGroupReq.getMiddleSystemName()) || StringUtils.isBlank(queryGroupReq.getMiddleSystemUid())) {
                    String error = messageUtils.getMessage(I18NKey.MIDDLE_SYSTEM_PARAM_MISSING);
                    return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, error);
                }
            } else if (StringUtils.isBlank(queryGroupReq.getRestUrl())) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "restUrl is missing");
            }

            List<GroupRuleResp> groupRuleRespList = labelSystemBusinessTodoService.queryThirdGroupList(queryGroupReq);
            return ResponseEntityWrapperUtil.wrapperOk(groupRuleRespList);
        } catch (Exception e) {
            log.error("query third group list error, param:{}", queryGroupReq, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/business/todo/queryThirdGroupList, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 查询业务待办数量
     *
     * @return 返回
     */
    @PostMapping("/queryBusinessTodoValue")
    public ResponseEntity<BaseResultDTO<GroupRuleValueResp>> queryBusinessTodoValue(@RequestBody QueryGroupReq queryGroupReq) {
        try {
            if (ObjectUtil.isNull(queryGroupReq.getSystemId())) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "param is missing");
            }

            // 根据组件id查询数据源信息
            List<LabelSystemData> systemDataList = labelSystemDataService
                    .querySystemDataBySystemId(Constants.LabelTypeEnum.SYSTEM_CUSTOM.getVal(), Collections.singletonList(queryGroupReq.getSystemId()), Utils.getTenantId());
            if (CollectionUtils.isEmpty(systemDataList)) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "the data source is not exist, systemId:" + queryGroupReq.getSystemId());
            }
            LabelSystemData systemData = systemDataList.get(0);
            log.info("queryBusinessTodoValue | query label system data result:{}", systemData);

            // 三方应用
            if (!ApplicationTypeEnum.VIRTUAL_APPLICATION.getType().equals(systemData.getDataType())) {
                // 入参校验
                if (StringUtils.isBlank(systemData.getCountCode()) && StringUtils.isBlank(systemData.getCountContentCode())) {
                    return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "countCode and countContentCode is missing");
                }
                if (Constants.DataModelEnum.MODEL_HYBRID_CLOUD.getVal().equals(systemData.getDataModel())) {
                    if (StringUtils.isBlank(systemData.getMiddleSystemName()) || StringUtils.isBlank(systemData.getMiddleSystemUid())) {
                        String error = messageUtils.getMessage(I18NKey.MIDDLE_SYSTEM_PARAM_MISSING);
                        return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, error);
                    }
                } else if (StringUtils.isBlank(systemData.getRestUrl())) {
                    return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "systemData restUrl is missing");
                }
                // 查询三方应用待办数
                GroupRuleValueResp groupRuleValueResp = labelSystemBusinessTodoService.queryThirdGroupValueList(systemData);
                return ResponseEntityWrapperUtil.wrapperOk(groupRuleValueResp);
            } else {
                // 查询小AI分组待办数
                List<GroupRuleValueResp> groupRuleRespList = labelSystemBusinessTodoService.queryAgileGroupValueList(queryGroupReq.getClientId(), systemData);
                return ResponseEntityWrapperUtil.wrapperOk(CollectionUtils.isEmpty(groupRuleRespList) ? new GroupRuleValueResp() : groupRuleRespList.get(0));
            }
        } catch (Exception e) {
            log.error("query business todo value error. param:{}", queryGroupReq, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/business/todo/queryBusinessTodoValue, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }
}