package com.digiwin.athena.semc.controller.portal;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.appcore.util.ResponseEntityWrapper;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.I18NKey;
import com.digiwin.athena.semc.dto.fineReport.QueryFineReportUrlReq;
import com.digiwin.athena.semc.dto.portal.LabelSystemCountDataDTO;
import com.digiwin.athena.semc.dto.portal.QueryCountLinkReq;
import com.digiwin.athena.semc.entity.applink.AppLinkDTO;
import com.digiwin.athena.semc.entity.portal.LabelSystemData;
import com.digiwin.athena.semc.entity.portal.LabelSystemPre;
import com.digiwin.athena.semc.service.applink.AppLinkService;
import com.digiwin.athena.semc.service.portal.LabelSystemClickService;
import com.digiwin.athena.semc.service.portal.LabelSystemDataService;
import com.digiwin.athena.semc.service.portal.LabelSystemPreService;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;
import com.digiwin.athena.semc.util.Utils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.validation.Valid;
import java.time.LocalDateTime;
import java.util.Collections;
import java.util.List;

/**
 * 消息发送渠道表(TMessageChannel)表控制层
 *
 * @author sungqz
 * @since 2023-11-30
 */
@Slf4j
@RestController
@RequestMapping("/semc/layout/component")
public class LabelSystemClickController {
    /**
     * 服务对象
     */
    @Resource
    private LabelSystemDataService labelSystemDataService;

    @Resource
    private LabelSystemClickService labelSystemClickService;

    @Resource
    private LabelSystemPreService labelSystemPreService;

    @Autowired
    private AppLinkService appLinkService;

    @Resource
    private MessageUtils messageUtils;

    /**
     * 点击自定义组件查询应用或作业链接
     *
     * @param systemId 自定义组件id
     * @return 查询结果
     */
    @GetMapping("/custom/click")
    public ResponseEntity<BaseResultDTO<AppLinkDTO>> queryCustomAppLink(@RequestParam Long systemId) {
        try {
            AppLinkDTO appLinkDTO = labelSystemClickService.queryCustomJumpLink(systemId);
            //权限判断，无权限提示
            if (null != appLinkDTO && !appLinkDTO.getIsHavePerm()) {
                String error = messageUtils.getMessage(I18NKey.MOBILE_WORK_NOT_AUTH);
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, error);
            }
            return ResponseEntityWrapperUtil.wrapperOk(appLinkDTO);
        } catch (Exception e) {
            log.error("query custom application link error. systemId:{}", systemId, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/layout/component/custom/click");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 查询统计分类跳转链接
     *
     * @param queryCountLinkReq 请求入参
     * @return 返回
     */
    @PostMapping("/queryCountLink")
    public ResponseEntity<?> queryCountLink(@RequestBody @Valid QueryCountLinkReq queryCountLinkReq) {
        // 根据自定义组件id查询配置的数据源
        List<LabelSystemData> systemDataList = labelSystemDataService.querySystemDataBySystemId(Constants.LabelTypeEnum.SYSTEM_CUSTOM.getVal(), Collections.singletonList(queryCountLinkReq.getSystemId()), Utils.getTenantId());
        if (CollectionUtils.isEmpty(systemDataList)) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "the data source is not exist, param:" + queryCountLinkReq);
        }

        // 统计类型参数
        LabelSystemData systemData = systemDataList.get(0);
        String countContentInfo = systemData.getCountContentCode();
        if (StringUtils.isBlank(countContentInfo)) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "the count content info is null, param:" + queryCountLinkReq);
        }
        List<LabelSystemCountDataDTO> countDataDTOList = JSON.parseArray(countContentInfo, LabelSystemCountDataDTO.class);
        LabelSystemCountDataDTO labelSystemCountDataDTO = countDataDTOList.stream().filter(x -> queryCountLinkReq.getContentCode().equals(x.getContentCode())).findFirst().orElse(null);
        if (ObjectUtils.isEmpty(labelSystemCountDataDTO)) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "the count content info is null, contnetCode:" + queryCountLinkReq.getContentCode());
        }

        // 构建作业链接及SSO必要字段
        AppLinkDTO appLinkDTO;
        // 如果配置了作业，则优先组装配置的作业链接
        LabelSystemCountDataDTO.JobInfo jobInfo = labelSystemCountDataDTO.getJobInfo();
        if (ObjectUtils.isNotEmpty(jobInfo)
                && ObjectUtils.isNotEmpty(jobInfo.getAppPrimaryId())
                && ObjectUtils.isNotEmpty(jobInfo.getDataType())
                && ((Constants.IsNeedJobEnum.NEED_CUSTOM_JOB.getValue().equals(jobInfo.getIsJob())
                && StringUtils.isNotBlank(jobInfo.getCallBackUrl())) || (Constants.IsNeedJobEnum.NOT_NEED_JOB.getValue().equals(jobInfo.getIsJob())
                && StringUtils.isNotBlank(jobInfo.getWorkCode())))) {
            appLinkDTO = labelSystemClickService.queryCountLink(labelSystemCountDataDTO);
        } else { // 否则取三方应用传的链接
            appLinkDTO = appLinkService.queryApplicationByDataType(systemData.getDataType(), systemData.getAppCode(), AppAuthContextHolder.getContext().getAuthoredUser());
            if (ObjectUtils.isEmpty(appLinkDTO)) {
                return ResponseEntityWrapper.wrapperOk(appLinkDTO);
            }
            String jumpUrl = appLinkService.parseJumpUrl(labelSystemCountDataDTO.getContentLink(), appLinkDTO);
            appLinkDTO.setCloudwebsite(jumpUrl);
        }
        return ResponseEntityWrapper.wrapperOk(appLinkDTO);
    }

    /**
     * 查询预设组件状态
     *
     * @param dataType 数据类型（0：待办、1：三方报表、2：新闻公告、3：邮件、4：常用、5：收藏、6：全局资讯、7：绩效、8：作业工作台、9：行事历）
     * @return 查询结果
     */
    @GetMapping("/queryStatus")
    public ResponseEntity<BaseResultDTO<Integer>> queryStatus(@RequestParam Integer dataType) {
        try {
            //三方待办 枚举值为 10
            if (null != dataType && dataType.equals(Constants.DataTypeEnum.TYPE_TODO.getVal())) {
                dataType = Constants.DataTypeEnum.TYPE_THIRD_TODO.getVal();
            }

            LabelSystemPre labelSystemPre = labelSystemPreService.getPreSystemID(dataType);
            if (labelSystemPre == null) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "the component is not exist");
            }
            return ResponseEntityWrapperUtil.wrapperOk(labelSystemPre.getValidStatus());
        } catch (Exception e) {
            log.error("query status error. dataType:{}", dataType, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/layout/component/queryStatus");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 获取帆软报表链接
     * 调用处：
     * 1、小AI全部报表页面点击，deviceType传1，表示pc端触发
     * 2、移动端后端，deviceType传2，表示移动端触发
     *
     * @param queryFineReportUrlReq 请求入参
     * @return 查询结果
     */
    @PostMapping("/queryFineReportUrl")
    public ResponseEntity<?> queryFineReportUrl(@RequestBody @Valid QueryFineReportUrlReq queryFineReportUrlReq) {
        JSONObject resp = labelSystemClickService.queryFineReportUrl(queryFineReportUrlReq);
        return ResponseEntityWrapperUtil.wrapperOk(resp);
    }
}

