package com.digiwin.athena.semc.controller.portal;

import com.alibaba.fastjson.JSON;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.appcore.util.ResponseEntityWrapper;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.ResultPageBean;
import com.digiwin.athena.semc.dto.portal.LabelSystemCustomDeleteReq;
import com.digiwin.athena.semc.dto.portal.LabelSystemCustomQueryReq;
import com.digiwin.athena.semc.dto.portal.LabelSystemCustomSaveReq;
import com.digiwin.athena.semc.entity.bench.JobBenchCustom;
import com.digiwin.athena.semc.entity.portal.LabelSystemCustom;
import com.digiwin.athena.semc.service.bench.JobBenchService;
import com.digiwin.athena.semc.service.menu.AuthService;
import com.digiwin.athena.semc.service.portal.LabelSystemCustomService;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;
import com.digiwin.athena.semc.util.SecurityLogUtil;
import com.digiwin.athena.semc.vo.bench.JobBenchSaveVO;
import com.digiwin.athena.semc.vo.portal.LabelSystemAllVo;
import io.swagger.v3.oas.annotations.Operation;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.validation.Valid;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * @author CR-7
 * create: 2024-01-24 17:04
 * Description: 自定义组件
 */
@RestController
@RequestMapping("/semc/custom/component")
public class LabelSystemCustomController {


    @Resource
    LabelSystemCustomService labelSystemCustomService;

    @Autowired
    JobBenchService jobBenchService;

    @Resource
    private MessageUtils messageUtils;

    @Resource
    private AuthService authService;



    /**
     *  查询自定义组件列表
     * @param labelSystemCustomQueryReq
     * @return
     */
    @Operation(method = "pageQuery",description = "查询自定义组件列表")
    @PostMapping("/pageQuery")
    public ResultPageBean pageQuery(@RequestBody @Valid LabelSystemCustomQueryReq labelSystemCustomQueryReq){
        return labelSystemCustomService.pageQuery(labelSystemCustomQueryReq);
    }

    /**
     *  自定义部件、通用部件查询（不分页） -- 门户使用
     * @param map
     * @return
     */
    @Operation(method = "query",description = "查询自定义组件不分页")
    @PostMapping("/query")
    public ResponseEntity<?> query(@RequestBody Map map){
        if (Objects.isNull(MapUtils.getInteger(map,"clientType"))){
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SAVE_LABEL_SYSTEM_CUSTOM, "param clientType is can not null");
        }
        return labelSystemCustomService.query(MapUtils.getInteger(map,"clientType"),MapUtils.getInteger(map,"isLayout",null));
    }




    /**
     *  保存自定义组件
     * @param labelSystemCustomSaveReq
     * @return
     */
    @Operation(method = "save",description = "保存自定义组件")
    @PostMapping("/save")
    public ResponseEntity<?> save(@RequestBody @Valid LabelSystemCustomSaveReq labelSystemCustomSaveReq, HttpServletRequest request){
        //门户管理/web端配置/自定义组件
        //暂时不阻断，只是写入安全日志
        if (!authService.hasAuth("custom-component")) {
            //写入安全日志
            SecurityLogUtil.writeSecurityLog(request, JSON.toJSONString(labelSystemCustomSaveReq));
        }
        return labelSystemCustomService.save(labelSystemCustomSaveReq);
    }

    /**
     *  自定义组件状态变更
     * @param labelSystemCustomSaveReq
     * @return
     */
    @Operation(method = "updateState",description = "更新自定义组件状态")
    @PostMapping("/updateState")
    public ResponseEntity<?> updateState(@RequestBody @Valid LabelSystemCustomSaveReq labelSystemCustomSaveReq){
        return labelSystemCustomService.updateState(labelSystemCustomSaveReq);
    }



    /**
     *  删除自定义组件
     * @param labelSystemCustomDeleteReq
     * @return
     */
    @Operation(method = "delete",description = "删除自定义组件")
    @PostMapping("/delete")
    public ResponseEntity<?> delete(@RequestBody LabelSystemCustomDeleteReq labelSystemCustomDeleteReq) {
        if (CollectionUtils.isEmpty(labelSystemCustomDeleteReq.getIdList())) {
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SAVE_LABEL_SYSTEM_CUSTOM, "id cannot be empty");
        }
        if (Objects.isNull(labelSystemCustomDeleteReq.getClientType())){
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SAVE_LABEL_SYSTEM_CUSTOM, "clientType cannot be null");
        }
        //查询是否有作业关联
        JobBenchSaveVO req = new JobBenchSaveVO();
        req.setSystemIdList(labelSystemCustomDeleteReq.getIdList());
        List<JobBenchCustom> jobBenchCustomList = jobBenchService.selectBySystem(req);
        if (CollectionUtils.isNotEmpty(jobBenchCustomList)) {
            String error = messageUtils.getMessage("error.message.job.system.data.del.error");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
        return labelSystemCustomService.delete(labelSystemCustomDeleteReq.getIdList(),labelSystemCustomDeleteReq.getClientType());
    }

    /**
     *  查询自定义组件部分也
     * @param req
     * @return
     */
    @Operation(method = "queryCustomSystem",description = "查询预设组件和自定义组件列表")
    @PostMapping("/queryCustomSystem")
    public ResponseEntity<BaseResultDTO<List<LabelSystemCustom>>> queryCustomSystem(@RequestBody LabelSystemAllVo req){
        return ResponseEntityWrapperUtil.wrapperOk(labelSystemCustomService.queryCustomSystem(req));
    }

}
