package com.digiwin.athena.semc.controller.portal;

import com.alibaba.excel.EasyExcel;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.I18NKey;
import com.digiwin.athena.semc.common.PageInfoResp;
import com.digiwin.athena.semc.dto.portal.QueryImportRecordReq;
import com.digiwin.athena.semc.entity.portal.LabelSystemCustomImport;
import com.digiwin.athena.semc.entity.portal.LabelSystemDataImport;
import com.digiwin.athena.semc.service.portal.LabelSystemCustomService;
import com.digiwin.athena.semc.service.portal.impl.SystemCustomWriteHandler;
import com.digiwin.athena.semc.util.DateUtils;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;
import com.digiwin.athena.semc.util.Utils;
import com.digiwin.athena.semc.vo.portal.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;
import javax.validation.Valid;
import java.nio.charset.StandardCharsets;
import java.time.LocalDate;
import java.util.List;

/**
 * @description: 自定义组件导入Controller
 */
@Slf4j
@RestController
@RequestMapping("/semc/custom/component")
public class LabelSystemCustomImportController {

    @Autowired
    private LabelSystemCustomService labelSystemCustomService;


    @Resource
    private MessageUtils messageUtils;


    /**
     * 导入自定义组件
     *
     * @return
     */
    @PostMapping("/importReport")
    public ResponseEntity<BaseResultDTO<ImportExcelResp>> importReport(@RequestBody LabelSystemDataImport req) {
        // 校验文件id
        if (StringUtils.isEmpty(req.getFileId())) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "file code cannot be empty");
        }
        ImportExcelResp excelResp = new ImportExcelResp();
        try {
            // 读取报表内容
            List<SystemCustomImportVo> excelVoList = labelSystemCustomService.readExcel(req.getFileId());
            if (CollectionUtils.isEmpty(excelVoList)) {
                String error = messageUtils.getMessage("error.message.import.cannot.be.empty");
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, error);
            }
            // 去掉前后空格
            excelVoList = trimDataList(excelVoList);
            excelResp = labelSystemCustomService.importReport(excelVoList);
            return ResponseEntityWrapperUtil.wrapperOk(excelResp);
        } catch (Exception e) {
            log.error("LabelSystemCustomImportController import excel exception", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 去掉空格
     *
     * @param excelVoList
     * @return
     */
    public List<SystemCustomImportVo> trimDataList(List<SystemCustomImportVo> excelVoList) {
        for (SystemCustomImportVo excelVo : excelVoList) {
            if (StringUtils.isNotEmpty(excelVo.getNameZh())) {
                excelVo.setNameZh(excelVo.getNameZh().trim());
            }
            if (StringUtils.isNotEmpty(excelVo.getDataCategory())) {
                excelVo.setDataCategory(excelVo.getDataCategory().trim());
            }
            if (StringUtils.isNotEmpty(excelVo.getValidStatus())) {
                excelVo.setValidStatus(excelVo.getValidStatus().trim());
            }
            if (StringUtils.isNotEmpty(excelVo.getRemark())) {
                excelVo.setRemark(excelVo.getRemark().trim());
            }
            if (StringUtils.isNotEmpty(excelVo.getDataName())) {
                excelVo.setDataName(excelVo.getDataName().trim());
            }
        }
        return excelVoList;
    }

    /**
     * 查询导入记录
     *
     * @param queryImportRecordReq 入参
     * @return
     */
    @PostMapping("/queryImportRecord")
    public ResponseEntity<BaseResultDTO<PageInfoResp<LabelSystemCustomImport>>> queryImportRecord(@RequestBody @Valid QueryImportRecordReq queryImportRecordReq) {
        try {
            // 校验时间格式
            if ((StringUtils.isNotBlank(queryImportRecordReq.getStartTime()) && !DateUtils.validateDate(queryImportRecordReq.getStartTime(), DateUtils.DATE_TIME_NORMAL_FORMATTER)) || (StringUtils.isNotBlank(queryImportRecordReq.getEndTime()) && !DateUtils.validateDate(queryImportRecordReq.getEndTime(), DateUtils.DATE_TIME_NORMAL_FORMATTER))) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.COMMON_PARAM_ILLEGAL));
            }
            PageInfoResp<LabelSystemCustomImport> pageInfoResp = labelSystemCustomService.queryImportRecord(queryImportRecordReq);
            return ResponseEntityWrapperUtil.wrapperOk(pageInfoResp);
        } catch (Exception e) {
            log.error("LabelSystemCustomImportController queryImportRecord exception", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 导出自定义组件
     *
     * @return
     */
    @GetMapping("/exportReport")
    public void exportReport(HttpServletResponse response) {
        try {
            List<SystemCustomExportVo> excelVoList = labelSystemCustomService.exportReport();
            // 设置文件名
            String filename = "自定义组件_" + LocalDate.now().toString().replace("-", "") + ".xlsx";
            // 设置下载信息
            response.setContentType("application/vnd.ms-excel");
            response.setCharacterEncoding("utf-8");
            response.setHeader("Content-disposition", "attachment;filename=" + new String(filename.getBytes(StandardCharsets.UTF_8), StandardCharsets.ISO_8859_1));
            EasyExcel.write(response.getOutputStream(), SystemCustomExportVo.class)
                    .registerWriteHandler(Utils.initHorizontalCell())
                    .sheet("自定义组件").relativeHeadRowIndex(1).registerWriteHandler(new SystemCustomWriteHandler()).doWrite(excelVoList);
        } catch (Exception e) {
            log.error("LabelSystemCustomImportController exportReport exception", e);
            throw new RuntimeException(e);
        }
    }

}