package com.digiwin.athena.semc.controller.portal;

import com.alibaba.fastjson.JSON;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.appcore.util.ResponseEntityWrapper;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.I18NKey;
import com.digiwin.athena.semc.common.PageInfoResp;
import com.digiwin.athena.semc.common.enums.ApplicationTypeEnum;
import com.digiwin.athena.semc.common.enums.VirtualApplicationEnum;
import com.digiwin.athena.semc.dto.erpsso.QueryAppListReq;
import com.digiwin.athena.semc.dto.portal.QueryGroupReq;
import com.digiwin.athena.semc.dto.portal.QueryLabelSystemReq;
import com.digiwin.athena.semc.dto.portal.ValidateLabelReq;
import com.digiwin.athena.semc.entity.applink.AppLinkDTO;
import com.digiwin.athena.semc.entity.menu.manage.ManageMenu;
import com.digiwin.athena.semc.entity.portal.LabelSystemData;
import com.digiwin.athena.semc.entity.temp.TemplateSystemData;
import com.digiwin.athena.semc.env.EnvProperties;
import com.digiwin.athena.semc.proxy.km.service.KmService;
import com.digiwin.athena.semc.proxy.km.service.model.AppJobDTO;
import com.digiwin.athena.semc.service.applink.AppLinkService;
import com.digiwin.athena.semc.service.bench.SyncJobInfoService;
import com.digiwin.athena.semc.service.menu.AuthService;
import com.digiwin.athena.semc.service.menu.ManageMenuService;
import com.digiwin.athena.semc.service.portal.LabelSystemBusinessTodoService;
import com.digiwin.athena.semc.service.portal.LabelSystemDataService;
import com.digiwin.athena.semc.service.portal.LabelSystemSourceService;
import com.digiwin.athena.semc.service.portal.TodoAppSortService;
import com.digiwin.athena.semc.service.temp.TemplateSystemDataService;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;
import com.digiwin.athena.semc.util.SecurityLogUtil;
import com.digiwin.athena.semc.util.Utils;
import com.digiwin.athena.semc.vo.portal.GroupRuleResp;
import io.github.linpeilie.Converter;
import io.swagger.v3.oas.annotations.Operation;
import net.sf.json.JSONObject;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.validation.Valid;
import java.text.Collator;
import java.time.LocalDateTime;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 数据源管理
 *
 * @author: dengkaiwen
 */
@RestController
@RequestMapping("/semc/label/system/data/")
public class LabelSystemDataController {

    private static final Logger logger = LoggerFactory.getLogger(LabelSystemDataController.class);

    @Autowired
    LabelSystemDataService labelSystemDataService;

    @Resource
    private MessageUtils messageUtils;

    @Autowired
    private AppLinkService appLinkService;

    @Autowired
    private LabelSystemSourceService labelSystemSourceService;

    @Autowired
    private LabelSystemBusinessTodoService labelSystemBusinessTodoService;

    @Autowired
    private SyncJobInfoService syncJobInfoService;

    @Autowired
    private KmService kmService;


    @Autowired
    private TemplateSystemDataService templateSystemDataService;

    @Autowired
    private AuthService authService;

    @Autowired
    private TodoAppSortService todoAppSortService;


    @Resource
    private Converter converter;
    /**
     * 异构消息页面下拉应用列表
     */
    private static final String FILTER_MESSAGE_TYPE = "message";


    /**
     * 管理后台，查询所有b/s应用、c/s应用、预设应用
     *
     * @return
     */
    @PostMapping("/queryAppList")
    public ResponseEntity<BaseResultDTO<List<AppLinkDTO>>> queryAppList(@RequestBody QueryAppListReq queryAppListReq) {
        try {
            List<AppLinkDTO> appLinkListDTO = queryAppLinkList(queryAppListReq);
            return ResponseEntityWrapperUtil.wrapperOk(appLinkListDTO);
        } catch (Exception e) {
            logger.error("queryAppList exceptiona", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/label/system/data/queryAppList");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 查询所有应用列表
     *
     * @param queryAppListReq 入参
     * @return 返回
     */
    private List<AppLinkDTO> queryAppLinkList(QueryAppListReq queryAppListReq) {
        AuthoredUser user = AppAuthContextHolder.getContext().getAuthoredUser();
        //以预设表里的为准，删除重复的
        List<AppLinkDTO> appLinkListDTO = appLinkService.queryManageList(user);
        appLinkListDTO = filterRepeatApp(appLinkListDTO);
        //过滤"鼎捷云控制台", "开发平台"
        //appLinkListDTO=appLinkListDTO.stream().filter(x -> !Constants.FILTER_SYSETEM_DATA.contains(x.getName())).collect(Collectors.toList());
        // 过滤掉IAM应用
        if (queryAppListReq != null && FILTER_MESSAGE_TYPE.equals(queryAppListReq.getFilterType())) {
            appLinkListDTO = appLinkListDTO.stream().filter(x -> !ApplicationTypeEnum.IAM_APPLICATION.getType().equals(x.getDataSource())
                    && StringUtils.isNotBlank(x.getApplicationAppId())).collect(Collectors.toList());
        }
        return appLinkListDTO;
    }

    /**
     * 查询所有应用列表，用户配置新增数据源页面的应用选择
     *
     * @param queryAppListReq 入参
     * @return 返回
     */
    @PostMapping("/queryAppListForDataSource")
    public ResponseEntity<BaseResultDTO<List<AppLinkDTO>>> queryAppListForDataSource(@RequestBody QueryAppListReq queryAppListReq) {
        try {
            // 查询所有应用列表
            List<AppLinkDTO> appLinkListDTO = queryAppLinkList(queryAppListReq);

            // 构建"雅典娜智驱平台"，作为数据源"业务待办"类型的源应用
            AppLinkDTO appLinkDTO =Utils.initAppInteraction();
            appLinkListDTO.add(appLinkDTO);
           //是否需要加入"鼎捷雅典娜APP"，需要
            if (null !=queryAppListReq.getIsNeedApp() && queryAppListReq.getIsNeedApp().intValue() ==1) {
                // 构建"鼎捷雅典娜APP"，作为数据源的源应用
                AppLinkDTO athenaAppLinkDTO = Utils.initAthenaApp();
                appLinkListDTO.add(athenaAppLinkDTO);
            }
            AppLinkDTO fineReportApp = Utils.initFineReportApp();
            appLinkListDTO.add(fineReportApp);
            return ResponseEntityWrapperUtil.wrapperOk(appLinkListDTO);
        } catch (Exception e) {
            logger.error("query app list for data source. param:{}", queryAppListReq, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/label/system/data/queryAppListForDataSource");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 统计组件-查询统计类型配置作业的所属应用列表
     *
     * @return 返回
     */
    @PostMapping("/queryAppListForCountJob")
    public ResponseEntity<BaseResultDTO<List<AppLinkDTO>>> queryAppListForDataSource() {
        // 查询所有应用列表
        List<AppLinkDTO> appLinkListDTO = queryAppLinkList(null);
        return ResponseEntityWrapperUtil.wrapperOk(appLinkListDTO);
    }

    /**
     * 查询已对接作业的源应用，涉及b/s应用、c/s应用、预设应用
     *
     * @return
     */
    @PostMapping("/queryJobAppList")
    public ResponseEntity<BaseResultDTO<List<AppLinkDTO>>> queryJobAppList(@RequestBody QueryAppListReq queryAppListReq) {
        try {
            AuthoredUser user = AppAuthContextHolder.getContext().getAuthoredUser();
            //如果预设里有的 以预设表里的为准，删除重复的
            List<AppLinkDTO> appLinkListDTO = appLinkService.queryManageList(user);
            appLinkListDTO = filterRepeatApp(appLinkListDTO);
            //去掉没有集成作业的应用
            appLinkListDTO = filterJobApp(appLinkListDTO);
            Comparator comparator = Collator.getInstance(Locale.CHINA);
            Collections.sort(appLinkListDTO, (o1, o2) -> comparator.compare(o1.getName(), o2.getName()));
            return ResponseEntityWrapperUtil.wrapperOk(appLinkListDTO);
        } catch (Exception e) {
            logger.error("queryAppList exceptiona", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/label/system/data/queryAppList");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 去掉没有集成作业的应用
     *
     * @param appLinkListDT
     * @return
     */
    public List<AppLinkDTO> filterJobApp(List<AppLinkDTO> appLinkListDT) {
        List<AppLinkDTO> appLinkDTOList = new ArrayList<>();
        if (CollectionUtils.isEmpty(appLinkListDT)) {
            return appLinkDTOList;
        }
        List<String> ssoAppIdList = appLinkListDT.stream().map(AppLinkDTO::getApplicationAppId).collect(Collectors.toList());
        //查询已集成作业的应用appid
        List<String> appIdList = syncJobInfoService.getJobAppIdList(ssoAppIdList);
        //从km查询租户下所有应用作业信息
        List<AppJobDTO> appJobListDTOList = kmService.queryAppJobList();
        Map<String, AppJobDTO> appJobListMap = new HashMap<>();
        if (CollectionUtils.isNotEmpty(appJobListDTOList)) {
            appJobListMap = appJobListDTOList.stream().collect(Collectors.toMap(AppJobDTO::getAppCode, Function.identity(), (a, b) -> a));
        }
        for (AppLinkDTO appLink : appLinkListDT) {
            //非IAM 应用 需要看是否已集成作业
            if (!appLink.getDataSource().equals(ApplicationTypeEnum.IAM_APPLICATION.getType())) {
                if (appIdList.contains(appLink.getApplicationAppId())) {
                    appLinkDTOList.add(appLink);
                }
            }
            //IAM 应用,看KM是否返回作业
            if (appLink.getDataSource().equals((ApplicationTypeEnum.IAM_APPLICATION.getType()))) {
                //KM应用没有对应的作业
                if (null == appJobListMap || null == appJobListMap.get(appLink.getPrimaryId())) {
                    continue;
                }
                AppJobDTO appJobDTO = appJobListMap.get(appLink.getPrimaryId());
                //取出当前应用的作业
                if (CollectionUtils.isNotEmpty(appJobDTO.getData())) {
                    appLinkDTOList.add(appLink);
                }
            }
        }
        return appLinkDTOList;
    }

    /**
     * 如果预设里有的 以预设表里的为准，删除重复的
     *
     * @param appLinkListDT
     * @return
     */
    public List<AppLinkDTO> filterRepeatApp(List<AppLinkDTO> appLinkListDT) {
        List<AppLinkDTO> appLinkDTOList = new ArrayList<>();
        //预设表里的数据
        List<AppLinkDTO> matchedLinkList = appLinkListDT.stream().filter(x -> x.getDataSource().equals(ApplicationTypeEnum.PRESET_APPLICATION.getType())).collect(Collectors.toList());
        Map<String, AppLinkDTO> preListMap = matchedLinkList.stream().collect(Collectors.toMap(AppLinkDTO::getAppId, Function.identity(), (a, b) -> a));
        //IAM 数据
        List<AppLinkDTO> iamLinkList = appLinkListDT.stream().filter(x -> x.getDataSource().equals(ApplicationTypeEnum.IAM_APPLICATION.getType())).collect(Collectors.toList());
        Map<String, AppLinkDTO> iamListMap = iamLinkList.stream().collect(Collectors.toMap(AppLinkDTO::getPrimaryId, Function.identity(), (a, b) -> a));
        for (AppLinkDTO appLink : appLinkListDT) {
            //预设表数据，且预设表的appid 和 cs、bs、iam返回的应用code 一致，则需要过滤改应用
            AppLinkDTO iamData = iamListMap.get(appLink.getApplicationAppId());
            if (appLink.getDataSource().equals(ApplicationTypeEnum.PRESET_APPLICATION.getType()) && null != iamData) {
                continue;
            }
            AppLinkDTO preData = preListMap.get(appLink.getPrimaryId());
            //非预设表数据，且预设表的appid 和 cs、bs、iam返回的应用code 一致，则应用名称取预设表的名称
            if (!appLink.getDataSource().equals(ApplicationTypeEnum.PRESET_APPLICATION.getType()) && null != preData) {
                appLink.setName(preData.getName());
            }
            appLinkDTOList.add(appLink);
        }
        return appLinkDTOList;
    }

    /**
     * 查询数据源
     *
     * @param req 请求实体
     * @return
     * @throws Exception
     */
    @Operation(summary = "查询数据源")
    @PostMapping("/queryDataList")
    public ResponseEntity<BaseResultDTO<PageInfoResp<LabelSystemData>>> queryDataList(@RequestBody QueryLabelSystemReq req) {
        try {
            if (null == req.getPageNum() || null == req.getPageSize()) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "missing paging parameter");
            }
            return ResponseEntityWrapperUtil.wrapperOk(labelSystemDataService.queryDataList(req));
        } catch (Exception e) {
            logger.error("queryDataList exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/label/system/data/queryDataList");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }


    /**
     * 启用停用数据源
     *
     * @param req 请求实体
     * @return
     * @throws Exception
     */
    @Operation(summary = "启用停用")
    @PostMapping("/updateValidStatus")
    public ResponseEntity<?> updateValidStatus(@RequestBody QueryLabelSystemReq req) {
        try {
            if (null == req.getId()) {
                return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "id cannot be empty");
            }
            if (null == req.getValidStatus()) {
                return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "the status cannot be empty");
            }
            return labelSystemDataService.updateValidStatus(req);
        } catch (Exception e) {
            logger.error("updateValidStatus exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/label/system/data/updateValidStatus");
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }


    /**
     * 批量删除数据源
     *
     * @param req 请求实体
     * @return
     * @throws Exception
     */
    @Operation(summary = "批量删除数据源")
    @PostMapping("/delData")
    public ResponseEntity<?> delData(@RequestBody QueryLabelSystemReq req) {
        try {
            if (CollectionUtils.isEmpty(req.getIds())) {
                return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "id cannot be empty");
            }
            return labelSystemDataService.delData(req);
        } catch (Exception e) {
            logger.error("delData exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/label/system/data/delData");
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }


    /**
     * 校验业务待办类型的数据源关联的分组是否存在
     *
     * @param validateLabelReq 请求参数
     * @return 返回
     */
    @Operation(summary = "校验分组是否存在")
    @PostMapping("/validateGroupExist")
    public ResponseEntity<BaseResultDTO<JSONObject>> validateGroupExist(@RequestBody ValidateLabelReq validateLabelReq) {
        try {
            JSONObject jsonObject = new JSONObject();
            // 参数校验
            if (ObjectUtils.isEmpty(validateLabelReq.getSystemId()) && ObjectUtils.isEmpty(validateLabelReq.getSourceId())) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "param is missing");
            }
            LabelSystemData systemData = new LabelSystemData();
            //请求来源，0：非ISV 数据源；1：ISV数据源
            if(validateLabelReq.getSourceType() == 0){
                // 数据源页面
                if (ObjectUtils.isNotEmpty(validateLabelReq.getSourceId())) {
                    systemData = labelSystemDataService.getBaseMapper().selectById(validateLabelReq.getSourceId());
                } else if (ObjectUtils.isNotEmpty(validateLabelReq.getSystemId())) { // 组件页面
                    List<Long> dataIdList = labelSystemSourceService.selectBySystem(validateLabelReq.getSystemId(), Constants.LabelTypeEnum.SYSTEM_CUSTOM.getVal());
                    if (CollectionUtils.isEmpty(dataIdList)) {
                        return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "the custom system rel data source is not exist, systemId:" + validateLabelReq.getSystemId());
                    }
                    List<LabelSystemData> systemDataList = labelSystemDataService.queryByIds(dataIdList);
                    if (CollectionUtils.isEmpty(systemDataList)) {
                        return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "the data source is not exist, sourceId:" + dataIdList.get(0) + ", systemId:" + validateLabelReq.getSystemId());
                    }
                    systemData = systemDataList.get(0);
                }
            } if(validateLabelReq.getSourceType() == 1){
                TemplateSystemData systemDataTemp=new TemplateSystemData();
                // 数据源页面
                if (ObjectUtils.isNotEmpty(validateLabelReq.getSourceId())) {
                    systemDataTemp = templateSystemDataService.getBaseMapper().selectById(validateLabelReq.getSourceId());
                } else if (ObjectUtils.isNotEmpty(validateLabelReq.getSystemId())) { // 组件页面
                    List<Long> dataIdList = templateSystemDataService.selectBySystemId(validateLabelReq.getSystemId());
                    if (CollectionUtils.isEmpty(dataIdList)) {
                        return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "the custom system rel data source is not exist, systemId:" + validateLabelReq.getSystemId());
                    }
                    List<TemplateSystemData> systemDataTempList = templateSystemDataService.queryByIds(dataIdList);
                    if (CollectionUtils.isEmpty(systemDataTempList)) {
                        return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "the data source is not exist, sourceId:" + dataIdList.get(0) + ", systemId:" + validateLabelReq.getSystemId());
                    }
                    systemDataTemp=systemDataTempList.get(0);
                }
                systemData=converter.convert(systemDataTemp, LabelSystemData.class);
            }
            if (null == systemData) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "system data is not exist. param:" + JSON.toJSONString(validateLabelReq));
            }

            String countCode = systemData.getCountCode();
            String countContentCode = systemData.getCountContentCode();
            // 三方应用
            if (!ApplicationTypeEnum.VIRTUAL_APPLICATION.getType().equals(systemData.getDataType())) {
                // 入参校验
                if (StringUtils.isBlank(systemData.getCountCode()) && StringUtils.isBlank(systemData.getCountContentCode())) {
                    return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "countCode and countContentCode is missing");
                }
                if (Constants.DataModelEnum.MODEL_HYBRID_CLOUD.getVal().equals(systemData.getDataModel())) {
                    if (StringUtils.isBlank(systemData.getMiddleSystemName()) || StringUtils.isBlank(systemData.getMiddleSystemUid())) {
                        return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "systemData is missing, middleSystemName:"
                                + systemData.getMiddleSystemName() + ", middleSystemUid:" + systemData.getMiddleSystemUid());
                    }
                } else if (StringUtils.isBlank(systemData.getRestUrl())) {
                    return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "systemData restUrl is missing");
                }

                QueryGroupReq queryGroupReq = new QueryGroupReq();
                queryGroupReq.setDataModel(systemData.getDataModel());
                queryGroupReq.setMiddleSystemName(systemData.getMiddleSystemName());
                queryGroupReq.setMiddleSystemUid(systemData.getMiddleSystemUid());
                queryGroupReq.setRestUrl(systemData.getRestUrl());
                List<GroupRuleResp> groupRuleRespList = labelSystemBusinessTodoService.queryThirdGroupList(queryGroupReq);
                boolean flag = Boolean.FALSE;
                if (StringUtils.isBlank(systemData.getCountContentCode())) {
                    List<String> codeList = groupRuleRespList.stream().map(GroupRuleResp::getCode).collect(Collectors.toList());
                    flag = codeList.stream().anyMatch(x -> x.equals(countCode));
                } else {
                    for (GroupRuleResp groupRuleResp : groupRuleRespList) {
                        if (groupRuleResp.getCode().equals(systemData.getCountCode())) {
                            List<String> codeList = groupRuleResp.getGroupConditionList().stream().map(GroupRuleResp::getCode).collect(Collectors.toList());
                            flag = codeList.stream().anyMatch(x -> x.equals(countContentCode));
                            break;
                        }
                    }
                }
                jsonObject.put("terminal", 1); // 三方业务待办
                jsonObject.put("flag", flag);
                return ResponseEntityWrapperUtil.wrapperOk(jsonObject);
            } else {
                jsonObject.put("terminal", 0); // 智驱业务待办
                QueryGroupReq queryGroupReq = new QueryGroupReq();
                List<GroupRuleResp> groupRuleRespList = labelSystemBusinessTodoService.queryAgileGroupList(queryGroupReq);
                // 判断当前配置的分组是否存在
                boolean flag = groupRuleRespList.stream().anyMatch(x -> x.getCode().equals(countCode));
                if (!flag) {
                    jsonObject.put("flag", false);
                    return ResponseEntityWrapperUtil.wrapperOk(jsonObject);
                }

                // 如果分组存在，则进一步判断分组条件是否存在
                queryGroupReq.setGroupCode(systemData.getCountCode());
                groupRuleRespList = labelSystemBusinessTodoService.queryAgileGroupList(queryGroupReq);
                flag = groupRuleRespList.stream().anyMatch(x -> x.getCode().equals(countContentCode));

                jsonObject.put("flag", flag);
                return ResponseEntityWrapperUtil.wrapperOk(jsonObject);
            }
        } catch (Exception e) {
            logger.error("validate group exist error. param:{}", validateLabelReq, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/label/system/data/validateGroupExist");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 保存和修改数据源
     *
     * @param req 请求实体
     * @return
     * @throws Exception
     */
    @Operation(summary = "保存和修改数据源")
    @PostMapping("/saveData")
    public ResponseEntity<?> saveData(@RequestBody @Valid LabelSystemData req, HttpServletRequest request) {
        try {
            //门户管理/web端配置/数据源配置
            //暂时不阻断，只是写入安全日志
            if (!authService.hasAuth("datasource-configuration")) {
                //写入安全日志
                SecurityLogUtil.writeSecurityLog(request, JSON.toJSONString(req));
            }
            return labelSystemDataService.saveData(req);
        } catch (Exception e) {
            logger.error("saveData exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/label/system/data/saveData");
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }


    /**
     * 查询组件下数据源的应用系统
     *
     * @param querySystemReq 请求实体
     * @return
     */
    @Operation(summary = "查询组件下数据源的应用系统")
    @PostMapping("/systemList")
    public ResponseEntity<BaseResultDTO<List<LabelSystemData>>> querySystemList(@RequestBody QueryLabelSystemReq querySystemReq) {

        //校验组件类型是否为空
        if (null == querySystemReq.getDataType()) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "component type cannot be empty");
        }
        try {
            //三方待办 枚举值为 10
            if(null !=querySystemReq.getDataType() && querySystemReq.getDataType().equals(Constants.DataTypeEnum.TYPE_TODO.getVal())){
                querySystemReq.setDataType(Constants.DataTypeEnum.TYPE_THIRD_TODO.getVal());
            }
            //校验组件是否存在
            List<Long> dataIds = labelSystemSourceService.selectPreSystem(querySystemReq.getDataType(), Constants.LabelTypeEnum.SYSTEM_PRE.getVal());
            List<LabelSystemData> labelSystemDataList = labelSystemDataService.queryDataInfoBy(dataIds, true);
            if (CollectionUtils.isEmpty(labelSystemDataList)) {
                return ResponseEntityWrapperUtil.wrapperOk(labelSystemDataList);
            }
            if (querySystemReq.getDataType().equals(Constants.DataTypeEnum.TYPE_THIRD_TODO.getVal())) {
                return ResponseEntityWrapperUtil.wrapperOk(todoAppSortService.sortLabelSystemData(labelSystemDataList));
            }
            return ResponseEntityWrapperUtil.wrapperOk(labelSystemDataList);
        } catch (Exception e) {
            logger.error("/semc/label/system/data/systemList exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/label/system/data/systemList");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 应用下作业
     *
     * @param req 请求实体
     * @return
     */
    @Operation(summary = "应用下作业")
    @PostMapping("/queryAppJobList")
    public ResponseEntity<BaseResultDTO<List<AppJobDTO.Job>>> queryAppJobList(@RequestBody LabelSystemData req) {

        //校验应用编码
        if (StringUtils.isEmpty(req.getAppCode())) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "app code cannot be empty");
        }
        //校验数据来源
        if (req.getDataSource() == null) {
            req.setDataSource(ApplicationTypeEnum.IAM_APPLICATION.getType());
        }
        try {
            return ResponseEntityWrapperUtil.wrapperOk(labelSystemDataService.getAppJobList(req.getAppCode(), req.getDataSource(), true , req.getType()));
        } catch (Exception e) {
            logger.error("/semc/label/system/data/queryAppJobList exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/label/system/data/queryAppJobList");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 查询已对接的应用下作业
     *
     * @param req 请求实体
     * @return
     */
    @Operation(summary = "查询已对接的应用下作业")
    @PostMapping("/queryJobPageList")
    public ResponseEntity<BaseResultDTO<PageInfoResp<AppJobDTO.Job>>> queryJobPageList(@RequestBody QueryLabelSystemReq req) {

        //校验应用编码
        if (StringUtils.isEmpty(req.getAppCode())) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "app code cannot be empty");
        }
        if (null == req.getPageNum()) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "pageNum cannot be empty");
        }
        if (null == req.getPageSize()) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "pageSize cannot be empty");
        }
        //校验应用编码
        if (req.getDataSource() == null) {
            req.setDataSource(ApplicationTypeEnum.IAM_APPLICATION.getType());
        }
        try {
            List<AppJobDTO.Job> jobList = labelSystemDataService.getAppJobList(req.getAppCode(), req.getDataSource(), false, null);
            List<AppJobDTO.Job> jobListResp = new ArrayList<>();
            for (AppJobDTO.Job job : jobList) {
                if (StringUtils.isEmpty(req.getJobCodeName())) {
                    jobListResp.add(job);
                    continue;
                }
                if (StringUtils.isNotEmpty(req.getJobCodeName()) && StringUtils.isEmpty(req.getJobCodeName().trim())) {
                    jobListResp.add(job);
                    continue;
                }
                //作业名称或者code 搜索
                if (job.getName().contains(req.getJobCodeName()) || job.getCode().contains(req.getJobCodeName())) {
                    jobListResp.add(job);
                }
            }
            jobListResp.sort(Comparator.nullsLast(Comparator.comparing(AppJobDTO.Job::getName, Comparator.nullsLast(String::compareTo))));
            //分页返回
            List<AppJobDTO.Job> jobPageList = Utils.subList(jobListResp, req.getPageSize(), req.getPageNum());
            PageInfoResp<AppJobDTO.Job> pageInfoResp = new PageInfoResp<>();
            int totalPages = jobListResp.size() / req.getPageSize();
            if (jobListResp.size() % req.getPageSize() != 0) {
                totalPages++;
            }
            pageInfoResp.setTotalPages(totalPages);
            pageInfoResp.setPageNo(req.getPageNum());
            pageInfoResp.setPageSize(req.getPageSize());
            pageInfoResp.setTotalRecords(jobListResp.size());
            pageInfoResp.setList(jobPageList);
            return ResponseEntityWrapperUtil.wrapperOk(pageInfoResp);
        } catch (Exception e) {
            logger.error("/semc/label/system/data/queryJobPageList exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/label/system/data/queryJobPageList");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

}
