package com.digiwin.athena.semc.controller.portal;

import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.appcore.util.ResponseEntityWrapper;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.I18NKey;
import com.digiwin.athena.semc.common.ResultBean;
import com.digiwin.athena.semc.dto.portal.AddLogoReq;
import com.digiwin.athena.semc.dto.portal.LogoReq;
import com.digiwin.athena.semc.entity.common.PlatformConfig;
import com.digiwin.athena.semc.env.EnvProperties;
import com.digiwin.athena.semc.service.portal.ILogoService;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;
import com.digiwin.athena.semc.util.Utils;
import io.swagger.v3.oas.annotations.Operation;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.validation.Valid;
import java.util.Arrays;

/**
 * @description 首页logo图片
 * @author: sungqz
 * @create: 2022-11-10
 */
@RestController
@RequestMapping("/tenant/api/homepage/logo")
public class LogoController {

    private static final Logger logger = LoggerFactory.getLogger(PreinstalledApplicationController.class);

    @Autowired
    ILogoService logoService;

    @Resource
    private MessageUtils messageUtils;

    @Resource
    private EnvProperties envProperties;

    /**
     * 添加首页logo图片
     *
     * @param req 请求实体
     * @return 返回体
     */
    @Operation(summary = "添加首页logo图片")
    @PostMapping("/add")
    public ResultBean addLogo(@RequestBody @Valid AddLogoReq req) {
        ResultBean resultBean = new ResultBean();
        resultBean.setResponse(logoService.addLogoPic(req));
        return resultBean;
    }

    /**
     * 查询首页logo图片地址
     *
     * @return 返回体
     */
    @Operation(summary = "查询logo图片地址")
    @GetMapping("/queryLogo")
    public ResultBean queryLogoByCondition(@RequestParam("sceneFlag") Integer sceneFlag) throws Exception {
        if (sceneFlag == null) {
            sceneFlag = Constants.SceneFlagEnum.LOGIN_AFTER_PAGE.getFlag();
        }
        if (!Arrays.asList(Constants.SceneFlagEnum.LOGIN_BEFORE_PAGE.getFlag(), Constants.SceneFlagEnum.LOGIN_AFTER_PAGE.getFlag()).contains(sceneFlag)) {
            throw new Exception("场景标识不合法:" + sceneFlag);
        }
        ResultBean resultBean = new ResultBean();
        String logoUrl = logoService.queryLogoByCondition(sceneFlag);
        resultBean.setResponse(logoUrl);
        return resultBean;
    }

    /**
     * 查询登录页logo图片地址
     *
     * @return 返回体
     */
    @Operation(summary = "查询登录页logo地址")
    @GetMapping("/login")
    public ResponseEntity<BaseResultDTO<LogoReq>> queryLogoLogin(@RequestParam(value="locale", required=false) String locale,
                                                        @RequestParam(value="channel", required=false) String channel) {
        try {
            return ResponseEntityWrapperUtil.wrapperOk(logoService.queryLoginLogo(locale,channel));
        } catch (Exception e) {
            logger.error("query Logo login occur error", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }


    /**
     * 添加首页logo图片
     *
     * @param req 请求实体
     * @return 返回体
     */
    @Operation(summary = "添加首页logo图片")
    @PostMapping("/saveSett")
    public ResponseEntity<?> saveSett(@RequestBody LogoReq req) {
        ResultBean resultBean = new ResultBean();
        try {
            //校验入参
            ResponseEntity<?> responseEntity=checkSettParam(req);
            if(null !=responseEntity){
                return responseEntity;
            }
            resultBean.setResponse(logoService.saveSett(req));
        } catch (Exception e) {
            logger.error("saveSett error", e);
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
        return ResponseEntityWrapper.wrapperOk();
    }

    /**
     * 校验入参
     * @param req
     * @return
     */
    public ResponseEntity<?> checkSettParam(LogoReq req) throws IllegalAccessException {
        //校验对象是否为空
        if(null == req || Utils.isAllFieldsNull(req)){
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "请输入参数");
        }
        //启用跳转官网，判断官网地址是否必填
        if (null != req.getJumpStatus() && req.getJumpStatus().equals(Constants.JumpStatusEnum.YES.getFlag())) {
            if (StringUtils.isEmpty(req.getWebSiteAddr())) {
                return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "请输入超链内容");
            }

        }
        //繁体 启用跳转官网，判断官网地址是否必填
        if (null != req.getJumpStatusTw() && req.getJumpStatusTw().equals(Constants.JumpStatusEnum.YES.getFlag())) {
            if (StringUtils.isEmpty(req.getWebSiteAddrTw())) {
                return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "请输入超链内容");
            }

        }
        //英文 启用跳转官网，判断官网地址是否必填
        if (null != req.getJumpStatusUs() && req.getJumpStatusUs().equals(Constants.JumpStatusEnum.YES.getFlag())) {
            if (StringUtils.isEmpty(req.getWebSiteAddrUs())) {
                return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "请输入超链内容");
            }

        }
        //网页标题长度最大20字符
        if (StringUtils.isNotEmpty(req.getTitle()) && req.getTitle().length() > 20) {
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_LENGTH_ERROR, "最多输入20个字符");
        }
        //繁体 网页标题长度最大20字符
        if (StringUtils.isNotEmpty(req.getTitleTw()) && req.getTitleTw().length() > 20) {
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_LENGTH_ERROR, "最多输入20个字符");
        }
        //英文  网页标题长度最大20字符
        if (StringUtils.isNotEmpty(req.getTitleUs()) && req.getTitleUs().length() > 20) {
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_LENGTH_ERROR, "最多输入20个字符");
        }
        return null;
    }

    /**
     * 查询企业标识设置
     * @return 返回体
     */
    @Operation(summary = "查询企业标识设置")
    @PostMapping("/getSett")
    public  ResponseEntity<BaseResultDTO<LogoReq>> getSett() {
        return ResponseEntityWrapperUtil.wrapperOk(logoService.getSett());
    }

    /**
     * 查询登录后企业标识信息
     * @return 返回体
     */
    @Operation(summary = "查询登录后企业标识信息")
    @PostMapping("/querySettInfo")
    public  ResponseEntity<BaseResultDTO<LogoReq>> querySettInfo() {
        return ResponseEntityWrapperUtil.wrapperOk(logoService.querySettInfo());
    }

    /**
     * 获取平台配置信息
     * @return 返回体
     */
    @Operation(summary = "获取平台配置信息")
    @PostMapping("/platform/getConfig")
    public  ResponseEntity<BaseResultDTO<PlatformConfig>> getPlatformConfig(@RequestBody PlatformConfig req) {
        return ResponseEntityWrapperUtil.wrapperOk(logoService.getPlatformConfig(req.getChannel()));
    }
}
