package com.digiwin.athena.semc.controller.portal;

import com.alibaba.fastjson.JSON;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.appcore.util.ResponseEntityWrapper;
import com.digiwin.athena.semc.common.*;
import com.digiwin.athena.semc.common.enums.EAIServiceNameEnum;
import com.digiwin.athena.semc.controller.BasicController;
import com.digiwin.athena.semc.dto.PageInfo;
import com.digiwin.athena.semc.dto.portal.LabelSystemPreReq;
import com.digiwin.athena.semc.dto.portal.NewsPageQueryResp;
import com.digiwin.athena.semc.dto.portal.NoticePageQueryReq;
import com.digiwin.athena.semc.dto.portal.TodoListResp;
import com.digiwin.athena.semc.entity.portal.News;
import com.digiwin.athena.semc.proxy.esp.service.ESPService;
import com.digiwin.athena.semc.service.portal.NewsService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.validation.Valid;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 公司新闻(News)表控制层
 *
 * @author sunyfa
 * @since 2022-12-06 13:38:38
 */
@RestController
@RequestMapping("/semc/news")
@Slf4j
public class NewsController extends BasicController<NewsService, News> {
    @Resource
    private ESPService espService;

    @Resource
    private NewsService newsService;

    @Resource
    private MessageUtils messageUtils;

    @PostMapping("/pageQuery")
    public ResultPageBean pageQueryNews(HttpServletRequest request,
                                        @Valid @RequestBody NoticePageQueryReq noticePageQueryReq) {
        log.info("Query news list param: {}", JsonUtils.objectToString(noticePageQueryReq));

        ResultPageBean resultPageBean;
        String tenantId = AppAuthContextHolder.getContext().getAuthoredUser().getTenantId();
        try {
            // 过滤后的公司新闻数据
            List<News> filteredNewsList = new ArrayList<>();
            // 公司新闻响应对象
            NewsPageQueryResp newsPageQueryResp = new NewsPageQueryResp();

            noticePageQueryReq.getEaiSysInfoList().forEach(eaiSysInfo -> {
                // 通过ESP查询三方系统的数据
                Map<String, Object> result = espService.queryByEsp(eaiSysInfo.getEaiSysName(),
                        eaiSysInfo.getEaiSysUid(), EAIServiceNameEnum.NEWS.getServiceName(), null, null, null, null);

                filteredNewsList.addAll(newsService.handleEspResult(result, eaiSysInfo, newsPageQueryResp,
                        noticePageQueryReq.getQryCondition()));
                filteredNewsList.forEach(x -> x.setNewsUrl(x.getNewsUrl() +
                        (x.getNewsUrl().contains("?") ? "&" : "?") + "appToken=" + eaiSysInfo.getAppToken() + "&tenantId=" + tenantId + "&bindMode=" + eaiSysInfo.getUserBindFlag()));
            });

            // 根据时间倒序排序
            List<News> sortedNewsAllList = filteredNewsList.stream()
                    .sorted(Comparator.comparing(News::getPublishTime).reversed()).collect(Collectors.toList());

            newsPageQueryResp.setNewsList(sortedNewsAllList.stream()
                    .skip((noticePageQueryReq.getPageNum() - 1) * noticePageQueryReq.getPageSize().longValue())
                    .limit(noticePageQueryReq.getPageSize()).collect(Collectors.toList()));

            resultPageBean = ResultPageBean.success(ResponseBody.getInstance(newsPageQueryResp, PageInfo.getPageInfo(
                    noticePageQueryReq.getPageNum(), noticePageQueryReq.getPageSize(), filteredNewsList.size())));
        } catch (BizException bizException) {
            return ResultPageBean.bizException(BizException.getDefaultBizException(ErrorCodeConstant.ADAPT_SYSTEM_ERROR,
                    messageUtils.getMessage(I18NKey.NEWS_OPEN_ERROR)));
        } catch (Exception e) {
            log.error("Query news Error", e);
            resultPageBean = ResultPageBean.sysException(new ServiceException(ErrorCodeConstant.SYSTEM_ERROR,
                    messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR)));
        }

        return resultPageBean;
    }

    /**
     * 查询通知公告数据
     *
     * @param newsListReq 请求参数
     * @return 返回对象
     */
    @PostMapping("/label/pageQueryNews")
    public ResultPageBean pageQueryNews(@Valid @RequestBody LabelSystemPreReq newsListReq) {
        log.info("pageQueryNews:{}", JSON.toJSONString(newsListReq));
        ResultPageBean resultPageBean;
        try {
            resultPageBean = newsService.pageQueryNews(newsListReq);
        } catch (RuntimeException bizException) {
            log.error("pageQueryNews exception", bizException);
            return ResultPageBean.bizException(BizException.getDefaultBizException(ErrorCodeConstant.ADAPT_SYSTEM_ERROR,
                    messageUtils.getMessage(I18NKey.NEWS_OPEN_ERROR)));
        } catch (Exception e) {
            log.error("pageQueryNews query exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/news/label/pageQueryNews");
            resultPageBean = ResultPageBean.sysException(new ServiceException(ErrorCodeConstant.SYSTEM_ERROR, error));
        }
        return resultPageBean;
    }
}
