package com.digiwin.athena.semc.controller.portal;

import com.alibaba.fastjson.JSON;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.semc.common.*;
import com.digiwin.athena.semc.common.enums.EAIServiceNameEnum;
import com.digiwin.athena.semc.controller.BasicController;
import com.digiwin.athena.semc.dto.PageInfo;
import com.digiwin.athena.semc.dto.portal.LabelSystemPreReq;
import com.digiwin.athena.semc.dto.portal.NoticePageQueryReq;
import com.digiwin.athena.semc.dto.portal.NoticePageQueryResp;
import com.digiwin.athena.semc.entity.portal.Notice;
import com.digiwin.athena.semc.env.EnvProperties;
import com.digiwin.athena.semc.proxy.esp.service.ESPService;
import com.digiwin.athena.semc.service.portal.NoticeService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.validation.Valid;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 通知控制层
 *
 * @author sunyfa
 * @since 2022-12-06 13:48:51
 */
@Slf4j
@RestController
@RequestMapping("/semc/notice")
public class NoticeController extends BasicController<NoticeService, Notice> {
    @Resource
    private NoticeService noticeService;

    @Resource
    private ESPService espService;

    @Resource
    private MessageUtils messageUtils;

    @Resource
    EnvProperties envProperties;

    @PostMapping("/pageQuery")
    public ResultPageBean pageQueryNotice(HttpServletRequest request,
        @Valid @RequestBody NoticePageQueryReq noticePageQueryReq) {
        log.info("Query notice list param: noticePageQueryReq:{}", JSON.toJSONString(noticePageQueryReq));

        ResultPageBean resultPageBean;
        String tenantId = AppAuthContextHolder.getContext().getAuthoredUser().getTenantId();
        try {
            Map<String, String> extHeader = new HashMap<>();
            extHeader.put("digi-userToken", AppAuthContextHolder.getContext().getAuthoredUser().getToken());

            // 查询条件过滤后的通知公告
            List<Notice> filteredNoticeList = new ArrayList<>();
            // 通知公告响应对象
            NoticePageQueryResp noticePageQueryResp = new NoticePageQueryResp();

            noticePageQueryReq.getEaiSysInfoList().forEach(eaiSysInfo -> {
                extHeader.put("digi-appToken", StringUtils.isBlank(eaiSysInfo.getAppToken()) ? envProperties.getAppToken() : eaiSysInfo.getAppToken());

                // 通过ESP查询三方系统的数据
                Map<String, Object> result = espService.queryByEsp(eaiSysInfo.getEaiSysName(),
                    eaiSysInfo.getEaiSysUid(), EAIServiceNameEnum.NOTICE.getServiceName(), extHeader, null, null, null);

                filteredNoticeList.addAll(noticeService.handleEspResult(result, eaiSysInfo,
                    noticePageQueryReq.getQryCondition(), noticePageQueryResp));

                filteredNoticeList.forEach(x -> x.setNoticeUrl(x.getNoticeUrl() +
                        (x.getNoticeUrl().contains("?") ? "&" : "?") + "appToken=" + eaiSysInfo.getAppToken() + "&tenantId=" + tenantId + "&bindMode=" + eaiSysInfo.getUserBindFlag()));
            });

            // 发布时间倒叙排序
            List<Notice> sortedNoticeAllList = filteredNoticeList.stream()
                .sorted(Comparator.comparing(Notice::getPublishTime).reversed()).collect(Collectors.toList());

            // 内存分页返回
            noticePageQueryResp.setNoticeList(sortedNoticeAllList.stream()
                .skip((noticePageQueryReq.getPageNum() - 1) * noticePageQueryReq.getPageSize().longValue())
                .limit(noticePageQueryReq.getPageSize()).collect(Collectors.toList()));

            resultPageBean = ResultPageBean.success(ResponseBody.getInstance(noticePageQueryResp, PageInfo.getPageInfo(
                noticePageQueryReq.getPageNum(), noticePageQueryReq.getPageSize(), sortedNoticeAllList.size())));
        }
        catch (Exception e) {
            log.error("notice query exception", e);
            resultPageBean = ResultPageBean.sysException(new ServiceException(ErrorCodeConstant.SYSTEM_ERROR,
                messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR)));
        }

        return resultPageBean;
    }

    /**
     * 查询公司新闻数据
     *
     * @param noticesListReq 请求参数
     * @return 返回对象
     */
    @PostMapping("/label/pageQueryNotice")
    public ResultPageBean pageQueryNotice(@Valid @RequestBody LabelSystemPreReq noticesListReq) {
        log.info("pageQueryNotices:{}", JSON.toJSONString(noticesListReq));
        ResultPageBean resultPageBean;
        try {
            resultPageBean = noticeService.pageQueryNotice(noticesListReq);
        } catch (Exception e) {
            log.error("pageQueryNotices query exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/notice/label/pageQueryNotices");
            resultPageBean = ResultPageBean.sysException(new ServiceException(ErrorCodeConstant.SYSTEM_ERROR, error));
        }
        return resultPageBean;
    }
}