package com.digiwin.athena.semc.controller.portal;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.appcore.util.ResponseEntityWrapper;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.I18NKey;
import com.digiwin.athena.semc.common.ResultPageBean;
import com.digiwin.athena.semc.dto.portal.*;
import com.digiwin.athena.semc.entity.mobile.MobilePortalInfo;
import com.digiwin.athena.semc.entity.portal.*;
import com.digiwin.athena.semc.mapper.mobile.MobilePortalInfoMapper;
import com.digiwin.athena.semc.mapper.portal.PortalInfoMapper;
import com.digiwin.athena.semc.proxy.dmc.service.DmcService;
import com.digiwin.athena.semc.proxy.dmc.service.FileInfo;
import com.digiwin.athena.semc.service.mobile.MobilePortalService;
import com.digiwin.athena.semc.service.portal.PortalInfoService;
import com.digiwin.athena.semc.service.portal.PortalVisitService;
import com.digiwin.athena.semc.service.workbench.constant.PublishSourceEnum;
import com.digiwin.athena.semc.util.InterceptorIgnoreUtil;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;
import com.digiwin.athena.semc.vo.portal.PortalInfoAuthReq;
import com.digiwin.athena.semc.vo.portal.PortalInfoAuthResp;
import com.digiwin.athena.semc.vo.portal.PortalUserAuthResp;
import io.swagger.v3.oas.annotations.Operation;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.annotation.Resource;
import javax.validation.Valid;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URLConnection;
import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * @description 多职能门户
 * @author: dengkaiwen
 */
@RestController
@RequestMapping("/semc/portal/")
public class PortalInfoController {

    private static final Logger logger = LoggerFactory.getLogger(PortalInfoController.class);

    @Autowired
    PortalInfoService portalInfoService;

    @Autowired
    PortalVisitService portalVisitService;

    @Resource
    private MessageUtils messageUtils;

    @Resource
    PortalInfoMapper portalInfoMapper;

    @Autowired
    MobilePortalInfoMapper mobilePortalInfoMapper;

    @Autowired
    private MobilePortalService mobilePortalService;

    /**
     * 查询用户选择门户接口
     *
     * @return
     */
    @Operation(method = "getUserPortal", description = "查询用户选择门户接口")
    @PostMapping("/info/getUserPortal")
    public ResponseEntity<?> getUserPortal() {
        PortalInfoUse infoUse = portalInfoService.getUserPortal();
        return ResponseEntityWrapper.wrapperOk(infoUse);
    }

    /**
     * 获取默认门户接口
     *
     * @return
     */
    @Operation(method = "getDefaultPortal", description = "获取默认门户接口")
    @GetMapping("/info/getDefaultPortal")
    public ResponseEntity<?> getDefaultPortal() {
        PortalInfo portalInfo = portalInfoService.getDefaultPortal();
        return ResponseEntityWrapper.wrapperOk(portalInfo);
    }


    /**
     * 门户保存
     *
     * @param req
     * @return
     */
    @Operation(method = "savePortal", description = "门户保存")
    @PostMapping("/info/savePortal")
    public ResponseEntity<?> savePortal(@RequestBody @Valid PortalInfo req) {
        req.setClientType(Optional.ofNullable(req.getClientType()).orElse(Constants.ClientTypeEnum.PC.getValue()));
        List<PortalInfo> portalInfoList = portalInfoService.checkPortal(req);
        if (CollectionUtils.isNotEmpty(portalInfoList)) {
            // 发布来源为ISV的门户不可编辑
            Optional<PortalInfo> optional = portalInfoList.stream()
                    .filter(portalInfo -> PublishSourceEnum.ISV.name().equals(portalInfo.getPublishSource()))
                    .findAny();
            if (optional.isPresent()) {
                String error = messageUtils.getMessage("error.unsupported.edit.isv.portal");
                return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, error);
            }

            if (null == req.getTemplateId()) {
                String error = messageUtils.getMessage("error.message.job.bench.name.repeat");
                return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, error);
            }
        }
        if (Constants.ClientTypeTypeEnum.MOBILE.getFlag().equals(req.getClientType())) {
            // 校验入参
            ResponseEntity<?> checkResult = checkMobilePortal(req);
            if (null != checkResult) {
                return checkResult;
            }
        }

        try {
            portalInfoService.savePortal(req);
        } catch (Exception e) {
            logger.error("savePortal exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/portal/info/savePortal");
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
        return ResponseEntityWrapper.wrapperOk();
    }

    public ResponseEntity<?> checkMobilePortal(PortalInfo req) {
        List<PortalInfoContent> contentList = req.getContentList();
        if (CollectionUtils.isEmpty(contentList)) {
            return null;
        }
        // 任务组件判断最多只有一个
        List<PortalInfoContent> taskDataList = contentList.stream().filter(item -> Constants.PrtalMobileContentTypeEnum.PRE_TASK.getVal().equals(item.getContentType())).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(taskDataList) && taskDataList.size() > 1) {
            String error = messageUtils.getMessage("error.message.mobile.portal.task.can.only.error");
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, error);
        }
        // 项目追踪组件判断最多只有一个
        List<PortalInfoContent> projectDataList = contentList.stream().filter(item -> Constants.PrtalMobileContentTypeEnum.PRE_PROJECT.getVal().equals(item.getContentType())).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(projectDataList) && taskDataList.size() > 1) {
            String error = messageUtils.getMessage("error.message.mobile.portal.project.can.only.error");
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, error);
        }
        // 金刚区判断
        List<PortalInfoContent> districtList = contentList.stream().filter(item -> Constants.PrtalMobileContentTypeEnum.PRE_KING_KONG_DISTRICT.getVal().equals(item.getContentType())).collect(Collectors.toList());
        int customDistrict = 0;
        for (PortalInfoContent district : districtList) {
            PortalContentKingKongDistrict kingKongDistrict = JSON.parseObject(JSON.toJSONString(district.getContentConfig()), PortalContentKingKongDistrict.class);

            if (kingKongDistrict.getCustomType().equals(Constants.DistrictCustomTypeEnum.USER_TYPE.getValue())) {
                customDistrict++;
            }
            // 用户自定义金刚区只能有一个
            if (customDistrict > 1) {
                String error = messageUtils.getMessage("error.message.mobile.portal.custom.district.can.only.error");
                return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, error);
            }
            if (Constants.DistrictinteractionTypeEnum.EXPAND_ALL.getValue().equals(kingKongDistrict.getInteractionType())) {
                kingKongDistrict.setLineNum(null);
            }
        }
        return null;
    }

    /**
     * 管理后台-查询门户列表
     *
     * @param req 请求对象
     * @return 返回
     */
    @Operation(method = "/manage/pageQuery", description = "查询门户列表")
    @PostMapping("/manage/pageQuery")
    public ResultPageBean pageQuery(@RequestBody @Valid PortalInfoQueryReq req) {
        return portalInfoService.pageQuery(req);
    }


    /**
     * 批量发布和取消发布
     *
     * @param req 请求对象
     * @return 返回
     */
    @Operation(method = "updateStatus", description = "批量发布和取消发布")
    @PostMapping("/info/updateStatus")
    public ResponseEntity<?> updateStatus(@RequestBody PortalInfoQueryReq req) {
        if (CollectionUtils.isEmpty(req.getIdList()) && CollectionUtils.isEmpty(req.getMobileIdReqList())) {
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "idList is null");
        }
        if (null == req.getStatus()) {
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "status is null");
        }
        // ISV门户不能发布或取消发布
        // pc端传idList
        if (CollectionUtils.isNotEmpty(req.getIdList())) {
            if (portalInfoService.checkISVPortal(req.getIdList())) {
                String error = messageUtils.getMessage("error.unsupported.push.isv.portal");
                return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, error);
            }
        } else if (CollectionUtils.isNotEmpty(req.getMobileIdReqList())) { // 移动端传mobileIdReqList
            List<Long> newPortalIdList = req.getMobileIdReqList().stream().filter(x -> Boolean.FALSE.equals(x.getOldPortalFlag())).map(PortalInfoQueryReq.MobileIdReq::getId).collect(Collectors.toList());
            if (CollectionUtils.isNotEmpty(newPortalIdList) && portalInfoService.checkISVPortal(newPortalIdList)) {
                String error = messageUtils.getMessage("error.unsupported.push.isv.portal");
                return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, error);
            }
        }
        portalInfoService.updateBat(req);
        return ResponseEntityWrapper.wrapperOk();
    }


    /**
     * 批量删除布局
     *
     * @param req 请求对象
     * @return 返回
     */
    @Operation(method = "batchDel", description = "批量删除布局")
    @PostMapping("info/batchDel")
    public ResponseEntity<?> batchDel(@RequestBody PortalInfoQueryReq req) {
        if (CollectionUtils.isEmpty(req.getIdList()) && CollectionUtils.isEmpty(req.getMobileIdReqList())) {
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "idList is null");
        }
        // ISV门户不能删除
        // pc端传idList
        if (CollectionUtils.isNotEmpty(req.getIdList())) {
            if (portalInfoService.checkISVPortal(req.getIdList())) {
                String error = messageUtils.getMessage("error.unsupported.delete.isv.portal");
                return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, error);
            }
        } else if (CollectionUtils.isNotEmpty(req.getMobileIdReqList())) { // 移动端传mobileIdReqList
            List<Long> newPortalIdList = req.getMobileIdReqList().stream().filter(x -> Boolean.FALSE.equals(x.getOldPortalFlag())).map(PortalInfoQueryReq.MobileIdReq::getId).collect(Collectors.toList());
            if (CollectionUtils.isNotEmpty(newPortalIdList) && portalInfoService.checkISVPortal(newPortalIdList)) {
                String error = messageUtils.getMessage("error.unsupported.delete.isv.portal");
                return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, error);
            }
        }
        portalInfoService.delBat(req);
        return ResponseEntityWrapper.wrapperOk();
    }

    /**
     * 保存组件权限
     *
     * @param req 请求实体
     * @return
     * @throws Exception
     */
    @Operation(summary = "保存组件权限")
    @PostMapping("/userAuth/savePortalAuth")
    public ResponseEntity<BaseResultDTO<Integer>> savePortalAuth(@RequestBody @Valid PortalInfoAuthReq req) {
        try {
            return ResponseEntityWrapperUtil.wrapperOk(portalInfoService.savePortalAuth(req));
        } catch (Exception e) {
            logger.error("savePortalAuth exception", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 查询门户权限信息
     *
     * @param req 请求实体
     * @return
     * @throws Exception
     */
    @Operation(summary = "查询门户权限信息")
    @PostMapping("/userAuth/selectByPortal")
    public ResponseEntity<BaseResultDTO<PortalInfoAuthResp>> selectByPortal(@RequestBody PortalInfoAuth req) {
        if (null == req.getId()) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "id  cannot be empty");
        }
        return ResponseEntityWrapperUtil.wrapperOk(portalInfoService.selectByPortalAuth(req.getId()));
    }


    /**
     * 查询用户职能门户列表
     *
     * @return
     * @throws Exception
     */
    @Operation(summary = "查询用户职能门户列表")
    @PostMapping("/info/userAuth/selectUserPortal")
    public ResponseEntity<BaseResultDTO<List<PortalUserAuthResp>>> selectUserPortal(@RequestBody PortalInfo req) {
        try {
            return ResponseEntityWrapperUtil.wrapperOk(portalInfoService.selectUserPortal(req.getClientType()));
        } catch (Exception e) {
            logger.error("selectUserPortal exception", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }


    /**
     * 获取门户数据详情-管理后台使用
     *
     * @param req 请求参数
     * @return 返回
     */
    @Operation(method = "getPortalDetail", description = "获取门户数据详情")
    @PostMapping("/info/getPortalDetail")
    public ResponseEntity<?> getPortalDetail(@RequestBody PortalInfo req) {
        // 前端根据门户列表里的门户类型传对应的门户id，即ISV门户传ISV设计时门户id，租户自定义门户传主键id
        if (null == req.getId()) {
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "id cannot be empty");
        }
        PortalInfo portalInfo;
        if (Constants.LabelOrPoralSourceEnum.TENANT.getFlag().equals(req.getPortalTypeSource())) {
            portalInfo = portalInfoService.getTempPortal(req.getId());
        } else {
            portalInfo = portalInfoService.queryPortalInfoByDesignerId(req.getId());
            portalInfo.setPortalTypeSource(Constants.PortalTypeSourceEnum.ISV.getFlag());
        }
        if (null == portalInfo) {
            String recentlyMsg = messageUtils.getMessage("error.message.choose.portal.recovery");
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, recentlyMsg);
        }
        return portalInfoService.getPortalDetail(portalInfo);
    }


    /**
     * 获取门户详情--用户首页使用
     *
     * @param req
     * @return
     */
    @Operation(method = "getPortalInfo", description = "获取门户详情")
    @PostMapping("/info/getPortalInfo")
    public ResponseEntity<?> getPortalInfo(@RequestBody UserPortalInfoRequest req) {
        req.setClientType(Optional.ofNullable(req.getClientType()).orElse(Constants.ClientTypeEnum.PC.getValue()));
        if (Constants.ClientTypeEnum.MOBILE.getValue().equals(req.getClientType())) {
            return this.getPortalInfoByMobile(req);
        } else {
            return this.getPortalInfoByPc(req);
        }
    }

    private ResponseEntity<?> getPortalInfoByPc(UserPortalInfoRequest req) {
        Long id = req.getId();
        if (null == id) {
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "id  cannot be empty");
        }
        Integer clientType = Optional.ofNullable(req.getClientType()).orElse(Constants.ClientTypeEnum.PC.getValue());
        Integer portalTypeSource = Optional.ofNullable(req.getPortalTypeSource()).orElse(Constants.PortalTypeSourceEnum.TENANT.getFlag());
        PortalInfo portalInfo;
        if (Constants.PortalTypeSourceEnum.ISV.getFlag().equals(portalTypeSource)) {
            // isv
            portalInfo = portalInfoService.queryPortalInfoByDesignerId(id);
        } else {
            LambdaQueryWrapper<PortalInfo> queryWrapper = new LambdaQueryWrapper<PortalInfo>()
                    .eq(PortalInfo::getId, id)
                    .eq(PortalInfo::getClientType, clientType);
            portalInfo = InterceptorIgnoreUtil.handler(() -> portalInfoMapper.selectOne(queryWrapper));
        }
        if (null == portalInfo) {
            portalInfo = new PortalInfo();
            String recentlyMsg = messageUtils.getMessage("error.message.choose.portal.recovery");
            portalInfo.setRecentlyMsg(recentlyMsg);
            return ResponseEntityWrapper.wrapperOk(portalInfo);
        }
        // 门户未发布 非默认门户（默认门户不控制状态）
        if (portalInfo.getStatus().equals(Constants.ReleaseEnum.NO_RELEASE.getVal())
                && !portalInfo.getDefaultFlag().equals(Constants.DEFAULT_CONFIGURED_FLAG_YES)) {
            String recentlyMsg = messageUtils.getMessage("error.message.choose.portal.recovery");
            portalInfo.setRecentlyMsg(recentlyMsg);
            return ResponseEntityWrapper.wrapperOk(portalInfo);
        }
        try {
            portalInfo.setPortalTypeSource(portalTypeSource);
            return portalInfoService.getPortalInfo(portalInfo);
        } catch (Exception e) {
            logger.error("getPortalInfo exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/portal/info/getPortalInfo");
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }


    private ResponseEntity<?> getPortalInfoByMobile(UserPortalInfoRequest req) {
        // 场景：0：正常用户访问、1：预览 ，默认0
        if (req.getScene() == 0) {
            PortalInfo portalInfo = portalInfoService.getUserMobilePortalInfo(req);
            // 门户布局的判断
            if (null == portalInfo
                    && req.getPortalType().equals(Constants.MobilePortalTypeEnum.PORTAL.getFlag())) {
                portalInfo = new PortalInfo();
                String recentlyMsg = messageUtils.getMessage("error.message.not.auth");
                portalInfo.setRecentlyMsg(recentlyMsg);
                return ResponseEntityWrapperUtil.wrapperOk(portalInfo);
            }
            // 自定义页面的判断
            if (null == portalInfo && req.getPortalType().equals(Constants.MobilePortalTypeEnum.CUSTOM_PORTAL.getFlag())) {
                String recentlyMsg = messageUtils.getMessage(I18NKey.MOBILE_PORTAL_CUSTOM_DEL);
                return ResponseEntityWrapperUtil.wrapperOk(recentlyMsg);
            }
            if (null == portalInfo) {
                portalInfo = new PortalInfo();
                String recentlyMsg = messageUtils.getMessage("error.message.not.auth");
                portalInfo.setRecentlyMsg(recentlyMsg);
                return ResponseEntityWrapperUtil.wrapperOk(portalInfo);
            }
            // 门户未发布 非默认门户（默认门户不控制状态）
            if (portalInfo.getPortalType().equals(Constants.MobilePortalTypeEnum.PORTAL.getFlag())
                    && portalInfo.getStatus().equals(Constants.ReleaseEnum.NO_RELEASE.getVal())
                    && !portalInfo.getDefaultFlag().equals(Constants.DEFAULT_CONFIGURED_FLAG_YES)) {
                String recentlyMsg = messageUtils.getMessage("error.message.not.auth");
                portalInfo.setRecentlyMsg(recentlyMsg);
                return ResponseEntityWrapperUtil.wrapperOk(portalInfo);
            }
            // 自定义页面-未发布场景 判断
            if (portalInfo.getPortalType().equals(Constants.MobilePortalTypeEnum.CUSTOM_PORTAL.getFlag())
                    && portalInfo.getStatus().equals(Constants.ReleaseEnum.NO_RELEASE.getVal())) {
                String recentlyMsg = messageUtils.getMessage(I18NKey.MOBILE_PORTAL_CUSTOM_DEL);
                return ResponseEntityWrapperUtil.wrapperOk(recentlyMsg);
            }
            try {
                return ResponseEntityWrapperUtil.wrapperOk(portalInfoService.getUserMobilePortalDetail(portalInfo));
            } catch (Exception e) {
                logger.error("getPortalInfo exception {}", e.getMessage());
                String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/portal/info/getPortalInfo");
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
            }
        }

        // 门户预览
        PortalInfo result = portalInfoService.getPreMobilePortal(req.getId());

        return ResponseEntityWrapperUtil.wrapperOk(result);
    }

    /**
     * 校验门户权限
     *
     * @param req
     * @return
     */
    @Operation(method = "checkPortalInfo", description = "校验门户权限")
    @PostMapping("/info/checkPortalInfo")
    public ResponseEntity<?> checkPortalInfo(@RequestBody PortalInfo req) {
        req.setClientType(Optional.ofNullable(req.getClientType()).orElse(Constants.ClientTypeEnum.PC.getValue()));
        if (Constants.ClientTypeEnum.MOBILE.getValue().equals(req.getClientType())) {
            // 移动端
            return this.checkPortalInfoByMobile(req);
        } else {
            // PC端
            return this.checkPortalInfoByPc(req);
        }
    }

    private ResponseEntity<?> checkPortalInfoByMobile(PortalInfo req) {
        if (null == req.getId()) {
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "id  cannot be empty");
        }
        PortalInfoCheckResp checkResp = new PortalInfoCheckResp();
        checkResp.setIsHavePort(false);
        PortalInfo portalInfo;
        MobilePortalInfo mobilePortalInfo = null;
        if (Constants.PortalTypeSourceEnum.ISV.getFlag().equals(req.getPortalTypeSource())) {
            // isv
            portalInfo = portalInfoService.queryPortalInfoByDesignerId(req.getId());
        } else {
            // 需使用clientType过滤, 兼容移动历史数据，防止不同客户端的门户ID相同
            LambdaQueryWrapper<PortalInfo> queryWrapper = new LambdaQueryWrapper<PortalInfo>()
                    .eq(PortalInfo::getId, req.getId())
                    .eq(PortalInfo::getClientType, req.getClientType());
            portalInfo = InterceptorIgnoreUtil.handler(() -> portalInfoMapper.selectOne(queryWrapper));
            if (portalInfo == null && Constants.ClientTypeEnum.MOBILE.getValue().equals(req.getClientType())) {
                // 兼容移动历史数据
                mobilePortalInfo = mobilePortalInfoMapper.selectById(req.getId());
                if (mobilePortalInfo != null) {
                    portalInfo = new PortalInfo();
                    BeanUtils.copyProperties(mobilePortalInfo, portalInfo);
                    portalInfo.setClientType(Constants.ClientTypeEnum.MOBILE.getValue());
                    portalInfo.setMenuTypeSource(Constants.MenuTypeSourceEnum.TENANT.getFlag());
                    portalInfo.setPublishSource(PublishSourceEnum.TENANT.name());
                }
            }
        }
        // 门户布局-删除场景 判断
        if (null == portalInfo && req.getPortalType().equals(Constants.MobilePortalTypeEnum.PORTAL.getFlag())) {
            checkResp.setErrorMsg(messageUtils.getMessage(I18NKey.PORTAL_NOT_AUTH));
            return ResponseEntityWrapper.wrapperOk(checkResp);
        }
        // 自定义页面的判断
        if (null == portalInfo && req.getPortalType().equals(Constants.MobilePortalTypeEnum.CUSTOM_PORTAL.getFlag())) {
            checkResp.setErrorMsg(messageUtils.getMessage(I18NKey.MOBILE_PORTAL_CUSTOM_DEL));
            return ResponseEntityWrapper.wrapperOk(checkResp);
        }
        if (null == portalInfo) {
            checkResp.setErrorMsg(messageUtils.getMessage(I18NKey.PORTAL_NOT_AUTH));
            return ResponseEntityWrapper.wrapperOk(checkResp);
        }
        checkResp.setPortalName(portalInfo.getName());
        // 门户布局-未发布场景 判断
        if (!Objects.equals(Constants.DEFAULT_CONFIGURED_FLAG_YES,portalInfo.getDefaultFlag()) && portalInfo.getPortalType().equals(Constants.MobilePortalTypeEnum.PORTAL.getFlag())
                && portalInfo.getStatus().equals(Constants.ReleaseEnum.NO_RELEASE.getVal())) {
            checkResp.setErrorMsg(messageUtils.getMessage(I18NKey.PORTAL_NOT_AUTH));
            return ResponseEntityWrapper.wrapperOk(checkResp);
        }
        // 自定义页面-未发布场景 判断
        if (portalInfo.getPortalType().equals(Constants.MobilePortalTypeEnum.CUSTOM_PORTAL.getFlag())
                && portalInfo.getStatus().equals(Constants.ReleaseEnum.NO_RELEASE.getVal())) {
            checkResp.setErrorMsg(messageUtils.getMessage(I18NKey.MOBILE_PORTAL_CUSTOM_DEL));
            return ResponseEntityWrapper.wrapperOk(checkResp);
        }
        // 若是默认门户则
        if (portalInfo.getDefaultFlag() == 1) {
            checkResp.setIsHavePort(true);
            return ResponseEntityWrapper.wrapperOk(checkResp);
        }
        try {
            if (mobilePortalInfo == null) {
                return portalInfoService.checkPortalInfo(portalInfo);
            } else {
                return mobilePortalService.checkPortalInfo(mobilePortalInfo);
            }
        } catch (Exception e) {
            logger.error("checkPortalInfo exception {}", e.getMessage());
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/mobilePortal/info/getPortalInfo");
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    private ResponseEntity<?> checkPortalInfoByPc(PortalInfo req) {
        if (null == req.getId()) {
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "id  cannot be empty");
        }
        PortalInfoCheckResp checkResp = new PortalInfoCheckResp();
        checkResp.setIsHavePort(false);
        PortalInfo portalInfo;
        if (Constants.PortalTypeSourceEnum.ISV.getFlag().equals(req.getPortalTypeSource())) {
            // isv
            portalInfo = portalInfoService.queryPortalInfoByDesignerId(req.getId());
        } else {
            LambdaQueryWrapper<PortalInfo> queryWrapper = new LambdaQueryWrapper<PortalInfo>()
                    .eq(PortalInfo::getId, req.getId())
                    .eq(PortalInfo::getClientType, req.getClientType());
            portalInfo = InterceptorIgnoreUtil.handler(() -> portalInfoMapper.selectOne(queryWrapper));
        }
        if (null == portalInfo) {
            return ResponseEntityWrapper.wrapperOk(checkResp);
        }
        // 门户未发布
        if (portalInfo.getStatus().equals(Constants.ReleaseEnum.NO_RELEASE.getVal())) {
            return ResponseEntityWrapper.wrapperOk(checkResp);
        }
        // 默认布局
        if (portalInfo.getDefaultFlag().equals(Constants.DEFAULT_CONFIGURED_FLAG_YES)) {
            checkResp.setIsHavePort(true);
            return ResponseEntityWrapper.wrapperOk(checkResp);
        }
        try {
            return portalInfoService.checkPortalInfo(portalInfo);
        } catch (Exception e) {
            logger.error("checkPortalInfo exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/portal/info/getPortalInfo");
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 记录用户使用门户记录
     *
     * @param req
     * @return
     */
    @PostMapping("/info/addUseRecord")
    public ResponseEntity<?> addUseRecord(@RequestBody AddUseRecordRequest req) {
        if (Constants.ClientTypeEnum.MOBILE.getValue().equals(req.getClientType())) {
            // 移动端访问记录
            return this.addUseRecordByMobile(req);
        } else {
            // PC端访问记录
            return this.addUseRecordByPc(req);
        }

    }

    private ResponseEntity<?> addUseRecordByMobile(AddUseRecordRequest req) {
       /* if (ObjectUtils.isEmpty(req.getPortalType())) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "未传入相关的参数");
        }
        if (req.getPortalType() == 1 && ObjectUtils.isEmpty(req.getId())) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "未传入相关的参数");
        }*/
        if (req.getClientType() == null) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "未传入相关的参数");
        }
        if (req.getPortalTypeSource() == null) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "未传入相关的参数");
        }
        portalInfoService.addUseRecordByMobile(req);
        return ResponseEntityWrapperUtil.wrapperOk(req.getId());
    }

    private ResponseEntity<?> addUseRecordByPc(AddUseRecordRequest req) {
        Long id = req.getId();
        if (null == id) {
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "id  cannot be empty");
        }
        Integer clientType = Optional.ofNullable(req.getClientType()).orElse(Constants.ClientTypeEnum.PC.getValue());
        Integer portalTypeSource = Optional.ofNullable(req.getPortalTypeSource()).orElse(Constants.PortalTypeSourceEnum.TENANT.getFlag());
        PortalInfo portalInfo;
        if (Constants.PortalTypeSourceEnum.ISV.getFlag().equals(portalTypeSource)) {
            // isv
            portalInfo = portalInfoService.queryPortalInfoByDesignerId(id);
        } else {
            LambdaQueryWrapper<PortalInfo> queryWrapper = new LambdaQueryWrapper<PortalInfo>()
                    .eq(PortalInfo::getId, id)
                    .eq(PortalInfo::getClientType, clientType);
            portalInfo = InterceptorIgnoreUtil.handler(() -> portalInfoMapper.selectOne(queryWrapper));
        }
        if (null == portalInfo) {
            portalInfo = new PortalInfo();
            String recentlyMsg = messageUtils.getMessage("error.message.choose.portal.recovery");
            portalInfo.setRecentlyMsg(recentlyMsg);
            return ResponseEntityWrapper.wrapperOk(portalInfo);
        }
        try {
            portalInfoService.addUseRecord(id, clientType, portalTypeSource);
            return ResponseEntityWrapper.wrapperOk(id);
        } catch (Exception e) {
            logger.error("getPortalInfo exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/portal/info/getPortalInfo");
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }


    /**
     * 查询链接选项
     *
     * @param req 设计时参数
     * @return
     */
    @PostMapping("/queryLinkList")
    public ResponseEntity<BaseResultDTO<List<PortalSelectOptionRespDTO>>> queryLinkList(@RequestBody Map<String, Integer> req) {
        try {
            return ResponseEntityWrapperUtil.wrapperOk(portalInfoService.queryLinkList(req.get("isvDesign")));
        } catch (Exception e) {
            logger.error("queryLinkList exception", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 校验链接选项的权限
     *
     * @return
     */
    @PostMapping("/checkLinkPerm")
    public ResponseEntity<BaseResultDTO<WebLinkPermissionDTO>> checkLinkPerm(@RequestBody WebLinkPermissionDTO req) {
        try {
            if (null == req.getType()) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, "type is null");
            }
            return ResponseEntityWrapperUtil.wrapperOk(portalInfoService.checkLinkPerm(req));
        } catch (Exception e) {
            logger.error("checkLinkPerm exception", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 记录ISV门户的访问记录
     *
     * @param visitPortalReq 请求对象
     * @return 返回
     */
    @PostMapping("/visitPortal")
    public ResponseEntity<?> visitPortal(@RequestBody @Valid VisitPortalReq visitPortalReq) {
        portalVisitService.recordVisit(visitPortalReq);
        return ResponseEntityWrapperUtil.wrapperOk(true);
    }

    /**
     * 复制门户
     *
     * @param req 请求实体
     * @return
     * @throws Exception
     */
    @Operation(summary = "复制门户")
    @PostMapping("info/copyPortal")
    public ResponseEntity<?> copyPortal(@RequestBody PortalInfo req) {
        try {
            if (null == req.getId() || req.getOldPortalFlag() == null) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "id  cannot be empty");
            }
            return portalInfoService.copyPortal(req.getId(), req.getOldPortalFlag());
        } catch (Exception e) {
            logger.error("copyPortal exception", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    @Autowired
    private DmcService dmcService;

    /**
     * 复制门户
     *
     * @param file 请求实体
     * @return
     * @throws Exception
     */
    @Operation(summary = "复制门户")
    @PostMapping("/upload")
    public ResponseEntity<?> upload(@RequestParam("file") MultipartFile file, @RequestParam("ids") String ids) {
        File file2 = convertMultiPartFileToFileV2(file);
        // 解析出 剔除 后缀名的文件名
        FileInfo fileInfo = new FileInfo();
        fileInfo.setId(ids);
        String contentType = URLConnection.guessContentTypeFromName(file2.getName());
        String id = dmcService.upload(file2, contentType, fileInfo);
        return ResponseEntityWrapper.wrapperOk(id);
    }

    public File convertMultiPartFileToFileV2(MultipartFile file) {
        File convFile = new File(file.getOriginalFilename());
        try (InputStream targetStream = file.getInputStream();
             FileOutputStream outStream = new FileOutputStream(convFile)) {
            byte[] buffer = new byte[4096];
            int bytesRead;
            while ((bytesRead = targetStream.read(buffer)) != -1) {
                outStream.write(buffer, 0, bytesRead);
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
        return convFile;
    }
}
