package com.digiwin.athena.semc.controller.portal;

import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.appcore.auth.GlobalConstant;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.appcore.util.ResponseEntityWrapper;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.I18NKey;
import com.digiwin.athena.semc.common.enums.PreinstalledApplicationTypeEnum;
import com.digiwin.athena.semc.dto.erpsso.PreApplicationUpdateDto;
import com.digiwin.athena.semc.dto.portal.PreinstalledApplicationConfigUpdateDto;
import com.digiwin.athena.semc.entity.portal.PreinstalledApplication;
import com.digiwin.athena.semc.service.portal.IPreinstalledApplicationService;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;
import com.digiwin.athena.semc.util.SpecialValidator;
import io.swagger.v3.oas.annotations.Operation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.validation.Valid;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.util.List;
import java.util.Objects;

/**
 * @Author: sunyfa
 * @CreateTime: 2023/1/29 9:38
 * @Description: 预装应用
 * @Version: 1.0
 */
@Slf4j
@RestController
@RequestMapping("/semc/management/preinstalledapp")
public class PreinstalledApplicationController {

    private static final Logger logger = LoggerFactory.getLogger(PreinstalledApplicationController.class);

    @Resource
    private IPreinstalledApplicationService preinstalledApplicationService;

    @Resource
    private MessageUtils messageUtils;

    /**
     * @return 预装应用列表
     * @description: 查询预装应用列表
     * @author: sunyfa
     */
    @GetMapping("/getPreinstalledApplicationConfig")
    public ResponseEntity<BaseResultDTO<List<PreinstalledApplication>>> queryPreinstalledAppList(@RequestAttribute(value = GlobalConstant.AUTH_USER) AuthoredUser user) {
        List<PreinstalledApplication> preinstalledApplicationList = preinstalledApplicationService.queryPreinstalledAppList(user);
        if (CollectionUtils.isNotEmpty(preinstalledApplicationList)) {
            preinstalledApplicationList.forEach(preinstalledApplication -> preinstalledApplication.setApplicationConfig(StringUtils.EMPTY));
        }
        return ResponseEntityWrapperUtil.wrapperOk(preinstalledApplicationList);
    }

    /**
     * @param preinstalledApplicationId 预装应用的id
     * @return preinstalledApplication 配置详情
     * @description: 查询预装应用的配置详情
     * @author: sunyfa
     */
    @GetMapping("/getPreinstalledApplicationConfig/{applicationType}/{preinstalledApplicationId}")
    public ResponseEntity<BaseResultDTO<JSONObject>> queryPreinstalledAppDetail(
            @RequestAttribute(value = GlobalConstant.AUTH_USER) AuthoredUser user,
            @PathVariable("applicationType") @NotEmpty(message = "应用类型不能为空") String applicationType,
            @PathVariable("preinstalledApplicationId") @NotNull(message = "应用id不能为空") long preinstalledApplicationId) {
        JSONObject jsonObject = preinstalledApplicationService.queryPreinstalledAppDetail(user, applicationType, preinstalledApplicationId);
        return ResponseEntityWrapperUtil.wrapperOk(jsonObject);
    }

    /**
     * @param preinstalledApplicationConfigUpdateDto 需要更新的配置信息
     * @return
     * @description: 更新应用
     * @author: sunyfa
     */
    @PostMapping("/updatePreinstalledApplicationConfig")
    public ResponseEntity<?> updatePreinstalledAppConfig(
            @RequestAttribute(value = GlobalConstant.AUTH_USER) AuthoredUser user,
            @RequestBody @Valid PreinstalledApplicationConfigUpdateDto preinstalledApplicationConfigUpdateDto) {
        // 参数校验
        if (!Objects.isNull(preinstalledApplicationConfigUpdateDto.getConfig()) && preinstalledApplicationConfigUpdateDto.getConfig().toString().length() > 200) {
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_LENGTH_ERROR, messageUtils.getMessage(I18NKey.COMMON_PARAM_LENGTH));
        }
        // 更新
        preinstalledApplicationService.updatePreinstalledAppConfig(user, preinstalledApplicationConfigUpdateDto);
        return ResponseEntityWrapper.wrapperOk();
    }

    /**
     * 更新应用是否显示在友情链接标识
     *
     * @param preApplicationUpdateDto 请求实体
     * @return
     */
    @Operation(summary = "更新应用友情链接显示标识")
    @PostMapping("/updatePreApplicationDisplayFlag")
    public ResponseEntity<BaseResultDTO<Boolean>> updatePreApplicationDisplayFlag(@RequestBody PreApplicationUpdateDto preApplicationUpdateDto) {
        try {
            preinstalledApplicationService.updateAppInfo(preApplicationUpdateDto.getPreApplicationIdList(), null, preApplicationUpdateDto.getLinkDisplayFlag());
            return ResponseEntityWrapperUtil.wrapperOk(Boolean.TRUE);
        } catch (Exception e) {
            logger.error("update pre application display flag error. preApplicationUpdateDto:{} ", preApplicationUpdateDto, e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 删除应用，内部使用的工具接口
     *
     * @param idList 请求实体
     * @return
     */
    @GetMapping("/delApplicationInstance")
    public ResponseEntity<BaseResultDTO<Integer>> delApplicationInstance(@RequestParam(value = "ids") List<Long> idList) {
        try {
            Integer count = preinstalledApplicationService.delApplicationInstance(idList);
            return ResponseEntityWrapperUtil.wrapperOk(count);
        } catch (Exception e) {
            logger.error("del app application instance error. idList:{} ", idList, e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }
}
