package com.digiwin.athena.semc.controller.portal;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.I18NKey;
import com.digiwin.athena.semc.common.PageInfoResp;
import com.digiwin.athena.semc.controller.BasicController;
import com.digiwin.athena.semc.dto.message.DeleteMessageConfigReq;
import com.digiwin.athena.semc.dto.message.QueryThirdMessageConfigReq;
import com.digiwin.athena.semc.entity.portal.ThirdTodoConfig;
import com.digiwin.athena.semc.quartz.ThirdTodoPullTaskJob;
import com.digiwin.athena.semc.service.menu.AuthService;
import com.digiwin.athena.semc.service.portal.LabelSystemDataService;
import com.digiwin.athena.semc.service.portal.ThirdTodoConfigService;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;
import com.digiwin.athena.semc.util.SecurityLogUtil;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.validation.Valid;

import lombok.extern.slf4j.Slf4j;

/**
 * @description: 三方异构系统待办接入
 * @createDate: 2024/3/4
 * @author: sungqz
 */
@Slf4j
@RestController
@RequestMapping("/semc/third/toDo")
public class ThirdTodoConfigontroller extends BasicController<ThirdTodoConfigService, ThirdTodoConfig> {

    @Resource
    private ThirdTodoConfigService thirdTodoConfigService;


    @Resource
    ThirdTodoPullTaskJob thirdTodoPullTaskJob;

    @Resource
    private MessageUtils messageUtils;

    @Autowired
    private LabelSystemDataService labelSystemDataService;

    @Autowired
    private AuthService authService;

    /**
     * 查询异构系统的待办配置
     *
     * @param queryImportRecordReq 入参
     * @return 返回
     */
    @PostMapping("/queryToDoConfigPage")
    public ResponseEntity<BaseResultDTO<PageInfoResp<ThirdTodoConfig>>> queryToDoConfigPage(@RequestBody @Valid QueryThirdMessageConfigReq queryImportRecordReq) {
        try {
            PageInfoResp<ThirdTodoConfig> pageInfoResp = thirdTodoConfigService.queryConfigPage(queryImportRecordReq);
            return ResponseEntityWrapperUtil.wrapperOk(pageInfoResp);
        } catch (Exception e) {
            log.error("queryToDoConfigPage error, param:{}", queryImportRecordReq, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/third/toDo/queryToDoConfigPage");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 保存异构系统的待办配置
     *
     * @param req 入参
     * @return 返回
     */
    @PostMapping("/saveApplication")
    public ResponseEntity<BaseResultDTO<Boolean>> saveApplication(@RequestBody @Valid ThirdTodoConfig req, HttpServletRequest  request) {
        try {
            //集成管理/待办接入配置
            //暂时不阻断，只是写入安全日志
            if (!authService.hasAuth("third-todo")) {
                //写入安全日志
                SecurityLogUtil.writeSecurityLog(request, JSON.toJSONString(req));
            }
            // 数据获取方式如果是拉取方式
            if (Constants.DataGetTypeEnum.PULL.getFlag().equals(req.getDataGetType())) {
                if (req.getPullPeriod() == null) {
                    return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "the pullPeriod cannot be null");
                }
                if (req.getAppAccessModel() == null) {
                    return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "the appAccessModel cannot be null");
                }

                // 混合云模式需要填地中台产品名称，系统UID
                if (Constants.DataModelEnum.MODEL_HYBRID_CLOUD.getVal().equals(req.getAppAccessModel())) {
                    if (StringUtils.isBlank(req.getMiddleSystemName())) {
                        return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "the middleSystemName cannot be null");
                    }
                    if (StringUtils.isBlank(req.getMiddleSystemUid())) {
                        return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "the middleSystemUid cannot be null");
                    }
                } else { // 非混合云
                    if (StringUtils.isBlank(req.getDomain())) {
                        return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "the domain cannot be null");
                    }
                }
            }
            // 判断应用是否存在
            boolean flag = thirdTodoConfigService.validConfigExist(req);
            if (flag) {
                String error = messageUtils.getMessage("error.message.app.repeat");
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, error);
            }
            if(req.getId()!=null){
                //删除定时任务
                thirdTodoPullTaskJob.deleteJob("ThirdTodoPullTaskJob","TodoJob_"+req.getId());
            }
            thirdTodoConfigService.saveThirdToDo(req);
            if(req.getDataGetType()==2&&req.getValidStatus()==1) {
                //启动定时任务
                Map<String,Object>params=new HashMap<>();
                params.put("id",req.getId());
                params.put("appPrimaryId",req.getAppPrimaryId());
                params.put("appSource",req.getAppSource());
                params.put("pullPeriod",req.getPullPeriod());
                params.put("appAccessModel",req.getAppAccessModel());
                params.put("middleSystemName",req.getMiddleSystemName());
                params.put("middleSystemUid",req.getMiddleSystemUid());
                params.put("domain",req.getDomain());
                //params.put("userToken",AppAuthContextHolder.getContext().getAuthoredUser().getToken());
                params.put("tenantsid",AppAuthContextHolder.getContext().getAuthoredUser().getTenantId());
                //params.put("tenantsid","athenaPaasW");
                //获取appToken
                Map<String, String> ssoInfoMap = labelSystemDataService.getSSoInfo(Long.parseLong(req.getAppPrimaryId()), req.getAppSource());
                if (null != ssoInfoMap && StringUtils.isNotEmpty(ssoInfoMap.get("appId"))) {
                    params.put("appId", ssoInfoMap.get("appId"));
                    params.put("appIdCode", ssoInfoMap.get("appIdCode"));
                    // params.put("appToken",ssoInfoMap.get("appToken").toString());
                }
                //params.put("appId","1");
                thirdTodoPullTaskJob.addJob(params);
            }
            return ResponseEntityWrapperUtil.wrapperOk(true);
        } catch (Exception e) {
            log.error("saveApplication error:{}", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/third/toDo/saveApplication");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }


    /**
     * 更新待办配置状态
     *
     * @param req 入参
     * @return 返回
     */
    @PostMapping("/updateStatus")
    public ResponseEntity<BaseResultDTO<Boolean>> updateStatus(@RequestBody ThirdTodoConfig req) {
        try {
            if (null == req.getId()) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, "id is Empty");
            }
            if (null == req.getValidStatus()) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, "validStatus is Empty");
            }
            QueryWrapper<ThirdTodoConfig> condition = new QueryWrapper<>();
            condition.eq("id", req.getId());
            List<ThirdTodoConfig> toDoConfigList = thirdTodoConfigService.getBaseMapper().selectList(condition);
            //恢复、暂停定时任务
            if(toDoConfigList.size()>0){
                if(toDoConfigList.get(0).getDataGetType()==2) {
                    if(req.getValidStatus()==0){
                        thirdTodoPullTaskJob.pauseJob("ThirdTodoPullTaskJob","TodoJob_"+req.getId());
                    }
                    if(req.getValidStatus()==1){
                        thirdTodoPullTaskJob.resumeJob("ThirdTodoPullTaskJob","TodoJob_"+req.getId());

                    }
                }
            }
            ThirdTodoConfig todoConfig=thirdTodoConfigService.getById(req.getId());
            todoConfig.setValidStatus(req.getValidStatus());
            thirdTodoConfigService.saveThirdToDo(todoConfig);
            return ResponseEntityWrapperUtil.wrapperOk(true);
        } catch (Exception e) {
            log.error("updateStatus error:{}", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/third/toDo/updateStatus");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 删除待办的配置
     *
     * @param deleteMessageConfigReq 入参
     * @return 返回
     */
    @PostMapping("/delToDoConfig")
    public ResponseEntity<BaseResultDTO<Boolean>> delToDoConfig(@RequestBody DeleteMessageConfigReq deleteMessageConfigReq) {
        try {
            if (CollectionUtils.isEmpty(deleteMessageConfigReq.getIdList())) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, "idList is not empty");
            }
            QueryWrapper<ThirdTodoConfig> condition = new QueryWrapper<>();
            condition.in("id", deleteMessageConfigReq.getIdList());
            condition.eq("valid_status", Constants.VALID_STATUS_ENABLE);
            List<ThirdTodoConfig> toDoConfigList = thirdTodoConfigService.getBaseMapper().selectList(condition);
            if (CollectionUtils.isNotEmpty(toDoConfigList)) {
                String error = messageUtils.getMessage("error.message.app.enable.del");
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, error);
            }
            //删除定时任务
            List<Long>idList=deleteMessageConfigReq.getIdList();
            for(Long id:idList){
                thirdTodoPullTaskJob.deleteJob("ThirdTodoPullTaskJob","TodoJob_"+id);
            }
            thirdTodoConfigService.delThirdToDo(deleteMessageConfigReq);
            return ResponseEntityWrapperUtil.wrapperOk(true);
        } catch (Exception e) {
            log.error("delToDoConfig error. e:{}", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/third/toDo/delToDoConfig");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }
}